/*
 * SpanDSP - a series of DSP components for telephony
 *
 * t4_states.h - state tables for T.4 FAX image processing
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2004 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: t4_states.h,v 1.3 2007/10/02 14:09:30 steveu Exp $
 */

/*
 * Note that these tables are ordered such that the
 * index into the table is known to be either the
 * run length, or (run length / 64) + a fixed offset.
 */
static const t4_run_table_entry_t t4_white_codes[] =
{
    {  8, 0x35,    0 },             /* 0011 0101 */
    {  6, 0x07,    1 },             /* 0001 11 */
    {  4, 0x07,    2 },             /* 0111 */
    {  4, 0x08,    3 },             /* 1000 */
    {  4, 0x0B,    4 },             /* 1011 */
    {  4, 0x0C,    5 },             /* 1100 */
    {  4, 0x0E,    6 },             /* 1110 */
    {  4, 0x0F,    7 },             /* 1111 */
    {  5, 0x13,    8 },             /* 1001 1 */
    {  5, 0x14,    9 },             /* 1010 0 */
    {  5, 0x07,   10 },             /* 0011 1 */
    {  5, 0x08,   11 },             /* 0100 0 */
    {  6, 0x08,   12 },             /* 0010 00 */
    {  6, 0x03,   13 },             /* 0000 11 */
    {  6, 0x34,   14 },             /* 1101 00 */
    {  6, 0x35,   15 },             /* 1101 01 */
    {  6, 0x2A,   16 },             /* 1010 10 */
    {  6, 0x2B,   17 },             /* 1010 11 */
    {  7, 0x27,   18 },             /* 0100 111 */
    {  7, 0x0C,   19 },             /* 0001 100 */
    {  7, 0x08,   20 },             /* 0001 000 */
    {  7, 0x17,   21 },             /* 0010 111 */
    {  7, 0x03,   22 },             /* 0000 011 */
    {  7, 0x04,   23 },             /* 0000 100 */
    {  7, 0x28,   24 },             /* 0101 000 */
    {  7, 0x2B,   25 },             /* 0101 011 */
    {  7, 0x13,   26 },             /* 0010 011 */
    {  7, 0x24,   27 },             /* 0100 100 */
    {  7, 0x18,   28 },             /* 0011 000 */
    {  8, 0x02,   29 },             /* 0000 0010 */
    {  8, 0x03,   30 },             /* 0000 0011 */
    {  8, 0x1A,   31 },             /* 0001 1010 */
    {  8, 0x1B,   32 },             /* 0001 1011 */
    {  8, 0x12,   33 },             /* 0001 0010 */
    {  8, 0x13,   34 },             /* 0001 0011 */
    {  8, 0x14,   35 },             /* 0001 0100 */
    {  8, 0x15,   36 },             /* 0001 0101 */
    {  8, 0x16,   37 },             /* 0001 0110 */
    {  8, 0x17,   38 },             /* 0001 0111 */
    {  8, 0x28,   39 },             /* 0010 1000 */
    {  8, 0x29,   40 },             /* 0010 1001 */
    {  8, 0x2A,   41 },             /* 0010 1010 */
    {  8, 0x2B,   42 },             /* 0010 1011 */
    {  8, 0x2C,   43 },             /* 0010 1100 */
    {  8, 0x2D,   44 },             /* 0010 1101 */
    {  8, 0x04,   45 },             /* 0000 0100 */
    {  8, 0x05,   46 },             /* 0000 0101 */
    {  8, 0x0A,   47 },             /* 0000 1010 */
    {  8, 0x0B,   48 },             /* 0000 1011 */
    {  8, 0x52,   49 },             /* 0101 0010 */
    {  8, 0x53,   50 },             /* 0101 0011 */
    {  8, 0x54,   51 },             /* 0101 0100 */
    {  8, 0x55,   52 },             /* 0101 0101 */
    {  8, 0x24,   53 },             /* 0010 0100 */
    {  8, 0x25,   54 },             /* 0010 0101 */
    {  8, 0x58,   55 },             /* 0101 1000 */
    {  8, 0x59,   56 },             /* 0101 1001 */
    {  8, 0x5A,   57 },             /* 0101 1010 */
    {  8, 0x5B,   58 },             /* 0101 1011 */
    {  8, 0x4A,   59 },             /* 0100 1010 */
    {  8, 0x4B,   60 },             /* 0100 1011 */
    {  8, 0x32,   61 },             /* 0011 0010 */
    {  8, 0x33,   62 },             /* 0011 0011 */
    {  8, 0x34,   63 },             /* 0011 0100 */
    {  5, 0x1B,   64 },             /* 1101 1 */
    {  5, 0x12,  128 },             /* 1001 0 */
    {  6, 0x17,  192 },             /* 0101 11 */
    {  7, 0x37,  256 },             /* 0110 111 */
    {  8, 0x36,  320 },             /* 0011 0110 */
    {  8, 0x37,  384 },             /* 0011 0111 */
    {  8, 0x64,  448 },             /* 0110 0100 */
    {  8, 0x65,  512 },             /* 0110 0101 */
    {  8, 0x68,  576 },             /* 0110 1000 */
    {  8, 0x67,  640 },             /* 0110 0111 */
    {  9, 0xCC,  704 },             /* 0110 0110 0 */
    {  9, 0xCD,  768 },             /* 0110 0110 1 */
    {  9, 0xD2,  832 },             /* 0110 1001 0 */
    {  9, 0xD3,  896 },             /* 0110 1001 1 */
    {  9, 0xD4,  960 },             /* 0110 1010 0 */
    {  9, 0xD5, 1024 },             /* 0110 1010 1 */
    {  9, 0xD6, 1088 },             /* 0110 1011 0 */
    {  9, 0xD7, 1152 },             /* 0110 1011 1 */
    {  9, 0xD8, 1216 },             /* 0110 1100 0 */
    {  9, 0xD9, 1280 },             /* 0110 1100 1 */
    {  9, 0xDA, 1344 },             /* 0110 1101 0 */
    {  9, 0xDB, 1408 },             /* 0110 1101 1 */
    {  9, 0x98, 1472 },             /* 0100 1100 0 */
    {  9, 0x99, 1536 },             /* 0100 1100 1 */
    {  9, 0x9A, 1600 },             /* 0100 1101 0 */
    {  6, 0x18, 1664 },             /* 0110 00 */
    {  9, 0x9B, 1728 },             /* 0100 1101 1 */
    { 11, 0x08, 1792 },             /* 0000 0001 000 */
    { 11, 0x0C, 1856 },             /* 0000 0001 100 */
    { 11, 0x0D, 1920 },             /* 0000 0001 101 */
    { 12, 0x12, 1984 },             /* 0000 0001 0010 */
    { 12, 0x13, 2048 },             /* 0000 0001 0011 */
    { 12, 0x14, 2112 },             /* 0000 0001 0100 */
    { 12, 0x15, 2176 },             /* 0000 0001 0101 */
    { 12, 0x16, 2240 },             /* 0000 0001 0110 */
    { 12, 0x17, 2304 },             /* 0000 0001 0111 */
    { 12, 0x1C, 2368 },             /* 0000 0001 1100 */
    { 12, 0x1D, 2432 },             /* 0000 0001 1101 */
    { 12, 0x1E, 2496 },             /* 0000 0001 1110 */
    { 12, 0x1F, 2560 },             /* 0000 0001 1111 */
};

static const t4_run_table_entry_t t4_black_codes[] =
{
    { 10, 0x37,    0 },             /* 0000 1101 11 */
    {  3, 0x02,    1 },             /* 010 */
    {  2, 0x03,    2 },             /* 11 */
    {  2, 0x02,    3 },             /* 10 */
    {  3, 0x03,    4 },             /* 011 */
    {  4, 0x03,    5 },             /* 0011 */
    {  4, 0x02,    6 },             /* 0010 */
    {  5, 0x03,    7 },             /* 0001 1 */
    {  6, 0x05,    8 },             /* 0001 01 */
    {  6, 0x04,    9 },             /* 0001 00 */
    {  7, 0x04,   10 },             /* 0000 100 */
    {  7, 0x05,   11 },             /* 0000 101 */
    {  7, 0x07,   12 },             /* 0000 111 */
    {  8, 0x04,   13 },             /* 0000 0100 */
    {  8, 0x07,   14 },             /* 0000 0111 */
    {  9, 0x18,   15 },             /* 0000 1100 0 */
    { 10, 0x17,   16 },             /* 0000 0101 11 */
    { 10, 0x18,   17 },             /* 0000 0110 00 */
    { 10, 0x08,   18 },             /* 0000 0010 00 */
    { 11, 0x67,   19 },             /* 0000 1100 111 */
    { 11, 0x68,   20 },             /* 0000 1101 000 */
    { 11, 0x6C,   21 },             /* 0000 1101 100 */
    { 11, 0x37,   22 },             /* 0000 0110 111 */
    { 11, 0x28,   23 },             /* 0000 0101 000 */
    { 11, 0x17,   24 },             /* 0000 0010 111 */
    { 11, 0x18,   25 },             /* 0000 0011 000 */
    { 12, 0xCA,   26 },             /* 0000 1100 1010 */
    { 12, 0xCB,   27 },             /* 0000 1100 1011 */
    { 12, 0xCC,   28 },             /* 0000 1100 1100 */
    { 12, 0xCD,   29 },             /* 0000 1100 1101 */
    { 12, 0x68,   30 },             /* 0000 0110 1000 */
    { 12, 0x69,   31 },             /* 0000 0110 1001 */
    { 12, 0x6A,   32 },             /* 0000 0110 1010 */
    { 12, 0x6B,   33 },             /* 0000 0110 1011 */
    { 12, 0xD2,   34 },             /* 0000 1101 0010 */
    { 12, 0xD3,   35 },             /* 0000 1101 0011 */
    { 12, 0xD4,   36 },             /* 0000 1101 0100 */
    { 12, 0xD5,   37 },             /* 0000 1101 0101 */
    { 12, 0xD6,   38 },             /* 0000 1101 0110 */
    { 12, 0xD7,   39 },             /* 0000 1101 0111 */
    { 12, 0x6C,   40 },             /* 0000 0110 1100 */
    { 12, 0x6D,   41 },             /* 0000 0110 1101 */
    { 12, 0xDA,   42 },             /* 0000 1101 1010 */
    { 12, 0xDB,   43 },             /* 0000 1101 1011 */
    { 12, 0x54,   44 },             /* 0000 0101 0100 */
    { 12, 0x55,   45 },             /* 0000 0101 0101 */
    { 12, 0x56,   46 },             /* 0000 0101 0110 */
    { 12, 0x57,   47 },             /* 0000 0101 0111 */
    { 12, 0x64,   48 },             /* 0000 0110 0100 */
    { 12, 0x65,   49 },             /* 0000 0110 0101 */
    { 12, 0x52,   50 },             /* 0000 0101 0010 */
    { 12, 0x53,   51 },             /* 0000 0101 0011 */
    { 12, 0x24,   52 },             /* 0000 0010 0100 */
    { 12, 0x37,   53 },             /* 0000 0011 0111 */
    { 12, 0x38,   54 },             /* 0000 0011 1000 */
    { 12, 0x27,   55 },             /* 0000 0010 0111 */
    { 12, 0x28,   56 },             /* 0000 0010 1000 */
    { 12, 0x58,   57 },             /* 0000 0101 1000 */
    { 12, 0x59,   58 },             /* 0000 0101 1001 */
    { 12, 0x2B,   59 },             /* 0000 0010 1011 */
    { 12, 0x2C,   60 },             /* 0000 0010 1100 */
    { 12, 0x5A,   61 },             /* 0000 0101 1010 */
    { 12, 0x66,   62 },             /* 0000 0110 0110 */
    { 12, 0x67,   63 },             /* 0000 0110 0111 */
    { 10, 0x0F,   64 },             /* 0000 0011 11 */
    { 12, 0xC8,  128 },             /* 0000 1100 1000 */
    { 12, 0xC9,  192 },             /* 0000 1100 1001 */
    { 12, 0x5B,  256 },             /* 0000 0101 1011 */
    { 12, 0x33,  320 },             /* 0000 0011 0011 */
    { 12, 0x34,  384 },             /* 0000 0011 0100 */
    { 12, 0x35,  448 },             /* 0000 0011 0101 */
    { 13, 0x6C,  512 },             /* 0000 0011 0110 0 */
    { 13, 0x6D,  576 },             /* 0000 0011 0110 1 */
    { 13, 0x4A,  640 },             /* 0000 0010 0101 0 */
    { 13, 0x4B,  704 },             /* 0000 0010 0101 1 */
    { 13, 0x4C,  768 },             /* 0000 0010 0110 0 */
    { 13, 0x4D,  832 },             /* 0000 0010 0110 1 */
    { 13, 0x72,  896 },             /* 0000 0011 1001 0 */
    { 13, 0x73,  960 },             /* 0000 0011 1001 1 */
    { 13, 0x74, 1024 },             /* 0000 0011 1010 0 */
    { 13, 0x75, 1088 },             /* 0000 0011 1010 1 */
    { 13, 0x76, 1152 },             /* 0000 0011 1011 0 */
    { 13, 0x77, 1216 },             /* 0000 0011 1011 1 */
    { 13, 0x52, 1280 },             /* 0000 0010 1001 0 */
    { 13, 0x53, 1344 },             /* 0000 0010 1001 1 */
    { 13, 0x54, 1408 },             /* 0000 0010 1010 0 */
    { 13, 0x55, 1472 },             /* 0000 0010 1010 1 */
    { 13, 0x5A, 1536 },             /* 0000 0010 1101 0 */
    { 13, 0x5B, 1600 },             /* 0000 0010 1101 1 */
    { 13, 0x64, 1664 },             /* 0000 0011 0010 0 */
    { 13, 0x65, 1728 },             /* 0000 0011 0010 1 */
    { 11, 0x08, 1792 },             /* 0000 0001 000 */
    { 11, 0x0C, 1856 },             /* 0000 0001 100 */
    { 11, 0x0D, 1920 },             /* 0000 0001 101 */
    { 12, 0x12, 1984 },             /* 0000 0001 0010 */
    { 12, 0x13, 2048 },             /* 0000 0001 0011 */
    { 12, 0x14, 2112 },             /* 0000 0001 0100 */
    { 12, 0x15, 2176 },             /* 0000 0001 0101 */
    { 12, 0x16, 2240 },             /* 0000 0001 0110 */
    { 12, 0x17, 2304 },             /* 0000 0001 0111 */
    { 12, 0x1C, 2368 },             /* 0000 0001 1100 */
    { 12, 0x1D, 2432 },             /* 0000 0001 1101 */
    { 12, 0x1E, 2496 },             /* 0000 0001 1110 */
    { 12, 0x1F, 2560 },             /* 0000 0001 1111 */
};

static const t4_table_entry_t t4_2d_table[128] =
{
    {S_Null    ,   1,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,   -1},
    {S_Vert    ,   1,    0},
    {S_Horiz   ,   3,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,    1},
    {S_Vert    ,   1,    0},
    {S_Pass    ,   4,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,   -1},
    {S_Vert    ,   1,    0},
    {S_Horiz   ,   3,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,    1},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   6,   -2},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,   -1},
    {S_Vert    ,   1,    0},
    {S_Horiz   ,   3,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,    1},
    {S_Vert    ,   1,    0},
    {S_Pass    ,   4,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,   -1},
    {S_Vert    ,   1,    0},
    {S_Horiz   ,   3,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,    1},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   7,   -3},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,   -1},
    {S_Vert    ,   1,    0},
    {S_Horiz   ,   3,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,    1},
    {S_Vert    ,   1,    0},
    {S_Pass    ,   4,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,   -1},
    {S_Vert    ,   1,    0},
    {S_Horiz   ,   3,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,    1},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   6,    2},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,   -1},
    {S_Vert    ,   1,    0},
    {S_Horiz   ,   3,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,    1},
    {S_Vert    ,   1,    0},
    {S_Pass    ,   4,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,   -1},
    {S_Vert    ,   1,    0},
    {S_Horiz   ,   3,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,    1},
    {S_Vert    ,   1,    0},
    {S_Ext     ,   7,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,   -1},
    {S_Vert    ,   1,    0},
    {S_Horiz   ,   3,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,    1},
    {S_Vert    ,   1,    0},
    {S_Pass    ,   4,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,   -1},
    {S_Vert    ,   1,    0},
    {S_Horiz   ,   3,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,    1},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   6,   -2},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,   -1},
    {S_Vert    ,   1,    0},
    {S_Horiz   ,   3,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,    1},
    {S_Vert    ,   1,    0},
    {S_Pass    ,   4,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,   -1},
    {S_Vert    ,   1,    0},
    {S_Horiz   ,   3,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,    1},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   7,    3},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,   -1},
    {S_Vert    ,   1,    0},
    {S_Horiz   ,   3,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,    1},
    {S_Vert    ,   1,    0},
    {S_Pass    ,   4,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,   -1},
    {S_Vert    ,   1,    0},
    {S_Horiz   ,   3,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,    1},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   6,    2},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,   -1},
    {S_Vert    ,   1,    0},
    {S_Horiz   ,   3,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,    1},
    {S_Vert    ,   1,    0},
    {S_Pass    ,   4,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,   -1},
    {S_Vert    ,   1,    0},
    {S_Horiz   ,   3,    0},
    {S_Vert    ,   1,    0},
    {S_Vert    ,   3,    1},
    {S_Vert    ,   1,    0}
};

static const t4_table_entry_t t4_1d_white_table[4096] =
{
    {S_Null    ,   1,    0},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   39},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   8,  576},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   55},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   45},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   53},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  448},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   35},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   51},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,   63},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1472},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   43},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1216},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   29},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   33},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   49},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   61},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   47},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   59},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   41},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  960},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   31},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   57},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   9,  704},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   37},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  320},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_MakeUp  ,  11, 1792},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   40},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  832},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   56},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   46},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   54},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  512},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   36},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   52},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,    0},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1600},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   44},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1344},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   30},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   34},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   50},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   62},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   48},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   60},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   42},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1088},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   32},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   58},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  640},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   38},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  384},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_Null    ,   1,    0},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   39},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   8,  576},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   55},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   45},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   53},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  448},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   35},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   51},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,   63},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1536},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   43},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1280},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   29},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   33},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   49},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   61},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   47},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   59},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   41},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1024},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   31},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   57},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   9,  768},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   37},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  320},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_MakeUp  ,  11, 1856},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   40},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  896},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   56},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   46},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   54},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  512},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   36},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   52},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,    0},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1728},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   44},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1408},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   30},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   34},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   50},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   62},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   48},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   60},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   42},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1152},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   32},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   58},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  640},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   38},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  384},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_Null    ,   1,    0},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   39},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   8,  576},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   55},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   45},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   53},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  448},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   35},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   51},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,   63},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1472},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   43},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1216},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   29},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   33},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   49},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   61},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   47},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   59},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   41},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  960},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   31},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   57},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   9,  704},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   37},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  320},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_MakeUp  ,  12, 2112},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   40},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  832},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   56},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   46},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   54},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  512},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   36},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   52},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,    0},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1600},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   44},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1344},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   30},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   34},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   50},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   62},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   48},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   60},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   42},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1088},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   32},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   58},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  640},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   38},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  384},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_Null    ,   1,    0},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   39},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   8,  576},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   55},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   45},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   53},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  448},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   35},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   51},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,   63},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1536},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   43},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1280},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   29},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   33},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   49},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   61},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   47},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   59},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   41},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1024},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   31},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   57},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   9,  768},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   37},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  320},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_MakeUp  ,  12, 2368},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   40},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  896},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   56},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   46},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   54},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  512},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   36},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   52},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,    0},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1728},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   44},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1408},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   30},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   34},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   50},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   62},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   48},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   60},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   42},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1152},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   32},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   58},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  640},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   38},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  384},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_Null    ,   1,    0},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   39},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   8,  576},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   55},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   45},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   53},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  448},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   35},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   51},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,   63},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1472},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   43},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1216},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   29},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   33},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   49},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   61},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   47},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   59},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   41},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  960},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   31},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   57},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   9,  704},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   37},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  320},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_MakeUp  ,  12, 1984},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   40},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  832},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   56},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   46},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   54},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  512},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   36},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   52},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,    0},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1600},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   44},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1344},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   30},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   34},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   50},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   62},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   48},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   60},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   42},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1088},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   32},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   58},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  640},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   38},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  384},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_Null    ,   1,    0},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   39},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   8,  576},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   55},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   45},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   53},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  448},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   35},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   51},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,   63},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1536},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   43},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1280},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   29},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   33},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   49},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   61},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   47},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   59},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   41},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1024},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   31},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   57},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   9,  768},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   37},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  320},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_MakeUp  ,  11, 1920},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   40},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  896},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   56},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   46},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   54},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  512},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   36},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   52},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,    0},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1728},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   44},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1408},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   30},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   34},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   50},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   62},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   48},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   60},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   42},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1152},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   32},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   58},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  640},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   38},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  384},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_Null    ,   1,    0},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   39},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   8,  576},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   55},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   45},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   53},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  448},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   35},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   51},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,   63},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1472},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   43},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1216},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   29},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   33},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   49},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   61},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   47},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   59},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   41},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  960},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   31},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   57},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   9,  704},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   37},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  320},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_MakeUp  ,  12, 2240},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   40},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  832},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   56},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   46},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   54},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  512},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   36},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   52},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,    0},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1600},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   44},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1344},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   30},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   34},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   50},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   62},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   48},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   60},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   42},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1088},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   32},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   58},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  640},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   38},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  384},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_Null    ,   1,    0},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   39},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   8,  576},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   55},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   45},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   53},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  448},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   35},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   51},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,   63},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1536},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   43},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1280},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   29},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   33},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   49},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   61},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   47},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   59},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   41},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1024},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   31},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   57},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   9,  768},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   37},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  320},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_MakeUp  ,  12, 2496},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   40},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  896},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   56},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   46},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   54},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  512},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   36},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   52},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,    0},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1728},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   44},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1408},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   30},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   34},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   50},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   62},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   48},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   60},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   42},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1152},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   32},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   58},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  640},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   38},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  384},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_EOL     ,  12,    0},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   39},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   8,  576},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   55},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   45},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   53},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  448},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   35},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   51},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,   63},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1472},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   43},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1216},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   29},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   33},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   49},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   61},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   47},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   59},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   41},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  960},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   31},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   57},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   9,  704},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   37},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  320},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_MakeUp  ,  11, 1792},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   40},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  832},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   56},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   46},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   54},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  512},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   36},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   52},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,    0},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1600},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   44},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1344},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   30},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   34},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   50},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   62},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   48},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   60},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   42},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1088},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   32},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   58},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  640},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   38},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  384},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_Null    ,   1,    0},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   39},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   8,  576},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   55},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   45},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   53},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  448},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   35},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   51},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,   63},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1536},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   43},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1280},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   29},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   33},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   49},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   61},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   47},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   59},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   41},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1024},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   31},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   57},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   9,  768},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   37},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  320},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_MakeUp  ,  11, 1856},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   40},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  896},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   56},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   46},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   54},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  512},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   36},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   52},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,    0},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1728},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   44},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1408},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   30},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   34},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   50},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   62},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   48},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   60},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   42},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1152},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   32},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   58},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  640},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   38},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  384},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_Null    ,   1,    0},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   39},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   8,  576},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   55},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   45},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   53},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  448},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   35},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   51},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,   63},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1472},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   43},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1216},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   29},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   33},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   49},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   61},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   47},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   59},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   41},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  960},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   31},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   57},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   9,  704},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   37},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  320},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_MakeUp  ,  12, 2176},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   40},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  832},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   56},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   46},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   54},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  512},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   36},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   52},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,    0},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1600},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   44},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1344},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   30},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   34},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   50},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   62},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   48},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   60},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   42},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1088},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   32},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   58},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  640},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   38},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  384},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_Null    ,   1,    0},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   39},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   8,  576},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   55},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   45},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   53},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  448},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   35},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   51},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,   63},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1536},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   43},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1280},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   29},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   33},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   49},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   61},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   47},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   59},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   41},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1024},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   31},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   57},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   9,  768},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   37},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  320},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_MakeUp  ,  12, 2432},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   40},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  896},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   56},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   46},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   54},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  512},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   36},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   52},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,    0},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1728},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   44},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1408},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   30},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   34},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   50},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   62},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   48},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   60},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   42},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1152},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   32},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   58},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  640},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   38},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  384},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_Null    ,   1,    0},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   39},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   8,  576},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   55},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   45},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   53},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  448},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   35},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   51},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,   63},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1472},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   43},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1216},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   29},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   33},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   49},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   61},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   47},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   59},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   41},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  960},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   31},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   57},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   9,  704},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   37},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  320},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_MakeUp  ,  12, 2048},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   40},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  832},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   56},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   46},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   54},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  512},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   36},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   52},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,    0},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1600},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   44},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1344},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   30},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   34},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   50},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   62},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   48},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   60},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   42},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1088},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   32},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   58},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  640},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   38},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  384},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_Null    ,   1,    0},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   39},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   8,  576},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   55},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   45},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   53},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  448},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   35},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   51},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,   63},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1536},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   43},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1280},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   29},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   33},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   49},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   61},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   47},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   59},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   41},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1024},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   31},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   57},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   9,  768},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   37},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  320},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_MakeUp  ,  11, 1920},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   40},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  896},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   56},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   46},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   54},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  512},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   36},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   52},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,    0},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1728},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   44},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1408},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   30},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   34},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   50},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   62},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   48},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   60},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   42},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1152},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   32},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   58},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  640},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   38},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  384},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_Null    ,   1,    0},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   39},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   8,  576},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   55},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   45},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   53},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  448},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   35},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   51},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,   63},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1472},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   43},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1216},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   29},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   33},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   49},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   61},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   47},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   59},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   41},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  960},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   31},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   57},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   9,  704},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   37},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  320},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_MakeUp  ,  12, 2304},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   40},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  832},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   56},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   46},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   54},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  512},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   36},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   52},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,    0},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1600},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   44},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1344},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   30},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   34},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   50},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   62},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   48},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   60},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   42},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1088},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   32},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   58},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  640},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   38},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  384},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_Null    ,   1,    0},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   39},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   8,  576},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   55},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   45},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   53},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  448},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   35},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   51},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,   63},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1536},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   43},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1280},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   29},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   33},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   49},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   61},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   47},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   59},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   41},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1024},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   31},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   57},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   9,  768},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   37},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  320},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_MakeUp  ,  12, 2560},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   20},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   24},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   7,   28},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   23},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   27},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   40},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9,  896},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   7,   19},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   56},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   46},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   54},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  512},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   36},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   52},
    {S_TermW   ,   6,   15},
    {S_TermW   ,   8,    0},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_MakeUpW ,   9, 1728},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   44},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   9, 1408},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   30},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   6,   12},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   6, 1664},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   34},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   8,   50},
    {S_TermW   ,   6,   14},
    {S_TermW   ,   8,   62},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   8,   48},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   8,   60},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   8,   42},
    {S_TermW   ,   6,   16},
    {S_MakeUpW ,   9, 1152},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   32},
    {S_TermW   ,   5,    8},
    {S_TermW   ,   8,   58},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   7,   22},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   5,   11},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   26},
    {S_TermW   ,   5,    9},
    {S_MakeUpW ,   8,  640},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   8,   38},
    {S_MakeUpW ,   5,  128},
    {S_TermW   ,   7,   25},
    {S_TermW   ,   6,   15},
    {S_MakeUpW ,   8,  384},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7},
    {S_TermW   ,   6,   13},
    {S_TermW   ,   4,    3},
    {S_TermW   ,   7,   18},
    {S_TermW   ,   4,    5},
    {S_TermW   ,   7,   21},
    {S_TermW   ,   6,   17},
    {S_MakeUpW ,   7,  256},
    {S_TermW   ,   4,    6},
    {S_TermW   ,   6,    1},
    {S_TermW   ,   5,    8},
    {S_MakeUpW ,   6,  192},
    {S_MakeUpW ,   5,   64},
    {S_TermW   ,   5,   10},
    {S_TermW   ,   4,    4},
    {S_TermW   ,   4,    2},
    {S_TermW   ,   4,    7}
};

static const t4_table_entry_t t4_1d_black_table[8192] =
{
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   9,   15},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   18},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   17},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  11, 1792},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   23},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   20},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   25},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  12,  128},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   56},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   30},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  11, 1856},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   57},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   21},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   54},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   9,   15},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   52},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   48},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  12, 2112},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   44},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   36},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  12,  384},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   28},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   60},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   40},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  12, 2368},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   16},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  10,   64},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   9,   15},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   18},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   17},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  12, 1984},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   50},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   34},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  13, 1664},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   26},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  13, 1408},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   32},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  11, 1920},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   61},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   42},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  13, 1024},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   9,   15},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  13,  768},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   62},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  12, 2240},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   46},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   38},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  13,  512},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   19},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   24},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   22},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  12, 2496},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   16},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  10,   64},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_EOL     ,  12,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   9,   15},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   18},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   17},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  11, 1792},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   23},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   20},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   25},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  12,  192},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  13, 1280},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   31},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  11, 1856},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   58},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   21},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  13,  896},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   9,   15},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  13,  640},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   49},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  12, 2176},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   45},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   37},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  12,  448},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   29},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  13, 1536},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   41},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  12, 2432},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   16},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  10,   64},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   9,   15},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   18},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   17},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  12, 2048},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   51},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   35},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  12,  320},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   27},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   59},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   33},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  11, 1920},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  12,  256},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   43},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  13, 1152},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   9,   15},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   55},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   63},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  12, 2304},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   47},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   39},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   53},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   19},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   24},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   22},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  12, 2560},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   16},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  10,   64},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   9,   15},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   18},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   17},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  11, 1792},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   23},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   20},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   25},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  12,  128},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   56},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   30},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  11, 1856},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   57},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   21},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   54},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   9,   15},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   52},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   48},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  12, 2112},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   44},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   36},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  12,  384},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   28},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   60},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   40},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  12, 2368},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   16},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  10,   64},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   9,   15},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   18},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   17},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  12, 1984},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   50},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   34},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  13, 1728},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   26},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  13, 1472},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   32},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  11, 1920},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   61},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   42},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  13, 1088},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   9,   15},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  13,  832},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   62},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  12, 2240},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   46},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   38},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  13,  576},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   19},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   24},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   22},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  12, 2496},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   16},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  10,   64},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_EOL     ,  12,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   9,   15},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   18},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   17},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  11, 1792},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   23},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   20},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   25},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  12,  192},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  13, 1344},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   31},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  11, 1856},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   58},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   21},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  13,  960},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   9,   15},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  13,  704},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   49},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  12, 2176},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   45},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   37},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  12,  448},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   29},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  13, 1600},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   41},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  12, 2432},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   16},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  10,   64},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   9,   15},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   18},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   17},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  12, 2048},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   51},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   35},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  12,  320},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   27},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   59},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   33},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  11, 1920},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  12,  256},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   43},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  13, 1216},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   9,   15},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   55},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   63},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  12, 2304},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   47},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   39},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  12,   53},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_Null    ,   1,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   13},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   19},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   24},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  11,   22},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUp  ,  12, 2560},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   10},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,   16},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,  10,    0},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_MakeUpB ,  10,   64},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    9},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   11},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   8,   14},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   6,    8},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   7,   12},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    6},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   5,    7},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    1},
    {S_TermB   ,   2,    2},
    {S_TermB   ,   4,    5},
    {S_TermB   ,   2,    3},
    {S_TermB   ,   3,    4},
    {S_TermB   ,   2,    2}
};

/*- End of file ------------------------------------------------------------*/
