/*
 * SpanDSP - a series of DSP components for telephony
 *
 * private/t38_core.h - An implementation of T.38, less the packet exchange part
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2005 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 2.1,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: t38_core.h,v 1.1 2008/11/30 13:44:35 steveu Exp $
 */

#if !defined(_SPANDSP_PRIVATE_T38_CORE_H_)
#define _SPANDSP_PRIVATE_T38_CORE_H_

/*!
    Core T.38 state, common to all modes of T.38.
*/
struct t38_core_state_s
{
    /*! Handler routine to transmit IFP packets generated by the T.38 protocol engine */
    t38_tx_packet_handler_t *tx_packet_handler;
    /*! An opaque pointer passed to tx_packet_handler */
    void *tx_packet_user_data;

    /*! Handler routine to process received indicator packets */
    t38_rx_indicator_handler_t *rx_indicator_handler;
    /*! Handler routine to process received data packets */
    t38_rx_data_handler_t *rx_data_handler;
    /*! Handler routine to process the missing packet condition */
    t38_rx_missing_handler_t *rx_missing_handler;
    /*! An opaque pointer passed to any of the above receive handling routines */
    void *rx_user_data;

    /*! NOTE - Bandwidth reduction shall only be done on suitable Phase C data, i.e., MH, MR
        and - in the case of transcoding to JBIG - MMR. MMR and JBIG require reliable data
        transport such as that provided by TCP. When transcoding is selected, it shall be
        applied to every suitable page in a call. */

    /*! Method 1: Local generation of TCF (required for use with TCP).
        Method 2: Transfer of TCF is required for use with UDP (UDPTL or RTP).
        Method 2 is not recommended for use with TCP. */
    int data_rate_management_method;
    
    /*! The emitting gateway may indicate a preference for either UDP/UDPTL, or
        UDP/RTP, or TCP for transport of T.38 IFP Packets. The receiving device
        selects the transport protocol. */
    int data_transport_protocol;

    /*! Indicates the capability to remove and insert fill bits in Phase C, non-ECM
        data to reduce bandwidth in the packet network. */
    int fill_bit_removal;

    /*! Indicates the ability to convert to/from MMR from/to the line format to
        improve the compression of the data, and reduce the bandwidth, in the
        packet network. */
    int mmr_transcoding;

    /*! Indicates the ability to convert to/from JBIG to reduce bandwidth. */
    int jbig_transcoding;

    /*! For UDP (UDPTL or RTP) modes, this option indicates the maximum
        number of octets that can be stored on the remote device before an overflow
        condition occurs. It is the responsibility of the transmitting application to
        limit the transfer rate to prevent an overflow. The negotiated data rate
        should be used to determine the rate at which data is being removed from
        the buffer. */
    int max_buffer_size;

    /*! This option indicates the maximum size of a UDPTL packet or the
        maximum size of the payload within an RTP packet that can be accepted by
        the remote device. */
    int max_datagram_size;

    /*! This is the version number of ITU-T Rec. T.38. New versions shall be
        compatible with previous versions. */
    int t38_version;

    /*! The fastest data rate supported by the T.38 channel. */
    int fastest_image_data_rate;

    /*! \brief The number of times an indicator packet will be sent. Numbers greater than one
               will increase reliability for UDP transmission. Zero is valid, to suppress all
               indicator packets for TCP transmission. */
    int indicator_tx_count;

    /*! \brief The number of times a data packet which does not end transmission will be sent.
               Numbers greater than one will increase reliability for UDP transmission. Zero
               is not valid. */
    int data_tx_count;

    /*! \brief The number of times a data packet which ends transmission will be sent. Numbers
               greater than one will increase reliability for UDP transmission. Zero is not valid. */
    int data_end_tx_count;

    /*! TRUE if IFP packet sequence numbers are relevant. For some transports, like TPKT
        over TCP they are not relevent. */
    int check_sequence_numbers;

    /*! The sequence number for the next packet to be transmitted */
    int tx_seq_no;
    /*! The sequence number expected in the next received packet */
    int rx_expected_seq_no;

    /*! The current receive indicator - i.e. the last indicator received */
    int current_rx_indicator;
    /*! The current receive data type - i.e. the last data type received */
    int current_rx_data_type;
    /*! The current receive field type - i.e. the last field_type received */
    int current_rx_field_type;
    /*! The current transmit indicator - i.e. the last indicator transmitted */
    int current_tx_indicator;
    /*! The bit rate for V.34 operation */
    int v34_rate;

    /*! A count of missing receive packets. This count might not be accurate if the
        received packet numbers jump wildly. */
    int missing_packets;

    logging_state_t logging;
};

#endif
/*- End of file ------------------------------------------------------------*/
