/*
 * instrument.c
 *
 * Basic code for GPIB over RS232C style control of instruments,
 * like the TAS 240, or the TAS1200D
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2007 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: instrument.c,v 1.1.1.1 2007/06/09 06:31:18 steveu Exp $
 */

#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <ctype.h>
#include <string.h>
#include <malloc.h>
#include <fcntl.h>
#include <termios.h>
#include <inttypes.h>
#include <sys/types.h>
#include <sys/unistd.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/ioctl.h>

#include "instrument.h"

#define FALSE 0
#define TRUE (!FALSE)

#define MAX_GPIB_CMD_LEN        128

extern int debug;

static void instr_get_response(int fd, char *response)
{
    int len;

    len = 0;
    while (read(fd, &response[len], 1) > 0)
    {
        if (response[len] == '\r')
        {
            response[len] = '\0';
            //printf("Got '%s'\n", response);
            return;
        }
        if (response[len] != '\n')
            len++;
    }
}
/*- End of function --------------------------------------------------------*/

static void instr_get_prompt(int fd)
{
    char buf[2];

    /* Clear out the "> " prompt that follows the response */
    while (read(fd, buf, 1) > 0)
    {
        if (buf[0] == ' ')
            return;
    }
}
/*- End of function --------------------------------------------------------*/

static int instr_put(int fd, const char *request)
{
    char requestx[MAX_GPIB_CMD_LEN + 1];
    char response[MAX_GPIB_CMD_LEN + 1];

    snprintf(requestx, sizeof(requestx), "/%s/", request);
    write(fd, requestx, strlen(requestx));
    write(fd, "\r", 1);
    /* Get the echo of what we just sent */
    instr_get_response(fd, response);
    if (strlen(response) < 2  ||  strcmp(requestx, response))
    {
        printf("Echo of '%s' was '%s'\n", requestx, response);
        return -1;
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

const char *scan_for_field(const char *field_id, const char *response, char *field)
{
    int i;
    int len;
    const char *x;
    const char *y;

    len = strlen(field_id);
    x = response;
    while (x[0])
    {
        if ((y = strchr(x, ',')) == NULL)
        {
            if ((y = strchr(x, ':')) == NULL)
                return NULL;
        }
        x = y;
        if (strncmp(x + 1, field_id, len) == 0)
        {
            x += len + 1;
            if (field)
            {
                for (i = 0;
                     x[i]  &&  x[i] != '/'  &&  x[i] != ','  &&  x[i] != ' ';
                     i++)
                {
                    field[i] = x[i];
                }
                field[i] = '\0';
            }
            return x;
        }
        x++;
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int scan_for_error(const tag_t error_codes[], const char *request, const char *response)
{
    int val;
    int i;
    const char *x;

    if ((x = scan_for_field("E", response, NULL)))
    {
        if (sscanf(x, "%d", &val) == 1)
        {
            for (i = 0;  error_codes[i].code >= 0;  i++)
            {
                if (error_codes[i].code == val)
                {
                    printf("Request: '%s'\n", request);
                    printf("Error: '%s'\n", error_codes[i].explanation);
                    return val;
                }
            }
        }
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

int instr_exchange(const instr_t *inst, const char *request, char *response, ...)
{
    char msg[MAX_GPIB_CMD_LEN + 1];
    va_list arg_ptr;
    int err;

    va_start(arg_ptr, response);
    vsnprintf(msg, sizeof(msg), request, arg_ptr);
    va_end(arg_ptr);
    if (debug)
        printf("Request '%s'\n", msg);
    if (instr_put(inst->fd, msg) < 0)
        return -1;
    instr_get_response(inst->fd, response);
    if (debug)
        printf("Response '%s'\n", response);
    instr_get_prompt(inst->fd);
    if ((err = scan_for_error(inst->error_codes, msg, response)))
        return err;
    if (debug)
    {
        printf("Proceed?\n");
        getchar();
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

int instr_exchange_c(const instr_t *inst, const char *request, ...)
{
    char msg[MAX_GPIB_CMD_LEN + 1];
    char response[MAX_GPIB_CMD_LEN + 1];
    va_list arg_ptr;
    int err;

    va_start(arg_ptr, request);
    vsnprintf(msg, sizeof(msg), request, arg_ptr);
    va_end(arg_ptr);
    if (debug)
        printf("Request '%s'\n", msg);
    if (instr_put(inst->fd, msg) < 0)
        return -1;
    instr_get_response(inst->fd, response);
    if (debug)
        printf("Response '%s'\n", response);
    instr_get_prompt(inst->fd);
    if ((err = scan_for_error(inst->error_codes, msg, response)))
        return err;
    if (strcmp(response, "/C/"))
    {
        printf("Received '%s'\n", response);
        return -1;
    }
#if 0
    if (debug)
    {
        printf("Proceed?\n");
        getchar();
    }
#endif
    return 0;
}
/*- End of function --------------------------------------------------------*/

instr_t *instr_open(const char *dev, int speed, tag_t *error_codes)
{
    struct termios tios;
    int fd;
    int speed_code;
    instr_t *inst;

    switch (speed)
    {
    case 50:
        speed_code = B50;
        break;
    case 75:
        speed_code = B75;
        break;
    case 110:
        speed_code = B110;
        break;
    case 134:
        speed_code = B134;
        break;
    case 150:
        speed_code = B150;
        break;
    case 200:
        speed_code = B200;
        break;
    case 1800:
        speed_code = B1800;
        break;
    case 2400:
        speed_code = B2400;
        break;
    case 4800:
        speed_code = B4800;
        break;
    case 9600:
        speed_code = B9600;
        break;
    case 19200:
        speed_code = B19200;
        break;
    case 38400:
        speed_code = B38400;
        break;
    case 57600:
        speed_code = B57600;
        break;
    case 115200:
        speed_code = B115200;
        break;
#if defined(B230400)
    case 230400:
        speed_code = B230400;
        break;
#endif
#if defined(B460800)
    case 460800:
        speed_code = B460800;
        break;
#endif
#if defined(B921600)
    case 921600:
        speed_code = B921600;
        break;
#endif
    default:
        return NULL;
    }
    /*endswitch*/

    if ((fd = open(dev, O_RDWR)) < 0)
        return NULL;
    /*endif*/
    if (tcgetattr(fd, &tios) == 0)
    {
        tios.c_cflag &= ~(CSIZE | CSTOPB);
        tios.c_cflag |= CS7 | CREAD | CLOCAL | PARENB | PARODD;
        tios.c_iflag = IGNPAR;
        tios.c_oflag = 0;
        tios.c_lflag = 0;
        cfsetispeed (&tios, speed_code);
        cfsetospeed (&tios, speed_code);
        tcsetattr(fd, TCSAFLUSH, &tios);
    }
    /*endif*/
    /* Send a carriage return, wait a bit, and flush. This should clean out
       any garbage in the buffers at either end. */
    write(fd, "//\r", 3);
    usleep(500000);
    tcflush(fd, TCIOFLUSH);
    inst = malloc(sizeof(*inst));
    inst->fd = fd;
    inst->error_codes = error_codes;
    return inst;
}
/*- End of function --------------------------------------------------------*/

int instr_close(instr_t *inst)
{
    if (inst)
    {
        close(inst->fd);
        free(inst);
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

const char *display_explanation(tag_t tags[], int code)
{
    int i;

    for (i = 0;  tags[i].code >= 0;  i++)
    {
        if (tags[i].code == code)
        {
            printf(">>> %s\n", tags[i].explanation);
            return tags[i].explanation;
        }
    }
    return NULL;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
