/*
 * tas1200d.c
 *
 * Basic control of a TAS1200D telephone network emulator.
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2007 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: tas1200d.c,v 1.1.1.1 2007/06/09 06:31:18 steveu Exp $
 */

#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <ctype.h>
#include <string.h>
#include <malloc.h>
#include <fcntl.h>
#include <termios.h>
#include <inttypes.h>
#include <sys/types.h>
#include <sys/unistd.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/ioctl.h>

#include "instrument.h"

#define FALSE 0
#define TRUE (!FALSE)

#define MAX_GPIB_CMD_LEN        128

/* TAS 1200D error codes */
tag_t error_codes[] =
{
    {1, "Parameter value error"},
    {2, "Command syntax error"},
    {5, "A->B signaling generator calibration failure"},
    {6, "A->B transmission channel output circuit calibration failure"},
    {7, "A->B SFI generator (SP3A #1) calibration failure"},
    {8, "A->B noise filter #1 (5kHz) calibration failure"},
    {9, "A->B input AGC failure (input signal level not between -23.0 and +7.0dBm)"},
    {13, "Central office module (NIM 2A) not present"},
    {15, "A->B impairment generator (SP3A #1) not present"},
    {17, "A->B output AGC failure (can't compensate for channel rolloff)"},
    {18, "A->B output AGC failure (output level below capture range"},
    {19, "A->B PCM/ADPCM module (SP3C #1) not present"},
    {26, "A->B impulse noise generator (SP3A #1) calibration failure"},
    {28, "A->B noise filter #2 (4kHz) calibration failure"},
    {38, "A->B noise filter #3 (20kHz) calibration failure"},
    {41, "A->B impairment generator (SP3A #1) bad response"},
    {42, "A->B impairment generator (SP3A #1) did not accept data"},
    {43, "A->B impairment generator (SP3A #1) no response"},
    {51, "Digital network section 1 (DNS1) bad response"},
    {52, "Digital network section 1 (DNS1) did not accept data"},
    {53, "Digital network section 1 (DNS1) no response"},
    {61, "Central office module (NIM 2A) bad response"},
    {62, "Central office module (NIM 2A) did not accept data"},
    {63, "Central office module (NIM 2A) no response"},
    {71, "Channel access module (SP3B) bad response"},
    {72, "Channel access module (SP3B) did not accept data"},
    {73, "Channel access module (SP3B) no response"},
    {75, "A->B signal generator level failure"},
    {76, "A->B input circuit level failure"},
    {77, "A->B impairment generator (SP3A #1) level failure"},
    {78, "A->B transmission channel output circuit level failure"},
    {79, "A->B summer circuit level failure"},
    {85, "Station B 4-wire receive level failure"},
    {86, "Station B 2-wire receive level failure"},
    {87, "A->B dial tone level failure"},
    {89, "A->B PCM/ADPCM module (SP3C #1) level failure"},
    {91, "A->B PCM/ADPCM module (SP3C #1) bad response"},
    {92, "A->B PCM/ADPCM module (SP3C #1) did not accept data"},
    {93, "A->B PCM/ADPCM module (SP3C #1) no response"},
    {95, "Extended PCM/ADPCM (EPAL) bad response"},
    {96, "Extended PCM/ADPCM (EPAL) did not accept data"},
    {97, "Extended PCM/ADPCM (EPAL) no response"},
    {99, "Cellular audio processing module (CAP) bad response"},
    {100, "Cellular audio processing module (CAP) did not accept data"},
    {101, "Cellular audio processing module (CAP) no response"},
    {105, "B->A signaling generator calibration failure"},
    {106, "B->A transmission channel output circuit calibration failure"},
    {107, "B->A SFI generator (SP3A #2) calibration failure"},
    {108, "B->A noise filter #1 (5kHz) calibration failure"},
    {109, "B->A input AGC failure (input signal level not between -23.0 and +7.0dBm)"},
    {113, "Central office module B (NUM 2B) not present"},
    {115, "B->A impairment generator (SP3A #2) not present"},
    {117, "B->A output AGC failure (can't compensate for channel rolloff)"},
    {118, "B->A output AGC failure (output level below capture range)"},
    {119, "B->A PCM/ADPCM module (SP3C #2) not present"},
    {126, "B->A impulse noise generator (SP3C #2) calibration failure"},
    {128, "B->A noise filter #2 (4kHz) calibration failure"},
    {138, "B->A noise filter #2 (20kHz) calibration failure"},
    {141, "B->A impairment generator (SP3A #2) bad response"},
    {142, "B->A impairment generator (SP3A #2) did not accept data"},
    {143, "B->A impairment generator (SP3A #2) no response"},
    {151, "Digital network section 2 (DNS2) bad response"},
    {152, "Digital network section 2 (DNS2) did not accept data"},
    {153, "Digital network section 2 (DNS2) no response"},
    {175, "B->A signal generator level failure"},
    {176, "B->A input circuit level failure"},
    {177, "B->A impairments generator (SP3A #2) level failure"},
    {178, "B->A transmission channel output circuit failure"},
    {179, "B->A summer circuit level failure"},
    {185, "Station A 4-wire receive level failure"},
    {186, "Station A 2-wire receive level failure"},
    {187, "B->A dial tone level failure"},
    {188, "B->A dial tone level failure"},
    {189, "B->A PCM/ADPCM module (SP3C #2) level failure"},
    {191, "B->A PCM/ADPCM module (SP3C #2) bad response"},
    {192, "B->A PCM/ADPCM module (SP3C #2) did not accept data"},
    {193, "B->A PCM/ADPCM module (SP3C #2) no response"},
    {194, "Invalid command for selected central office emulation mode"},
    {200, "A->B extended PCM/ADPCM (EPAL) level failure"},
    {201, "B->A extended PCM/ADPCM (EPAL) level failure"},
    {202, "A->B digital network section 2 (DNS1) level failure"},
    {203, "B->A digital network section 2 (DNS1) level failure"},
    {204, "A->B digital network section 2 (DNS2) level failure"},
    {205, "B->A digital network section 2 (DNS2) level failure"},
    {-1, ""}
};

tag_t gd_w_tags[] =
{
    {0, "Flat gain characteristic"},
    {1, "Low frequency gain slope characteristic #1"},
    {2, "Low frequency gain slope characteristic #2"},
    {3, "Low frequency gain slope characteristic #3"},
    {4, "Low frequency gain slope characteristic #4"},
    {5, "Low frequency gain slope characteristic #5"},
    {6, "Low frequency gain slope characteristic #6"},
    {7, "Low frequency gain slope characteristic #7"},
    {8, "SEG 2002 gain characteristic emulation (Seg FA-1445)"},
    {9, "Worst case Bell 3002 (C0) gain characteristic"},
    {10, "Worst case Bell C1 gain characteristic"},
    {11, "Worst case Bell C2 gain characteristic"},
    {12, "Worst case Bell C4 gain characteristic"},
    {13, "Worst case CCITT M1020 gain characteristic"},
    {14, "Worst case CCITT M1025 gain characteristic"},
    {15, "Worst case CCITT M1040 gain characteristic"},
#if 1
    {16, "EIA 'A' enhanced standard gain characteristic for modem testing"},
    {17, "EIA 'B' enhanced standard gain characteristic for modem testing"},
    {18, "EIA 'C' enhanced standard gain characteristic for modem testing"},
#else
    {16, "EIA 'A' standard gain characteristic for modem testing"},
    {17, "EIA 'B' standard gain characteristic for modem testing"},
    {18, "EIA 'C' standard gain characteristic for modem testing"},
#endif
    {19, "High frequency gain slope characteristic #1"},
    {20, "High frequency gain slope characteristic #2"},
    {21, "High frequency gain slope characteristic #3"},
    {22, "High frequency gain slope characteristic #4"},
    {23, "High frequency gain slope characteristic #5"},
    {24, "High frequency gain slope characteristic #6"},
    {25, "High frequency gain slope characteristic #7"},
    {26, "CCITT cable - 1 gain characteristic"},
    {27, "CCITT cable - 2 gain characteristic"},
    {28, "CCITT cable - 3 gain characteristic"},
    {33, "CONUS mid data gain characteristic"},
    {34, "CONUS mid voice gain characteristic"},
    {35, "CONUS poor data gain characteristic"},
    {36, "CONUS poor voice gain characteristic"},
    {37, "European mid data gain characteristic"},
    {38, "European mid voice gain characteristic"},
    {39, "European poor voice gain characteristic"},
    {40, "NSB gain characteristic"},
    {41, "NTB gain characteristic"},
    {42, "European poor data gain characteristic"},
    {43, "Japanese (JPN) link 1 gain characteristic"},
    {44, "Japanese (JPN) link 2 gain characteristic"},
    {45, "Japanese (JPN) link 3 gain characteristic"},
    {46, "Japanese (JPN) link 4 gain characteristic"},
    {47, "Japanese (JPN) link 5 gain characteristic"},
    {48, "Japanese (JPN) link 6 gain characteristic"},
    {49, "Japanese (JPN) link 7 gain characteristic"},
    {50, "Japanese 4dB local loop gain characteristic"},
    {51, "Japanese 8dB local loop gain characteristic"},
    {52, "Japanese 12dB local loop gain characteristic"},
    {53, "CCITT R28 gain characteristic"},
    {54, "French line 1 gain characteristic"},
    {55, "French line 2 gain characteristic"},
    {56, "French line 3 gain characteristic"},
    {57, "French line 4 gain characteristic"},
    {58, "NET 20 test channel 1 gain characteristic"},
    {59, "RITT (China) gain characteristic"},
    {60, "TR 50150 ('True Voice') gain characteristic"},
    {61, "ASIA 1 gain characteristic"},
    {62, "ASIA 2 gain characteristic"},
    {-1, ""}
};

tag_t gd_x_tags[] =
{
    {0, "Flat gain characteristic"},
    {1, "High frequency gain slope characteristic #1"},
    {2, "High frequency gain slope characteristic #2"},
    {3, "High frequency gain slope characteristic #3"},
    {4, "High frequency gain slope characteristic #4"},
    {5, "High frequency gain slope characteristic #5"},
    {6, "High frequency gain slope characteristic #6"},
    {7, "High frequency gain slope characteristic #7"},
    {8, "SEG 3002 gain characteristic emulation (Seg FA-1445)"},
    {9, "Worst case Bell 3002 (C0) gain characteristic"},
    {10, "Worst case Bell C1 gain characteristic"},
    {11, "Worst case Bell C2 gain characteristic"},
    {12, "Worst case Bell C4 gain characteristic"},
    {13, "Worst case CCITT M1020 gain characteristic"},
    {14, "Worst case CCITT M1025 gain characteristic"},
    {15, "Worst case CCITT M1040 gain characteristic"},
#if 1
    {16, "EIA 'A' enhanced standard gain characteristic for modem testing"},
    {17, "EIA 'B' enhanced standard gain characteristic for modem testing"},
    {18, "EIA 'C' enhanced standard gain characteristic for modem testing"},
#else
    {16, "EIA 'A' standard gain characteristic for modem testing"},
    {17, "EIA 'B' standard gain characteristic for modem testing"},
    {18, "EIA 'C' standard gain characteristic for modem testing"},
#endif
    {19, "Low frequency gain slope characteristic #1"},
    {20, "Low frequency gain slope characteristic #2"},
    {21, "Low frequency gain slope characteristic #3"},
    {22, "Low frequency gain slope characteristic #4"},
    {23, "Low frequency gain slope characteristic #5"},
    {24, "Low frequency gain slope characteristic #6"},
    {25, "Low frequency gain slope characteristic #7"},
    {26, "CCITT cable - 1 gain characteristic"},
    {27, "CCITT cable - 2 gain characteristic"},
    {28, "CCITT cable - 3 gain characteristic"},
    {33, "CONUS mid data gain characteristic"},
    {34, "CONUS mid voice gain characteristic"},
    {35, "CONUS poor data gain characteristic"},
    {36, "CONUS poor voice gain characteristic"},
    {37, "European mid data gain characteristic"},
    {38, "European mid voice gain characteristic"},
    {39, "European poor voice gain characteristic"},
    {40, "NSB gain characteristic"},
    {41, "NTB gain characteristic"},
    {42, "European poor data gain characteristic"},
    {43, "Japanese (JPN) link 1 gain characteristic"},
    {44, "Japanese (JPN) link 2 gain characteristic"},
    {45, "Japanese (JPN) link 3 gain characteristic"},
    {46, "Japanese (JPN) link 4 gain characteristic"},
    {47, "Japanese (JPN) link 5 gain characteristic"},
    {48, "Japanese (JPN) link 6 gain characteristic"},
    {49, "Japanese (JPN) link 7 gain characteristic"},
    {50, "Japanese 4dB local loop gain characteristic"},
    {51, "Japanese 8dB local loop gain characteristic"},
    {52, "Japanese 12dB local loop gain characteristic"},
    {53, "CCITT R28 gain characteristic"},
    {54, "French line 1 gain characteristic"},
    {55, "French line 2 gain characteristic"},
    {56, "French line 3 gain characteristic"},
    {57, "French line 4 gain characteristic"},
    {58, "NET 20 test channel 1 gain characteristic"},
    {59, "RITT (China) gain characteristic"},
    {60, "TR 50150 ('True Voice') gain characteristic"},
    {61, "ASIA 1 gain characteristic"},
    {62, "ASIA 2 gain characteristic"},
    {-1, ""}
};

tag_t gd_y_tags[] =
{
    {0, "Flat delay characteristic"},
    {1, "Low frequency delay slope characteristic #1"},
    {2, "Low frequency delay slope characteristic #2"},
    {3, "Low frequency delay slope characteristic #3"},
    {4, "Low frequency delay slope characteristic #4"},
    {5, "Low frequency delay slope characteristic #5"},
    {6, "Low frequency delay slope characteristic #6"},
    {7, "Low frequency delay slope characteristic #7"},
    {8, "SEG 3002 group delay characteristic emulation (Seg FA-1445)"},
    {9, "Worst case Bell 3002 (C0) group delay characteristic"},
    {10, "Worst case Bell C1 group delay characteristic"},
    {11, "Worst case Bell C2 group delay characteristic"},
    {12, "Worst case Bell C4 group delay characteristic"},
    {13, "Worst case CCITT M1020 group delay characteristic"},
    {14, "Worst case CCITT M1025 group delay characteristic"},
#if 1
    {15, "EIA '1' enhanced standard group delay characteristic for modem testing"},
    {16, "EIA '2' enhanced standard group delay characteristic for modem testing"},
    {17, "EIA '3' enhanced standard group delay characteristic for modem testing"},
    {18, "EIA '4' enhanced standard group delay characteristic for modem testing"},
    {19, "EIA '5' enhanced standard group delay characteristic for modem testing"},
#else
    {15, "EIA '1' standard group delay characteristic for modem testing"},
    {16, "EIA '2' standard group delay characteristic for modem testing"},
    {17, "EIA '3' standard group delay characteristic for modem testing"},
    {18, "EIA '4' standard group delay characteristic for modem testing"},
    {19, "EIA '5' standard group delay characteristic for modem testing"},
#endif
    {20, "High frequency delay slope characteristic #1"},
    {21, "High frequency delay slope characteristic #2"},
    {22, "High frequency delay slope characteristic #3"},
    {23, "High frequency delay slope characteristic #4"},
    {24, "High frequency delay slope characteristic #5"},
    {25, "High frequency delay slope characteristic #6"},
    {26, "High frequency delay slope characteristic #7"},
    {33, "CONUS mid data delay characteristic"},
    {34, "CONUS mid voice delay characteristic"},
    {35, "CONUS poor data delay characteristic"},
    {36, "CONUS poor voice delay characteristic"},
    {37, "European mid data delay characteristic"},
    {38, "European mid voice delay characteristic"},
    {39, "European poor voice delay characteristic"},
    {40, "NSB delay characteristic"},
    {41, "NTB delay characteristic"},
    {42, "European poor data delay characteristic"},
    {43, "Japanese (JPN) link 1 delay characteristic"},
    {44, "Japanese (JPN) link 2 delay characteristic"},
    {45, "Japanese (JPN) link 3 delay characteristic"},
    {46, "Japanese (JPN) link 4 delay characteristic"},
    {47, "Japanese (JPN) link 5 delay characteristic"},
    {48, "Japanese (JPN) link 6 delay characteristic"},
    {49, "Japanese (JPN) link 7 delay characteristic"},
    {50, "CCITT R28 delay characteristic"},
    {51, "French line 1 delay characteristic"},
    {52, "French line 2 delay characteristic"},
    {53, "French line 3 delay characteristic"},
    {54, "French line 4 delay characteristic"},
    {55, "NET 20 test channel 1 delay characteristic"},
    {56, "RITT1 (China) delay characteristic"},
    {57, "RITT2 (China) delay characteristic"},
    {58, "ASIA 1 delay characteristic"},
    {59, "ASIA 2 delay characteristic"},
    {-1, ""}
};

tag_t gd_z_tags[] =
{
    {0, "Flat delay characteristic"},
    {1, "High frequency delay slope characteristic #1"},
    {2, "High frequency delay slope characteristic #2"},
    {3, "High frequency delay slope characteristic #3"},
    {4, "High frequency delay slope characteristic #4"},
    {5, "High frequency delay slope characteristic #5"},
    {6, "High frequency delay slope characteristic #6"},
    {7, "High frequency delay slope characteristic #7"},
    {8, "SEG 3002 group delay characteristic emulation (Seg FA-1445)"},
    {9, "Worst case Bell 3002 (C0) group delay characteristic"},
    {10, "Worst case Bell C1 group delay characteristic"},
    {11, "Worst case Bell C2 group delay characteristic"},
    {12, "Worst case Bell C4 group delay characteristic"},
    {13, "Worst case CCITT M1020 group delay characteristic"},
    {14, "Worst case CCITT M1025 group delay characteristic"},
#if 1
    {15, "EIA '1' enhanced standard group delay characteristic for modem testing"},
    {16, "EIA '2' enhanced standard group delay characteristic for modem testing"},
    {17, "EIA '3' enhanced standard group delay characteristic for modem testing"},
    {18, "EIA '4' enhanced standard group delay characteristic for modem testing"},
    {19, "EIA '5' enhanced standard group delay characteristic for modem testing"},
#else
    {15, "EIA '1' standard group delay characteristic for modem testing"},
    {16, "EIA '2' standard group delay characteristic for modem testing"},
    {17, "EIA '3' standard group delay characteristic for modem testing"},
    {18, "EIA '4' standard group delay characteristic for modem testing"},
    {19, "EIA '5' standard group delay characteristic for modem testing"},
#endif
    {20, "Low frequency delay slope characteristic #1"},
    {21, "Low frequency delay slope characteristic #2"},
    {22, "Low frequency delay slope characteristic #3"},
    {23, "Low frequency delay slope characteristic #4"},
    {24, "Low frequency delay slope characteristic #5"},
    {25, "Low frequency delay slope characteristic #6"},
    {26, "Low frequency delay slope characteristic #7"},
    {33, "CONUS mid data delay characteristic"},
    {34, "CONUS mid voice delay characteristic"},
    {35, "CONUS poor data delay characteristic"},
    {36, "CONUS poor voice delay characteristic"},
    {37, "European mid data delay characteristic"},
    {38, "European mid voice delay characteristic"},
    {39, "European poor voice delay characteristic"},
    {40, "NSB delay characteristic"},
    {41, "NTB delay characteristic"},
    {42, "European poor data delay characteristic"},
    {43, "Japanese (JPN) link 1 delay characteristic"},
    {44, "Japanese (JPN) link 2 delay characteristic"},
    {45, "Japanese (JPN) link 3 delay characteristic"},
    {46, "Japanese (JPN) link 4 delay characteristic"},
    {47, "Japanese (JPN) link 5 delay characteristic"},
    {48, "Japanese (JPN) link 6 delay characteristic"},
    {49, "Japanese (JPN) link 7 delay characteristic"},
    {50, "CCITT R28 delay characteristic"},
    {51, "French line 1 delay characteristic"},
    {52, "French line 2 delay characteristic"},
    {53, "French line 3 delay characteristic"},
    {54, "French line 4 delay characteristic"},
    {55, "NET 20 test channel 1 delay characteristic"},
    {56, "RITT1 (China) delay characteristic"},
    {57, "RITT2 (China) delay characteristic"},
    {58, "ASIA 1 delay characteristic"},
    {59, "ASIA 2 delay characteristic"},
    {-1, ""}
};

tag_t dint_ff_tags[] =
{
    {0, "T1 D4/SF (superframe format)"},
    {1, "T1 ESF (extended superframe format)"},
    {2, "E1 (MFC/R2)"},
    {3, "TDM (time division multiplexed)"},
    {4, "ISDN BRI (U interface)"},
    {5, "ISDN PRI T1 D4/SF (superframe format"},
    {6, "ISDN PRI T1 ESF (extended superframe format"},
    {7, "ISDN PRI E1"},
    {-1, ""}
};

tag_t sf_m_tags[] =
{
    {0, "Fixed frequency generated"},
    {1, "Single frequency sweep"},
    {2, "Continuous frequency sweeps"},
    {-1, ""}
};

tag_t mm_r_tags[] =
{
    {0, "Station A transmit level and frequency"},
    {1, "Station B receive 4-wire level and frequency"},
    {2, "Station B receive 2-wire level and frequency"},
    {3, "Station B transmit level and frequency"},
    {4, "Station A receive 4-wire level and frequency"},
    {5, "Station A receive 2-wire level and frequency"},
    {-1, ""}
};

static instr_t *inst;

int universal_central_office_present = FALSE;
int nim2a_present = FALSE;
int satellite_delay_echo_present = FALSE;
int advanced_gain_delay_module_present = FALSE;
int pcm_adpcm_1_present = FALSE;
int pcm_adpcm_2_present = FALSE;
int channel_access_module_present = FALSE;
int extended_pcm_adpcm_module_present = FALSE;
int cellular_audio_processing_present = FALSE;
int dfe_present = FALSE;
int analog_network_section_1_present = FALSE;
int analog_network_section_2_present = FALSE;
int dns1_present = FALSE;
int dns2_present = FALSE;

int use_ieee_impulse_noise = FALSE;
int universal_central_office_mode = TRUE;

int debug = FALSE;

int check_test_set_config(void)
{
    char buf[MAX_GPIB_CMD_LEN];
    char field[MAX_GPIB_CMD_LEN];
    char field2[MAX_GPIB_CMD_LEN];
    const char *x;

    /* Check some stuff - modules present, version numbers, etc.
       at startup */

    /* Inquire about the software version and power-up diagnostics */
    if (instr_exchange(inst, "AD:R", buf) == 0)
    {
        if ((x = scan_for_field("M", buf, field)))
            printf("Test set model %s\n", field);
        if ((x = scan_for_field("V", buf, field)))
            printf("Software version %s\n", field);
        if ((x = scan_for_field("R", buf, field)))
            printf("System power up test result %s\n", field);
        printf("\n");
    }
    /* Inquire about the installed options */
    if (instr_exchange(inst, "AD:O", buf) == 0)
    {
        if (scan_for_field("O", buf, field))
        {
            if (strlen(field) > 1  &&  field[1] == '1')
            {
                printf("Network simulator module (NIM 2A) present");
                if (instr_exchange(inst, "AD:V01", buf) == 0)
                {
                    if ((x = scan_for_field("V", buf, field2)))
                        printf(" - version %s", field2);
                }
                printf("\n");
                nim2a_present = TRUE;
            }
            if (strlen(field) > 3  &&  field[3] == '1')
            {
                printf("Satellite delay/echo module present");
                if (instr_exchange(inst, "AD:V03", buf) == 0)
                {
                    if ((x = scan_for_field("V", buf, field2)))
                        printf(" - version %s", field2);
                }
                printf("\n");
                satellite_delay_echo_present = TRUE;
            }
            if (strlen(field) > 4  &&  field[4] == '1')
            {
                printf("Advanced gain/delay module present");
                if (instr_exchange(inst, "AD:V04", buf) == 0)
                {
                    if ((x = scan_for_field("V", buf, field2)))
                        printf(" - version %s", field2);
                }
                printf("\n");
                advanced_gain_delay_module_present = TRUE;
            }
        }
    }
    /* The UCO present indicator doesn't seem to be in the bit fields above */
    if (instr_exchange(inst, "AD:Q4", buf) == 0)
    {
        if ((x = scan_for_field("Q4", buf, field))  &&  strcmp(field, "1") == 0)
        {
            printf("Universal central office (UCO) module present\n");
            universal_central_office_present = TRUE;
        }
    }
    /* Inquire about the installed options */
    if (instr_exchange(inst, "AD:Q1", buf) == 0)
    {
        if ((x = scan_for_field("Q1", buf, field))  &&  strcmp(field, "1") == 0)
        {
            printf("PCM/ADPCM links module #1 present");
            if (instr_exchange(inst, "AD:V05", buf) == 0)
            {
                if ((x = scan_for_field("V", buf, field2)))
                    printf(" - version %s\n", field2);
            }
            printf("\n");
            pcm_adpcm_1_present = TRUE;
        }
    }
    if (instr_exchange(inst, "AD:Q2", buf) == 0)
    {
        if ((x = scan_for_field("Q2", buf, field))  &&  strcmp(field, "1") == 0)
        {
            printf("PCM/ADPCM links module #2 present");
            if (instr_exchange(inst, "AD:V06", buf) == 0)
            {
                if ((x = scan_for_field("V", buf, field2)))
                    printf(" - version %s\n", field2);
            }
            printf("\n");
            pcm_adpcm_2_present = TRUE;
        }
    }
    if (instr_exchange(inst, "AD:Q3", buf) == 0)
    {
        if ((x = scan_for_field("Q3", buf, field))  &&  strcmp(field, "1") == 0)
        {
            printf("Channel access module (CAM) present\n");
            channel_access_module_present = TRUE;
        }
    }
    if (instr_exchange(inst, "AD:Q5", buf) == 0)
    {
        if ((x = scan_for_field("Q5", buf, field))  &&  strcmp(field, "1") == 0)
        {
            printf("Extended PCM/ADPCM (EPAL) module present");
            if (instr_exchange(inst, "AD:V09", buf) == 0)
            {
                if ((x = scan_for_field("V", buf, field2)))
                    printf(" - version %s", field2);
            }
            printf("\n");
            extended_pcm_adpcm_module_present = TRUE;
        }
    }
    if (instr_exchange(inst, "AD:Q6", buf) == 0)
    {
        if ((x = scan_for_field("Q6", buf, field))  &&  strcmp(field, "1") == 0)
        {
            printf("Cellular audio processing (CAP) module present\n");
            if (instr_exchange(inst, "AD:V10", buf) == 0)
            {
                if ((x = scan_for_field("V", buf, field2)))
                    printf(" - version %s", field2);
            }
            printf("\n");
            cellular_audio_processing_present = TRUE;
        }
    }
    if (instr_exchange(inst, "AD:Q7", buf) == 0)
    {
        if ((x = scan_for_field("Q7", buf, field)))
        {
            switch (field[0])
            {
            case '1':
                printf("DFE with T1");
                dfe_present = TRUE;
                break;
            case '2':
                printf("DFE with T1 and TDM");
                dfe_present = TRUE;
                break;
            case '3':
                printf("DFE with T1, TDM and BRI");
                dfe_present = TRUE;
                break;
            case '4':
                printf("DFE with T1, TDM, BRI and E1");
                dfe_present = TRUE;
                break;
            case '5':
                printf("DFE with T1, TDM, BRI, E1 and PRI");
                dfe_present = TRUE;
                break;
            }
            if (dfe_present)
            {
                printf(" present");
                if (instr_exchange(inst, "AD:V11", buf) == 0)
                {
                    if ((x = scan_for_field("V", buf, field2)))
                        printf(" - version %s", field2);
                }
                printf("\n");
            }
        }
    }
    if (instr_exchange(inst, "AD:Q8", buf) == 0)
    {
        if ((x = scan_for_field("Q8", buf, field))  &&  strcmp(field, "1") == 0)
        {
            printf("Analog network section 1 (ANS) module present\n");
            analog_network_section_1_present = TRUE;
        }
    }
    if (instr_exchange(inst, "AD:Q9", buf) == 0)
    {
        if ((x = scan_for_field("Q9", buf, field))  &&  strcmp(field, "1") == 0)
        {
            printf("Analog network section 2 (ANS) module present\n");
            analog_network_section_2_present = TRUE;
        }
    }
    if (instr_exchange(inst, "AD:V07", buf) == 0)
    {
        if ((x = scan_for_field("V", buf, field2)))
            printf("A->B signal processor (SP3A #1) module - version %s\n", field2);
    }
    if (instr_exchange(inst, "AD:V08", buf) == 0)
    {
        if ((x = scan_for_field("V", buf, field2)))
            printf("B->A signal processor (SP3A #2) module - version %s\n", field2);
    }
    if (instr_exchange(inst, "AD:Q10", buf) == 0)
    {
        if ((x = scan_for_field("Q10", buf, field))  &&  strcmp(field, "1") == 0)
        {
            printf("Digital network section 1 (DNS) module present");
            if (instr_exchange(inst, "AD:V12", buf) == 0)
            {
                if ((x = scan_for_field("V", buf, field2)))
                    printf(" - version %s", field2);
            }
            printf("\n");
            dns1_present = TRUE;
        }
    }
    if (instr_exchange(inst, "AD:Q11", buf) == 0)
    {
        if ((x = scan_for_field("Q11", buf, field))  &&  strcmp(field, "1") == 0)
        {
            printf("Digital network section 2 (DNS) module present");
            if (instr_exchange(inst, "AD:V13", buf) == 0)
            {
                if ((x = scan_for_field("V", buf, field2)))
                    printf(" - version %s", field2);
            }
            printf("\n");
            dns2_present = TRUE;
        }
    }
    printf("\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int do_self_test(void)
{
    return instr_exchange_c(inst, "IO:C");
}
/*- End of function --------------------------------------------------------*/

int config_pcm_adpcm(void)
{
    if (!pcm_adpcm_1_present  &&  !pcm_adpcm_2_present)
        return -1;
    
    /* Section 3.6.19 */

    printf("Configuring PCM/ADPCM");

    if (instr_exchange_c(inst, "PC:C1%d", 0) == 0)
        printf(".");
    
    if (instr_exchange_c(inst, "PC:C2%d", 0) == 0)
        printf(".");
    
    if (instr_exchange_c(inst, "PC:C3%d", 0) == 0)
        printf(".");
    
    if (instr_exchange_c(inst, "PC:C4%d", 0) == 0)
        printf(".");
    
    if (instr_exchange_c(inst, "PC:Q1%d", 0) == 0)
        printf(".");
    
    if (instr_exchange_c(inst, "PC:Q2%d", 0) == 0)
        printf(".");
    
    if (instr_exchange_c(inst, "PC:Q3%d", 0) == 0)
        printf(".");
    
    if (instr_exchange_c(inst, "PC:Q4%d", 0) == 0)
        printf(".");
    
    if (instr_exchange_c(inst, "PC:P%s", "1010") == 0)
        printf(".");
    
    if (instr_exchange_c(inst, "PC:D%s", "0000") == 0)
        printf(".");

    /* Enable (1) or disable (0) RBS */
    if (instr_exchange_c(inst, "PC:S%d", 0) == 0)
        printf(".");
    
    if (instr_exchange_c(inst, "PC:E%d", 1) == 0)
        printf(".");
    
    if (instr_exchange_c(inst, "PC:I%d", 0) == 0)
        printf(".");
    
    if (instr_exchange_c(inst, "PC:B%d", 0) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_epal(void)
{
    if (!extended_pcm_adpcm_module_present)
        return -1;
        
    /* Section 3.6.7 */

    printf("Configuring EPAL");

    if (instr_exchange_c(inst, "EP:C%d", 11) == 0)
        printf(".");

    if (instr_exchange_c(inst, "EP:Q%d", 10) == 0)
        printf(".");

    if (instr_exchange_c(inst, "EP:M%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "EP:I%d", -50) == 0)
        printf(".");

    if (instr_exchange_c(inst, "EP:O%d", -100) == 0)
        printf(".");

    if (instr_exchange_c(inst, "EP:T%d", 600) == 0)
        printf(".");

    if (instr_exchange_c(inst, "EP:U%d", 1) == 0)
        printf(".");

    if (instr_exchange_c(inst, "EP:B%d", 1) == 0)
        printf(".");

    if (instr_exchange_c(inst, "EP:R%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "EP:E%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "EP:L%d", 1) == 0)
        printf(".");

    /* Enable (1) or disable (0) frame slips */
    if (instr_exchange_c(inst, "EP:S%d", 0) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int do_frame_slip(void)
{
    if (instr_exchange_c(inst, "EP:G") == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_frequency_shifter(void)
{
    /* Section 3.6.8 */

    printf("Configuring frequency shifter");

    /* Set frequency shift to 0.005Hz steps */
    if (instr_exchange_c(inst, "FS:M%d", 0) == 0)
        printf(".");

    /* Set frequency shift to -10Hz */
    if (instr_exchange_c(inst, "FS:F%d", -1999) == 0)
        printf(".");
    
    /* Enable (1) or disable (0) frequency shift */
    if (instr_exchange_c(inst, "FS:S%d", 0) == 0)
        printf(".");
    
    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_gain_delay(int w, int x, int y, int z)
{
    /* Section 3.6.9 */

    printf("Configuring gain/delay");

    /* Set gain to flat */
    if (instr_exchange_c(inst, "GD:W%02d", w) == 0)
        display_explanation(gd_w_tags, w);

    /* Set gain filter to flat */
    if (instr_exchange_c(inst, "GD:X%02d", x) == 0)
        display_explanation(gd_x_tags, x);

    /* Set group delay filter #2 to none */
    if (instr_exchange_c(inst, "GD:Y%02d", y) == 0)
        display_explanation(gd_y_tags, y);

    /* Set group delay filter #1 to none */
    if (instr_exchange_c(inst, "GD:Z%02d", z) == 0)
        display_explanation(gd_z_tags, z);
    
    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_gain_hits(void)
{
    /* Section 3.6.10 */

    printf("Configuring gain hits");

    if (instr_exchange_c(inst, "GH:L%d", -100) == 0)
        printf(".");

    if (instr_exchange_c(inst, "GH:R%d", 2) == 0)
        printf(".");

    if (instr_exchange_c(inst, "GH:D%d", 8) == 0)
        printf(".");

    if (instr_exchange_c(inst, "GH:I%d", 100) == 0)
        printf(".");

    if (instr_exchange_c(inst, "GH:M%d", 1) == 0)
        printf(".");

    /* Enable (1) or disable (0) gain hits */
    if (instr_exchange_c(inst, "GH:S%d", 0) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int do_gain_hit(void)
{
    if (instr_exchange_c(inst, "GH:T") == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_phase_hits(void)
{
    /* Section 3.6.20 */

    printf("Configuring phase hits");

    /* Phase hit level, in 0.022degree steps (0.0-180.0degrees) */
    if (instr_exchange_c(inst, "PH:L%d", 2048) == 0)
        printf(".");

    if (instr_exchange_c(inst, "PH:R%d", 2) == 0)
        printf(".");

    if (instr_exchange_c(inst, "PH:D%d", 8) == 0)
        printf(".");

    if (instr_exchange_c(inst, "PH:I%d", 100) == 0)
        printf(".");

    /* Pseudo-random (1) or regular (0) hit interval */
    if (instr_exchange_c(inst, "PH:M%d", 1) == 0)
        printf(".");

    /* Enable (1) or disable (0) phase hits */
    if (instr_exchange_c(inst, "PH:S%d", 0) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int do_phase_hit(void)
{
    if (instr_exchange_c(inst, "PH:T") == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_impulse_noise(void)
{
    /* Section 3.6.11 and 3.6.12 */
    
    printf("Configuring impulse noise");
    
    if (use_ieee_impulse_noise)
    {
        /* Set impulse noise (IEEE) ??? */
        if (instr_exchange_c(inst, "IMP1:L%d", 340) == 0)
            printf(".");
    
        if (instr_exchange_c(inst, "IMP1:I%d", 100) == 0)
            printf(".");
    
        if (instr_exchange_c(inst, "IMP1:W%d", 0) == 0)
            printf(".");
    
        if (instr_exchange_c(inst, "IMP1:M%d", 0) == 0)
            printf(".");
    
        if (instr_exchange_c(inst, "IMP1:P%d", 0) == 0)
            printf(".");
    
        if (instr_exchange_c(inst, "IMP1:S%d", 0) == 0)
            printf(".");
    }
    else
    {
        /* Set impulse noise (bipolar) ??? */
        if (instr_exchange_c(inst, "IMP2:L%d", -250) == 0)
            printf(".");

        if (instr_exchange_c(inst, "IMP2:I%d", 100) == 0)
            printf(".");

        if (instr_exchange_c(inst, "IMP2:D%d", 2) == 0)
            printf(".");

        if (instr_exchange_c(inst, "IMP2:M%d", 0) == 0)
            printf(".");

        if (instr_exchange_c(inst, "IMP2:P%d", 0) == 0)
            printf(".");

        if (instr_exchange_c(inst, "IMP2:S%d", 0) == 0)
            printf(".");
    }
    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int do_impulse_noise(void)
{
    if (use_ieee_impulse_noise)
    {
        /* Set impulse noise (IEEE) ??? */
        if (instr_exchange_c(inst, "IMP1:T") == 0)
            printf(".");
    }
    else
    {
        /* Set impulse noise (bipolar) ??? */
        if (instr_exchange_c(inst, "IMP2:T") == 0)
            printf(".");
    }
    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_non_linear_distortion(void)
{
    /* Section 3.6.18 */

    printf("Configuring non-linear distortion");
    
    /* Second order distortion, in 0.1dB steps (20.0-60.0dB below the signal) */
    if (instr_exchange_c(inst, "NL:Q%d", 400) == 0)
        printf(".");

    /* Third order distortion, in 0.1dB steps (20.0-60.0dB below the signal) */
    if (instr_exchange_c(inst, "NL:C%d", 200) == 0)
        printf(".");

    /* Expansive (0) or compressive (1) distortion */
    if (instr_exchange_c(inst, "NL:M%d", 0) == 0)
        printf(".");

    /* Enable (1) or disable (0) second order distortion */
    if (instr_exchange_c(inst, "NL:X%d", 0) == 0)
        printf(".");

    /* Enable (1) or disable (0) third order distortion */
    if (instr_exchange_c(inst, "NL:Y%d", 0) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_network_signaling(void)
{
    if (universal_central_office_mode)
        return -1;

    /* Section 3.6.26 */

    printf("Configuring network signaling");
    
    /* Busy tone on time, in 50ms steps (0-60000ms) */
    if (instr_exchange_c(inst, "SG:BA%d", 10) == 0)
        printf(".");

    /* Busy tone off time, in 50ms steps (0-60000ms) */
    if (instr_exchange_c(inst, "SG:BB%d", 10) == 0)
        printf(".");

    if (instr_exchange_c(inst, "SG:C%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "SG:D%d", -10) == 0)
        printf(".");

    /* Enable (1) or disable (0) routing tone */
    if (instr_exchange_c(inst, "SG:E%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "SG:FA%d", 4400) == 0)
        printf(".");

    /* Loop current, in 6mA steps (18-90mA) - obsolete */
    //if (instr_exchange_c(inst, "SG:I%d", 9) == 0)
    //    printf(".");

    /* Station A loop current, in 2mA steps (10-126mA) */
    if (instr_exchange_c(inst, "SG:JA%d", 9) == 0)
        printf(".");

    /* Station B loop current, in 2mA steps (10-126mA) */
    if (instr_exchange_c(inst, "SG:JB%d", 9) == 0)
        printf(".");

    /* Station A loop current polarity (0=positive, 1=negative) */
    if (instr_exchange_c(inst, "SG:KA%d", 0) == 0)
        printf(".");

    /* Station B loop current polarity (0=positive, 1=negative) */
    if (instr_exchange_c(inst, "SG:KB%d", 0) == 0)
        printf(".");

    /* Ring voltage, in 5Vrms steps (5-100Vrms) - obsolete */
    //if (instr_exchange_c(inst, "SG:L%d", 17) == 0)
    //    printf(".");

    /* Ring level in 1V steps (1-100V) */
    if (instr_exchange_c(inst, "SG:A%d", 80) == 0)
        printf(".");

    if (instr_exchange_c(inst, "SG:MC") == 0)
        printf(".");

    if (instr_exchange_c(inst, "SG:PA%d", 45) == 0)
        printf(".");

    if (instr_exchange_c(inst, "SG:PB%d", 75) == 0)
        printf(".");

    if (instr_exchange_c(inst, "SG:PC%d", 45) == 0)
        printf(".");

    if (instr_exchange_c(inst, "SG:PD%d", 75) == 0)
        printf(".");

    if (instr_exchange_c(inst, "SG:Q%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "SG:RA%d", 10) == 0)
        printf(".");

    if (instr_exchange_c(inst, "SG:RB%d", 10) == 0)
        printf(".");

    if (instr_exchange_c(inst, "SG:RC%d", 10) == 0)
        printf(".");

    if (instr_exchange_c(inst, "SG:RD%d", 10) == 0)
        printf(".");

    if (instr_exchange_c(inst, "SG:SA") == 0)
        printf(".");

    if (instr_exchange_c(inst, "SG:W%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "SG:Y%d", 200) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int monitor_network_signaling_status(void)
{
    char buf[MAX_GPIB_CMD_LEN];
    char field[MAX_GPIB_CMD_LEN];
    const char *x;
    int i;

    if (universal_central_office_mode)
        return -1;

    for (i = 'A';  i <= 'B';  i++)
    {
        if (instr_exchange(inst, "SG:Z%c", buf, i) == 0)
        {
            if ((x = scan_for_field("Z", buf, field)))
            {
                printf("Station %c status %s\n", i, field);
                if (strlen(field) >= 8)
                {
                    if (field[0] == '1')
                        printf("  Audible ringing on\n");
                    if (field[1] == '1')
                        printf("  Ringing on\n");
                    if (field[2] == '1')
                        printf("  Busy on\n");
                    if (field[3] == '1')
                        printf("  Dial tone on\n");
                    if (field[4] == '1')
                        printf("  Off hook\n");
                    if (field[5] == '1')
                        printf("  Connected\n");
                    if (field[6] == '1')
                        printf("  Awaiting DTMF\n");
                    if (field[7] == '1')
                        printf("  Awaiting dial pulses\n");
                }
            }
        }
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_single_frequency_interference(void)
{
    /* Section 3.6.25 */

    printf("Configuring single frequency interference");
    
    /* Frequency, in Hz (100-3400Hz) */
    if (instr_exchange_c(inst, "SF:F%d", 2600) == 0)
        printf(".");

    /* Frequency step for sweep, in Hz (1-100Hz) */
    if (instr_exchange_c(inst, "SF:I%d", 10) == 0)
        printf(".");

    /* Level below signal, in 0.1dB steps (0.0-50.0dB) */
    if (instr_exchange_c(inst, "SF:L%d", 100) == 0)
        printf(".");

    /* Mode */
    if (instr_exchange_c(inst, "SF:M%d", 0) == 0)
        display_explanation(sf_m_tags, 0);

    /* Sweep period, in seconds (1-999s) */
    if (instr_exchange_c(inst, "SF:P%d", 300) == 0)
        printf(".");

    /* 1/3rd Hz steps to add to the frequency (0-2) */
    if (instr_exchange_c(inst, "SF:Q%d", 0) == 0)
        printf(".");

    /* Generator off (0) or on (1) */
    if (instr_exchange_c(inst, "SF:S%d", 0) == 0)
        printf(".");

    /* The lower limit of the frequency sweep, in Hz (16-3400Hz) */
    if (instr_exchange_c(inst, "SF:X%d", 2000) == 0)
        printf(".");

    /* The upper limit of the frequency sweep, in Hz (16-3400Hz) */
    if (instr_exchange_c(inst, "SF:Y%d", 2100) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_satellite_delays(int which)
{
    /* Section 3.6.23 and 3.6.24 */

    printf("Configuring satellite delays %d", which);

    /* Channel delay in 0.125ms steps (0-1279.875ms for delay 1, 0-425ms
       for delay 2) */
    if (instr_exchange_c(inst, "SAT%d:D%d", which, 2500) == 0)
        printf(".");

    /* Delay off (0) or on (1) */
    if (instr_exchange_c(inst, "SAT%d:S%d", which, 0) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_amplitude_jitter(void)
{
    /* Section 3.6.2 */

    printf("Configuring amplitude jitter");

    /* Set level to 10% */
    if (instr_exchange_c(inst, "AJ:L%d", 410) == 0)
        printf(".");

    /* Set frequency to 10Hz */
    if (instr_exchange_c(inst, "AJ:F%d", 100) == 0)
        printf(".");

    /* Set waveform to sine */
    if (instr_exchange_c(inst, "AJ:W%d", 0) == 0)
        printf(".");

    /* Enable (1) or disable (0) amplitude jitter */
    if (instr_exchange_c(inst, "AJ:S%d", 0) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_phase_jitter(void)
{
    /* Section 3.6.21 */

    printf("Configuring phase jitter");

    if (instr_exchange_c(inst, "PJ:L%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "PJ:F%d", 600) == 0)
        printf(".");

    if (instr_exchange_c(inst, "PJ:W%d", 0) == 0)
        printf(".");

    /* Enable (1) or disable (0) phase jitter */
    if (instr_exchange_c(inst, "PJ:S%d", 0) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_white_noise(void)
{
    /* Section 3.6.22 */

    printf("Configuring white noise");

    /* Level, in 0.1dB steps (15.0-90.0dBm) */
    if (instr_exchange_c(inst, "RN:L%d", 320) == 0)
        printf(".");

    /* Weight (0=c-message, 1=3kHz flat, 2=15kHz flat, 3=NET 20, 4=psophometric) */
    if (instr_exchange_c(inst, "RN:W%d", 0) == 0)
        printf(".");

    /* Bandwidth (0=5kHz, 1=4kHz, 2=20kHz) */
    if (instr_exchange_c(inst, "RN:B%d", 0) == 0)
        printf(".");

    /* The period of the pseudo-random noise generator (0=20.97s, 1=5.97 hours) */
    if (instr_exchange_c(inst, "RN:P%d", 0) == 0)
        printf(".");

    /* Enable (1) or disable (0) white noise */
    if (instr_exchange_c(inst, "RN:S%d", 0) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_digital_central_office(void)
{
    if (!universal_central_office_mode)
        return -1;

    printf("Configuring digital central office (DCO)");

    /* Set T1/E1 digital line mode to switched digital to 2-wire switched analog */
    if (instr_exchange_c(inst, "DCO:MODE=%d", 1) == 0)
        printf(".");
        
    /* Addressing mode - DTMF (0) or MF (1) */
    if (instr_exchange_c(inst, "DCO:AM=%d", 0) == 0)
        printf(".");
        
    /* E1 compelled combination from CO */
    if (instr_exchange_c(inst, "DCO:ECCC=%s", "1111116") == 0)
        printf(".");
        
    /* E1 compelled signaling mode */
    if (instr_exchange_c(inst, "DCO:ECSM=%d", 0) == 0)
        printf(".");
        
    /* E1 expected combination sequence */
    if (instr_exchange_c(inst, "DCO:EECS=%s", "12345678") == 0)
        printf(".");
        
    /* E1 outward combination sequence */
    if (instr_exchange_c(inst, "DCO:EOCS=%s", "12345678") == 0)
        printf(".");

    /* TODO: a bunch of ISDN message definition */
    
    
    if (instr_exchange_c(inst, "DCO:MFF=%d", 0) == 0)
        printf(".");
    
    if (instr_exchange_c(inst, "DCO:ODN=%s", "5551234") == 0)
        printf(".");
    
    if (instr_exchange_c(inst, "DCO:SM=%d", 0) == 0)
        printf(".");
    
    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int monitor_abcd(void)
{
    char buf[MAX_GPIB_CMD_LEN];
    char field[MAX_GPIB_CMD_LEN];
    const char *x;

    if (!universal_central_office_mode)
        return -1;

    if (instr_exchange(inst, "DCO:ABCD", buf) == 0)
    {
        if ((x = scan_for_field("ABCD=", buf, field)))
            printf("ABCD bits %s\n", field);
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/
        
int do_direct_outward_dialing(void)
{
    if (!universal_central_office_mode)
        return -1;

    if (instr_exchange_c(inst, "DCO:OD=%d", 0) == 0)
        printf(".");
    
    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int do_isdn_response(int which)
{
    if (!universal_central_office_mode)
        return -1;

    if (instr_exchange_c(inst, "DCO:TIM=%d", which) == 0)
        printf(".");
    
    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int monitor_dco(void)
{
    char buf[MAX_GPIB_CMD_LEN];
    char field[MAX_GPIB_CMD_LEN];
    const char *x;

    if (instr_exchange(inst, "DCO:QIM", buf) == 0)
    {
        if ((x = scan_for_field("QIM", buf, field)))
            printf("Last status message %s\n", field);
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_dint(void)
{
    if (!universal_central_office_mode)
        return -1;

    printf("Configuring DINT");

    /* Set T1/E1 clock source */
    if (instr_exchange_c(inst, "DINT:CLK=%d", 0) == 0)
        printf(".");

    /* E1 transmission channel number (1-30) */
    if (instr_exchange_c(inst, "DINT:ECN=%d", 1) == 0)
        printf(".");

    /* E1 zero code suppression method - HDB3 (0) or none (1) */
    if (instr_exchange_c(inst, "DINT:EZCS=%d", 0) == 0)
        printf(".");

    /* Set frame format */
    if (instr_exchange_c(inst, "DINT:FF=%d", 2) == 0)
        display_explanation(dint_ff_tags, 2);

    /* Set PCM coding */
    if (instr_exchange_c(inst, "DINT:PCMC=%d", 0) == 0)
        printf(".");

    /* Set PRI-E1 transmission channel number */
    if (instr_exchange_c(inst, "DINT:PECN=%d", 1) == 0)
        printf(".");

    /* Set PRI-T1 transmission channel number */
    if (instr_exchange_c(inst, "DINT:PTCN=%d", 1) == 0)
        printf(".");

    /* Set T1 transmission channel number */
    if (instr_exchange_c(inst, "DINT:TCN=%d", 1) == 0)
        printf(".");
    
    /* Set TDM mode */
    if (instr_exchange_c(inst, "DINT:TDMM=%d", 0) == 0)
        printf(".");
    
    /* Set T1 line build out */
    if (instr_exchange_c(inst, "DINT:TLBO=%d", 0) == 0)
        printf(".");
    
    /* Set T1 zero crossing suppression method */
    if (instr_exchange_c(inst, "DINT:TZCS=%d", 0) == 0)
        printf(".");
    
    /* Set unused channel data */
    if (instr_exchange_c(inst, "DINT:UC=%02X", 0x7F) == 0)
        printf(".");
    
    /* Set unused channel data source */
    if (instr_exchange_c(inst, "DINT:UDS=%d", 1) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int monitor_digital_link_status(void)
{
    char buf[MAX_GPIB_CMD_LEN];
    char field[MAX_GPIB_CMD_LEN];
    const char *x;

    if (!universal_central_office_mode)
        return -1;

    /* Set T1 alarm status polling */
    if (instr_exchange(inst, "DINT:ALRM0", buf) == 0)
    {
        if ((x = scan_for_field("ALRM0=", buf, field)))
            printf("Alarms %s\n", field);
    }
    if (instr_exchange(inst, "DINT:SYNC", buf) == 0)
    {
        if ((x = scan_for_field("SYNC=", buf, field)))
            printf("Sync status %s\n", field);
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_dns(int which)
{
    /* Section 3.2 DNS */
    printf("Configuring DNS %d", which);

    /* Set DNS bypass status */
    if (instr_exchange_c(inst, "DS%d:BYP=%d", which, 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "DS%d:DELAY=%d", which, 0) == 0)
        printf(".");

    if (which == 1)
    {
        if (instr_exchange_c(inst, "DS1:ECLB=%d", 210) == 0)
            printf(".");

        /* Set intermediate echo status to disabled */
        if (instr_exchange_c(inst, "DS1:ECSB=%d", 0) == 0)
            printf(".");
    }
    else
    {
        if (instr_exchange_c(inst, "DS2:ECLA=%d", 210) == 0)
            printf(".");

        /* Set intermediate echo status to disabled */
        if (instr_exchange_c(inst, "DS2:ECSA=%d", 0) == 0)
            printf(".");
    }

    /* Set 2nd order IMD level to 40dB below signal */
    if (instr_exchange_c(inst, "DS%d:IMD2L=%d", which, 400) == 0)
        printf(".");

    if (instr_exchange_c(inst, "DS%d:IMD2S=%d", which, 0) == 0)
        printf(".");

    /* Set 3rd order IMD level to 40dB below signal */
    if (instr_exchange_c(inst, "DS%d:IMD3L=%d", which, 500) == 0)
        printf(".");

    if (instr_exchange_c(inst, "DS%d:IMD3S=%d", which, 0) == 0)
        printf(".");

    /* Set gain to -3dB */
    if (instr_exchange_c(inst, "DS%d:GAIN=%d", which, -30) == 0)
        printf(".");

    /* Set general impairment bypass to enable impairments */
    if (instr_exchange_c(inst, "DS%d:GIB=%d", which, 1) == 0)
        printf(".");

    /* Set noise level to 30.0dBm */
    if (instr_exchange_c(inst, "DS%d:NSEL=%d", which, 300) == 0)
        printf(".");

    /* Set noise status to on */
    if (instr_exchange_c(inst, "DS%d:NSES=%d", which, 1) == 0)
        printf(".");

    /* Set PCM coding for link 1 to A-law */
    if (instr_exchange_c(inst, "DS%d:PCMC1=%d", which, 2) == 0)
        printf(".");

    /* Set PCM coding for link 2 to A-law */
    if (instr_exchange_c(inst, "DS%d:PCMC2=%d", which, 2) == 0)
        printf(".");

    /* Enable the PCM links */
    if (instr_exchange_c(inst, "DS%d:PLB=%d", which, 1) == 0)
        printf(".");

    if (which == 1)
    {
        /* Set position of DNS module to after impairment summer (DS1 only) */
        if (instr_exchange_c(inst, "DS1:POS=%d", 1) == 0)
            printf(".");
    }

    /* Set PCM/ADPCM rate */
    if (instr_exchange_c(inst, "DS%d:PQ=%d", which, 0) == 0)
        printf(".");

    /* Set location of ADPCM quantisation to link 1 */
    if (instr_exchange_c(inst, "DS%d:PQL=%d", which, 1) == 0)
        printf(".");

    /* Set RBS impairment */
    if (instr_exchange_c(inst, "DS%d:RBSD1=%s", which, "1111") == 0)
        printf(".");
    
    if (instr_exchange_c(inst, "DS%d:RBSD2=%s", which, "1111") == 0)
        printf(".");
    
    /* Set RBS impairment link control to no RBS links */
    if (instr_exchange_c(inst, "DS%d:RBSL1=%d", which, 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "DS%d:RBSL2=%d", which, 0) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_micro_interruptions(int which)
{
    /* Section 3.6.15 and 3.6.16 */

    printf("Configuring MIC %d", which);

    /* MIC1: Interruption duration, in 1ms steps (1-20000ms) */
    /* MIC2: Interruption duration, in 0.1ms steps (1-6600ms) */
    if (instr_exchange_c(inst, "MIC%d:D%d", which, 10) == 0)
        printf(".");

    /* MIC1: Interruption interval (>duration), in 10ms steps (0.1-320.0s) */
    /* MIC2: Interruption interval (>duration), in 10ms steps (0.1-106.0s) */
    if (instr_exchange_c(inst, "MIC%d:I%d", which, 100) == 0)
        printf(".");

    /* Enable (1) or disable (0) micro-interruptions */
    if (instr_exchange_c(inst, "MIC%d:S%d", which, 0) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int do_micro_interruption(int which)
{
    if (instr_exchange_c(inst, "MIC%d:T", which) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_switching(void)
{
    if (universal_central_office_mode)
        return -1;

    /* Section 3.6.27 */

    printf("Configuring switching");

    /* Telephone number of station A (1-15 digits) */
    if (instr_exchange_c(inst, "SW:TA%s", "12345678") == 0)
        printf(".");

    /* Telephone number of station B (1-15 digits) */
    if (instr_exchange_c(inst, "SW:TB%s", "87654321") == 0)
        printf(".");

    /* Switching delay, in ms (1-60000ms) */
    if (instr_exchange_c(inst, "SW:M%d", 1) == 0)
        printf(".");

    /* Dial tone delay, in ms (1-60000ms) */
    if (instr_exchange_c(inst, "SW:N%d", 1) == 0)
        printf(".");

    /* On hook recognition disconnect delay, in ms (1-255ms) */
    if (instr_exchange_c(inst, "SW:Q%d", 255) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int monitor_switching(void)
{
    char buf[MAX_GPIB_CMD_LEN];
    char field[MAX_GPIB_CMD_LEN];
    const char *x;

    if (universal_central_office_mode)
        return -1;

    /* Section 3.6.27 */

    if (instr_exchange(inst, "SW:ZA", buf) == 0)
    {
        if ((x = scan_for_field("ZA", buf, field)))
            printf("Station A telephone number %s\n", field);
    }
    if (instr_exchange(inst, "SW:ZB", buf) == 0)
    {
        if ((x = scan_for_field("ZB", buf, field)))
            printf("Station B telephone number %s\n", field);
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_line_control(void)
{
    if (universal_central_office_mode)
        return -1;

    /* Section 3.6.14 */

    printf("Configuring line control");

    if (channel_access_module_present)
    {
        if (instr_exchange_c(inst, "LC:AA%d", 0) == 0)
            printf(".");

        if (instr_exchange_c(inst, "LC:AB%d", 0) == 0)
            printf(".");

        if (instr_exchange_c(inst, "LC:AC%d", 0) == 0)
            printf(".");

        if (instr_exchange_c(inst, "LC:AD%d", 0) == 0)
            printf(".");
    }
    if (instr_exchange_c(inst, "LC:BA%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "LC:BB%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "LC:D%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "LC:E%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "LC:G%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "LC:IA%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "LC:IB%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "LC:LA%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "LC:LB%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "LC:M%d", 1) == 0)
        printf(".");

    if (instr_exchange_c(inst, "LC:S%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "LC:V%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "LC:XA%d", 1) == 0)
        printf(".");

    if (instr_exchange_c(inst, "LC:XB%d", 1) == 0)
        printf(".");

    if (instr_exchange_c(inst, "LC:YA%d", 1) == 0)
        printf(".");

    if (instr_exchange_c(inst, "LC:YB%d", 1) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_echo_control(void)
{
    /* Section 3.6.6 */

    printf("Configuring line control");

    /* Station A near end echo attenuator */
    if (instr_exchange_c(inst, "EC:LA%d", 210) == 0)
        printf(".");

    /* Station A near end echo attenuator path */
    if (instr_exchange_c(inst, "EC:PA%d", 0) == 0)
        printf(".");

    /* Station A near end, enable (1) or disable (0) echo attenuator */
    if (instr_exchange_c(inst, "EC:SA%d", 0) == 0)
        printf(".");

    /* Station A far end echo attenuator */
    if (instr_exchange_c(inst, "EC:LB%d", 210) == 0)
        printf(".");

    /* Station A far end echo attenuator path */
    if (instr_exchange_c(inst, "EC:PB%d", 0) == 0)
        printf(".");

    /* Station A far end, enable (1) or disable (0) echo attenuator */
    if (instr_exchange_c(inst, "EC:SB%d", 0) == 0)
        printf(".");

    /* Station B near end echo attenuator */
    if (instr_exchange_c(inst, "EC:LC%d", 210) == 0)
        printf(".");

    /* Station B near end echo attenuator path */
    if (instr_exchange_c(inst, "EC:PC%d", 0) == 0)
        printf(".");

    /* Station B near end, enable (1) or disable (0) echo attenuator */
    if (instr_exchange_c(inst, "EC:SC%d", 0) == 0)
        printf(".");

    /* Station B far end echo attenuator */
    if (instr_exchange_c(inst, "EC:LD%d", 210) == 0)
        printf(".");

    /* Station B far end echo attenuator path */
    if (instr_exchange_c(inst, "EC:PD%d", 0) == 0)
        printf(".");

    /* Station B far end, enable (1) or disable (0) echo attenuator */
    if (instr_exchange_c(inst, "EC:SD%d", 0) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_impairment_generator(void)
{
    /* Section 3.6.13 */

    printf("Configuring impairment generator");

    if (instr_exchange_c(inst, "IO:D%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "IO:E%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "IO:F%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "IO:I%d", -100) == 0)
        printf(".");

    if (instr_exchange_c(inst, "IO:L%d", -100) == 0)
        printf(".");

    if (instr_exchange_c(inst, "IO:M%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "IO:N%d", 1) == 0)
        printf(".");

    if (instr_exchange_c(inst, "IO:R%d", -100) == 0)
        printf(".");

    if (instr_exchange_c(inst, "IO:T%d", -100) == 0)
        printf(".");

    if (instr_exchange_c(inst, "IO:V%d", 0) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int do_agc(void)
{
    /* Section 3.6.13 */

    printf("Performing AGC");

    if (instr_exchange_c(inst, "IO:A%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "IO:G%d", 0) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int reset_all_parameters(void)
{
    /* Reset all parameters, except
        /AD,S07/
        /AD,S08/
        /DINT:CLK/
        /DINT:B8ZS/
        /DINT:FF/
        /DINT:TDMM/
        /DINT:TLBO/
        /DINT:TZCS/
        /DCO:IMS/ group
        /DCO:IMR/ group
        Calibration parameters
    */
    if (instr_exchange_c(inst, "IO:Z1") == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/
    
int monitor_impairment_generator(void)
{
    char buf[MAX_GPIB_CMD_LEN];
    char field[MAX_GPIB_CMD_LEN];
    const char *x;

    if (instr_exchange(inst, "IO:IR", buf) == 0)
    {
        if ((x = scan_for_field("IR", buf, field)))
            printf("Measurement IR %s\n", field);
    }

    if (instr_exchange(inst, "IO:LR", buf) == 0)
    {
        if ((x = scan_for_field("LR", buf, field)))
            printf("Measurement LR %s\n", field);
    }

    if (instr_exchange(inst, "IO:RR", buf) == 0)
    {
        if ((x = scan_for_field("RR", buf, field)))
            printf("Measurement RR %s\n", field);
    }

    if (instr_exchange(inst, "IO:TR", buf) == 0)
    {
        if ((x = scan_for_field("TR", buf, field)))
            printf("Measurement TR %s\n", field);
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

int monitor_signal(int which)
{
    char buf[MAX_GPIB_CMD_LEN];
    char field[MAX_GPIB_CMD_LEN];
    const char *x;

    /* Section 3.6.17 */

    if (instr_exchange(inst, "MM:R%d", buf, which) == 0)
    {
        display_explanation(mm_r_tags, which);
        /* Level, in 0.1dB steps */
        if ((x = scan_for_field("L", buf, field)))
            printf("Measured level %s\n", field);
        /* Frequency, in Hz */
        if ((x = scan_for_field("F", buf, field)))
            printf("Measured frequency %s\n", field);
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_auxiliary_echo(int which)
{
    /* Section 3.6.4 and 3.6.5 */

    printf("Configuring auxiliary echo %d", which);

    /* Echo delay to 20ms */
    if (instr_exchange_c(inst, "AXE%d:D%d", which, 160) == 0)
        printf(".");

    /* Echo level to 20dB */
    if (instr_exchange_c(inst, "AXE%d:L%d", which, 200) == 0)
        printf(".");

    /* Immediate talker */
    if (instr_exchange_c(inst, "AXE%d:T%d", which, 0) == 0)
        printf(".");

    /* Enable (1) or disable (0) auxiliary echo */
    if (instr_exchange_c(inst, "AXE%d:S%d", which, 0) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_audio_processor_module(void)
{
    if (!cellular_audio_processing_present)
        return -1;
        
    /* Section 3.6.3 */

    printf("Configuring audio processor module");

    if (instr_exchange_c(inst, "AP:LA%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "AP:LB%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "AP:LC%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "AP:LD%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "AP:TA%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "AP:TB%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "AP:X%d%d", 0, 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "AP:Y%d%d", 0, 0) == 0)
        printf(".");

    /* Enable (1) or disable (0) CAP function */
    if (instr_exchange_c(inst, "AP:SA%d", 0) == 0)
        printf(".");

    /* Enable (1) or disable (0) CAP function */
    if (instr_exchange_c(inst, "AP:SB%d", 0) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_dfe_impairments(void)
{
    int i;

    if (!universal_central_office_mode)
        return -1;

    printf("Configuring digital facility emulator (DFE) impairments");

    /* We must load an entire pad table in consecutive operations */
    for (i = 0;  i < 256;  i++)
    {
        if (instr_exchange_c(inst, "DIMP:D=%02X", i) == 0)
            printf(".");
    }
    if (instr_exchange_c(inst, "DIMP:DELAY=%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "DIMP:DSECT=%d", 1) == 0)
        printf(".");

    if (instr_exchange_c(inst, "DIMP:DTL=%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "DIMP:ECHOL=%d", 210) == 0)
        printf(".");

    if (instr_exchange_c(inst, "DIMP:ECHOS=%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "DIMP:ILG=%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "DIMP:IMD2L=%d", 520) == 0)
        printf(".");

    if (instr_exchange_c(inst, "DIMP:IMD2S=%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "DIMP:IMD3L=%d", 500) == 0)
        printf(".");

    if (instr_exchange_c(inst, "DIMP:IMD3S=%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "DIMP:OLG=%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "DIMP:PCMBIT=%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "DIMP:PCMRF=%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "DIMP:PCMTF=%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "DIMP:RBSD=%s", "0000") == 0)
        printf(".");

    if (instr_exchange_c(inst, "DIMP:RBSL1=%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "DIMP:RBSL2=%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "DIMP:RBSL3=%d", 0) == 0)
        printf(".");

    if (instr_exchange_c(inst, "DIMP:UDTL=%d", 1) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_tone_generator(void)
{
    /* Section 3.6.28 */

    printf("Configuring tone generator");

    if (instr_exchange_c(inst, "TN:F%d", 1004) == 0)
        printf(".");

    if (instr_exchange_c(inst, "TN:S%d", 0) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_system_admin(void)
{
    /* Section 3.6.1 */

    printf("Configuring system admin");

    /* Central office emulation - basic central office (00) or UCO (01) */
    if (instr_exchange_c(inst, "AD:S07=%02d", universal_central_office_mode) == 0)
        printf(".");

    if (universal_central_office_mode)
    {
        /* Set the station to be controlled while in UCO emulation mode
           to A (1), B (2) or both (3) */
        if (instr_exchange_c(inst, "AD:C%d", 3) == 0)
            printf(".");
    }

    /* Set impairment generator to be programmed to both */
    if (instr_exchange_c(inst, "AD:I%d", 3) == 0)
        printf(".");

    /* Set IEEE impulses (0) or bipolar impulses (1) */
    if (instr_exchange_c(inst, "AD:S01=%02d", 0) == 0)
        printf(".");

    /* Set the IMD algorithm to TAS1010 (0) or proprietary (1) */
    if (instr_exchange_c(inst, "AD:S02=%02d", 0) == 0)
        printf(".");

    /* Set 1010 compatible (1) or enhanced (0) gain/delay curves */
    if (instr_exchange_c(inst, "AD:S03=%02d", 0) == 0)
        printf(".");

    /* Set loop current step size ro 6mA (0) or 2mA (1) */
    if (instr_exchange_c(inst, "AD:S04=01") == 0)
        printf(".");

    /* Set ringing voltage step size to 5Vrms (0) or 1Vrms (1)  */
    if (instr_exchange_c(inst, "AD:S05=01") == 0)
        printf(".");

    /* Set station A interface type to analogue (0) or digital (1) */
    if (instr_exchange_c(inst, "AD:S08=%02d", 0) == 0)
        printf(".");

    /* Set resource configuration to type 1 */
    if (instr_exchange_c(inst, "AD:S09=%02d", 0) == 0)
        printf(".");

    /* Set the test channel configuration to EIA/CCITT (0),
       ETSI-1, NET 20 (1), ETSI-2, NET 20 (2), or analogue bypass (3) */
    if (instr_exchange_c(inst, "AD:T%d", 0) == 0)
        printf(".");

    printf("OK\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int main(int argc, char *argv[])
{
    if ((inst = instr_open("/dev/ttyS0", 9600, error_codes)) == NULL)
    {
        printf("Can't open the serial port\n");
        exit(2);
    }

    check_test_set_config();

    if (argc > 1  &&  strcmp(argv[1], "--check-config") == 0)
        exit(0);
    if (argc > 1  &&  strcmp(argv[1], "--test") == 0)
    {
        /* Set impairment generator I/O self calibrate and diagnose */
        printf("Running self-test. This takes a couple of minutes...\n");
        if (do_self_test() == 0)
        {
            printf("OK\n");
            exit(0);
        }
        printf("Failed\n");
        exit(2);
    }
    
    config_system_admin();

    config_epal();
    config_auxiliary_echo(1);
    config_auxiliary_echo(2);
    config_echo_control();
    config_frequency_shifter();
    config_non_linear_distortion();
    config_single_frequency_interference();
    config_gain_delay(0, 0, 0, 0);
    config_gain_hits();
    config_amplitude_jitter();
    config_phase_hits();
    config_phase_jitter();
    config_white_noise();
    config_impulse_noise();
    config_micro_interruptions(1);
    config_micro_interruptions(2);
    config_impairment_generator();
    config_line_control();

    config_pcm_adpcm();

    config_satellite_delays(1);
    config_satellite_delays(2);

    config_network_signaling();
    
    config_switching();

    config_dns(1);
    config_dns(2);
    
    config_digital_central_office();
    config_dint();
    config_dfe_impairments();
    
    config_tone_generator();

    if (universal_central_office_mode)
    {
        printf("Configuring odds and ends");
        /* Set dial number */
        if (instr_exchange_c(inst, "DIAL:DLN=%s", "1234") == 0)
            printf(".");
        if (instr_exchange_c(inst, "CPG1:DLN=%s", "1234") == 0)
            printf(".");
        if (instr_exchange_c(inst, "CPG2:DLN=%s", "1234") == 0)
            printf(".");
        if (instr_exchange_c(inst, "CPG3:DLN=%s", "1234") == 0)
            printf(".");
        if (instr_exchange_c(inst, "CPG4:DLN=%s", "1234") == 0)
            printf(".");

        if (instr_exchange_c(inst, "DIAL:DLT=%d", 2) == 0)
            printf(".");
        if (instr_exchange_c(inst, "CPG1:DLT=%d", 2) == 0)
            printf(".");
        if (instr_exchange_c(inst, "CPG2:DLT=%d", 2) == 0)
            printf(".");
        if (instr_exchange_c(inst, "CPG3:DLT=%d", 2) == 0)
            printf(".");
        if (instr_exchange_c(inst, "CPG4:DLT=%d", 2) == 0)
            printf(".");

        if (instr_exchange_c(inst, "CONN:DCS=%d", 0) == 0)
            printf(".");
        if (instr_exchange_c(inst, "CONN:PTS=%d", 0) == 0)
            printf(".");

        /* Set analog line mode */
        if (instr_exchange_c(inst, "EXCH:MODE=%d", 1) == 0)
            printf(".");

        /* Set ??? */
        if (instr_exchange_c(inst, "EXCH:BAL=%d", 1) == 0)
            printf(".");

        /* Set ??? */
        if (instr_exchange_c(inst, "FILE:CSEQ=%s", "usa1") == 0)
            printf(".");

        /* Set line ground reference command */
        if (instr_exchange_c(inst, "LSIG:GND=%d", 0) == 0)
            printf(".");
            
        printf("\n");
    }
    
    printf("Configuration complete\n");

    for (;;)
    {
        monitor_impairment_generator();
        monitor_signal(2);
        monitor_signal(5);
        monitor_network_signaling_status();
        monitor_switching();
        monitor_digital_link_status();
        monitor_abcd();
    }

    return 0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
