/*
 * tas240.c
 *
 * Basic control of a TAS240 subscriber loop emulator.
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2007 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: tas240.c,v 1.1.1.1 2007/06/09 06:31:18 steveu Exp $
 */

#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <ctype.h>
#include <string.h>
#include <malloc.h>
#include <fcntl.h>
#include <termios.h>
#include <inttypes.h>
#include <sys/types.h>
#include <sys/unistd.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/ioctl.h>

#include "instrument.h"

#define FALSE 0
#define TRUE (!FALSE)

#define MAX_GPIB_CMD_LEN        128

enum
{
    TAS240_LOOP_NULL = 0,
    TAS240_LOOP_EIA1,
    TAS240_LOOP_EIA2,
    TAS240_LOOP_EIA3,
    TAS240_LOOP_EIA4,
    TAS240_LOOP_EIA5,
    TAS240_LOOP_EIA6,
    TAS240_LOOP_EIA7,
    TAS240_LOOP_ETSI1,
    TAS240_LOOP_ETSI2,
    TAS240_LOOP_PAD,
    TAS240_LOOP_CUSTOM
};

enum
{
    TAS240_BALANCE_AB2 = 0,
    TAS240_BALANCE_RC1,
    TAS240_BALANCE_RC2,
    TAS240_BALANCE_EXT
};

enum
{
    TAS240_TERMINATION_600_OHMS = 0,
    TAS240_TERMINATION_EXTERNAL
};

enum
{
    TAS240_APPEND_NONE = 0,
    TAS240_APPEND_EIA6,
    TAS240_APPEND_EIA7
};


enum
{
    TAS240_METER_PUSE_PORT_A1EO = 0,
    TAS240_METER_PUSE_PORT_A1NI,
    TAS240_METER_PUSE_PORT_B1EO,
    TAS240_METER_PUSE_PORT_B1NI,
    TAS240_METER_PUSE_PORT_REAR
};

enum
{
    TAS240_MEASURE_A1NI_EO = 0,
    TAS240_MEASURE_A1EO_NI,
    TAS240_MEASURE_A2NI_EO,
    TAS240_MEASURE_A2EO_NI,
    TAS240_MEASURE_B1NI_EO,
    TAS240_MEASURE_B1EO_NI,
    TAS240_MEASURE_B2NI_EO,
    TAS240_MEASURE_B2EO_NI
};

/* TAS 240 error codes */
tag_t error_codes[] =
{
    {1, "Parameter value error"},
    {2, "Command syntax error"},
    {5, "????"},
    {-1, ""}
};

int debug = FALSE;

static instr_t *inst;

int check_test_set_config(void)
{
    char buf[MAX_GPIB_CMD_LEN];
    char field[MAX_GPIB_CMD_LEN];
    const char *x;

    /* Check some stuff - modules present, version numbers, etc.
       at startup */

    /* Inquire about the software version and power-up diagnostics */
    if (instr_exchange(inst, "CNFG:MODL", buf) == 0)
    {
        if ((x = scan_for_field("MODL=", buf, field)))
            printf("Test set model %s\n", field);
    }
    if (instr_exchange(inst, "CNFG:VERS", buf) == 0)
    {
        if ((x = scan_for_field("VERS=", buf, field)))
            printf("Test set version %s\n", field);
    }
    if (instr_exchange(inst, "CNFG:OPTS", buf) == 0)
    {
        if ((x = scan_for_field("OPTS=", buf, field)))
            printf("Test set options %s\n", field);
    }

    if (instr_exchange(inst, "CNFG:VCON", buf) == 0)
    {
        if ((x = scan_for_field("VCON=", buf, field)))
            printf("Test set VCON %s\n", field);
    }
    if (instr_exchange(inst, "CNFG:LOAD", buf) == 0)
    {
        if ((x = scan_for_field("LOAD=", buf, field)))
            printf("Test set LOAD %s\n", field);
    }
    if (instr_exchange(inst, "CNFG:STAT", buf) == 0)
    {
        if ((x = scan_for_field("STAT=", buf, field)))
            printf("Test set status %s\n", field);
    }
    printf("\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int reset_test_set_config(void)
{
    if (instr_exchange_c(inst, "CNFG:RESET") == 0)
        printf(".");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int front_panel_control(int lock)
{
    if (lock)
    {
        /* Panel controls locked */
        if (instr_exchange_c(inst, "CNFG:LOCK") == 0)
            printf(".");
    }
    else
    {
        /* Panel controls enabled (local control) */
        if (instr_exchange_c(inst, "CNFG:LOC") == 0)
            printf(".");
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_loop_a(int loop, int bal, int term, int awg_26_len, int awg_24_len, int append, int bt)
{
    static const char *loops[] =
    {
        "NULL",
        "EIA1",
        "EIA2",
        "EIA3",
        "EIA4",
        "EIA5",
        "EIA6",
        "EIA7",
        "ETSI1",
        "ETSI2",
        "PAD",
        "CUSTA"
    };
    static const char *balances[] =
    {
        "A2",
        "RC1",
        "RC2",
        "EXT"
    };
    static const char *terminations[] =
    {
        "600",
        "EXT"
    };
    static const char *appends[] =
    {
        "NONE",
        "EIA6",
        "EIA7"
    };
    
    printf("Configuring loop A");
    if (instr_exchange_c(inst, "LOOPA:LOOP=%s", loops[loop]) == 0)
        printf(".");
    if (instr_exchange_c(inst, "LOOPA:BAL=%s", balances[bal]) == 0)
        printf(".");
    if (instr_exchange_c(inst, "LOOPA:TERM=%s", terminations[term]) == 0)
        printf(".");
    if (loop == TAS240_LOOP_CUSTOM)
    {
        /* 0 to 15km of line */
        if (instr_exchange_c(inst, "LOOPA:26AWG=%d", awg_26_len) == 0)
            printf(".");
        /* 0 to 15km of line */
        if (instr_exchange_c(inst, "LOOPA:24AWG=%d", awg_24_len) == 0)
            printf(".");
        if (instr_exchange_c(inst, "LOOPA:APPEND=%s", appends[append]) == 0)
            printf(".");
        if (instr_exchange_c(inst, "LOOPA:BT=%s", (bt)  ?  "ON"  :  "OFF") == 0)
            printf(".");
    }
    printf("\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_loop_b(int loop, int bal, int term, int awg_26_len, int awg_24_len, int append, int bt)
{
    static const char *loops[] =
    {
        "NULL",
        "EIA1",
        "EIA2",
        "EIA3",
        "EIA4",
        "EIA5",
        "EIA6",
        "EIA7",
        "ETSI1",
        "ETSI2",
        "PAD",
        "CUSTB"
    };
    static const char *balances[] =
    {
        "B2",
        "RC1",
        "RC2",
        "EXT"
    };
    static const char *terminations[] =
    {
        "600",
        "EXT"
    };
    static const char *appends[] =
    {
        "NONE",
        "EIA6",
        "EIA7"
    };
    
    printf("Configuring loop B");
    if (instr_exchange_c(inst, "LOOPB:LOOP=%s", loops[loop]) == 0)
        printf(".");
    if (instr_exchange_c(inst, "LOOPB:BAL=%s", balances[bal]) == 0)
        printf(".");
    if (instr_exchange_c(inst, "LOOPB:TERM=%s", terminations[term]) == 0)
        printf(".");
    if (loop == TAS240_LOOP_CUSTOM)
    {
        /* 0 to 15km of line */
        if (instr_exchange_c(inst, "LOOPB:26AWG=%d", awg_26_len) == 0)
            printf(".");
        /* 0 to 15km of line */
        if (instr_exchange_c(inst, "LOOPB:24AWG=%d", awg_24_len) == 0)
            printf(".");
        if (instr_exchange_c(inst, "LOOPB:APPEND=%s", appends[append]) == 0)
            printf(".");
        if (instr_exchange_c(inst, "LOOPB:BT=%s", (bt)  ?  "ON"  :  "OFF") == 0)
            printf(".");
    }
    printf("\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int config_metering_puses()
{
    static const char *ports[] =
    {
        "A1EO",
        "A1NI",
        "B1EO",
        "B1NI",
        "REAR"
    };
    
    if (instr_exchange_c(inst, "METER:FREQ=%d", 42) == 0)
        printf(".");
    if (instr_exchange_c(inst, "METER:LEVEL=%d", 42) == 0)
        printf(".");
    if (instr_exchange_c(inst, "METER:ON=%d", 42) == 0)
        printf(".");
    if (instr_exchange_c(inst, "METER:OFF=%d", 42) == 0)
        printf(".");
    if (instr_exchange_c(inst, "METER:SIGNAL=%d", 42) == 0)
        printf(".");
    if (instr_exchange_c(inst, "METER:PORT=%s", ports[2]) == 0)
        printf(".");
    if (instr_exchange_c(inst, "METER:IMPED=%d", 42) == 0)
        printf(".");
    if (instr_exchange_c(inst, "METER:CAL=%d", 42) == 0)
        printf(".");
    return 0;
}
/*- End of function --------------------------------------------------------*/

int measure_loss(int freq, int pair)
{
    char buf[MAX_GPIB_CMD_LEN];
    char field[MAX_GPIB_CMD_LEN];
    const char *x;
    static const char *pairs[] =
    {
        "A1NI-EO",
        "A1EO-NI",
        "A2NI-EO",
        "A2EO-NI",
        "B1NI-EO",
        "B1EO-NI",
        "B2NI-EO",
        "B2EO-NI"
    };

    /* The frequency at which loss is measured (200Hz to 3600Hz) */
    if (instr_exchange_c(inst, "MEAS:FREQ=%d", freq) == 0)
        printf(".");
    if (instr_exchange_c(inst, "MEAS:PAIR=%s", pairs[pair]) == 0)
        printf(".");
    /* Now perform the measurement */
    if (instr_exchange(inst, "MEAS:RP", buf) == 0)
    {
        if ((x = scan_for_field("RP=", buf, field)))
            printf("Loss at %dHz between pair %s is %sdB\n", freq, pairs[pair], field);
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/
    
int main(int argc, char *argv[])
{
    if ((inst = instr_open("/dev/ttyS0", 4800, error_codes)) == NULL)
    {
        printf("Can't open the serial port\n");
        exit(2);
    }

    check_test_set_config();

    config_loop_a(11, 1, 0, 4, 5, 1, 0);
    config_loop_b(11, 2, 1, 6, 7, 2, 1);

    measure_loss(1200, 5);
    return 0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
