/*
 * libunicall - a library for universal call handling on both analogue and
 *              digital telephone circuits.
 *
 * unicall.h
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2002,2004 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 2.1,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: unicall.h,v 1.37 2008/07/06 11:01:26 steveu Exp $
 */

/*! \file */

#if !defined(_UNICALL_H_)
#define _UNICALL_H_

/* The universal call library is based on the way ISDN and SS7 work. Since
   these are kind of a superset of other signaling schemes, things work out
   OK for most other signaling. Of course, not all signaling schemes support
   identical features. */

/* The protocol library may be switchable between representing the CPE or the CO end.
   The protocol library may support only CPE or CO mode.
   The protocol may be inherently peer-to-peer. */
enum
{
    UC_MODE_CPE                                 = 0x01,
    UC_MODE_CO                                  = 0x02,
    UC_MODE_PEER                                = 0x03
};

/* Call clearing causes, based on the ISDN names and values. */
enum
{
    /* Normal class */
    UC_CAUSE_UNASSIGNED_NUMBER                  = 0x01,
    UC_CAUSE_NO_ROUTE_TO_TRANSIT_NETWORK        = 0x02,
    UC_CAUSE_NO_ROUTE_TO_DESTINATION            = 0x03,
    UC_CAUSE_CHANNEL_UNACCEPTABLE               = 0x06,
    UC_CAUSE_CALL_AWARDED_DELIVERED             = 0x07,
    UC_CAUSE_NORMAL_CLEARING                    = 0x10,
    UC_CAUSE_USER_BUSY                          = 0x11,
    UC_CAUSE_NO_USER_RESPONDING                 = 0x12,
    UC_CAUSE_NO_ANSWER_FROM_USER                = 0x13,     /* User alerted */
    UC_CAUSE_CALL_REJECTED                      = 0x15,
    UC_CAUSE_NUMBER_CHANGED                     = 0x16,
    UC_CAUSE_NON_SELECT_USER_CLEARED            = 0x1A,
    UC_CAUSE_DEST_OUT_OF_ORDER                  = 0x1B,
    UC_CAUSE_INVALID_NUMBER_FORMAT              = 0x1C,     /* Address incomplete */
    UC_CAUSE_FACILITY_REJECTED                  = 0x1D,
    UC_CAUSE_RESP_TO_STAT_ENQ                   = 0x1E,
    UC_CAUSE_UNSPECIFIED_CAUSE                  = 0x1F,     /* Normal, unspecified cause */

    /* Resource unavailable class */
    UC_CAUSE_NO_CIRCUIT_AVAILABLE               = 0x22,
    UC_CAUSE_NETWORK_OUT_OF_ORDER               = 0x26,
    UC_CAUSE_TEMPORARY_FAILURE                  = 0x29,
    UC_CAUSE_NETWORK_CONGESTION                 = 0x2A,
    UC_CAUSE_ACCESS_INFO_DISCARDED              = 0x2B,
    UC_CAUSE_REQ_CHANNEL_NOT_AVAILABLE          = 0x2C,
    UC_CAUSE_PRE_EMPTED                         = 0x2D,

    /* Resource unavailable class */
    UC_CAUSE_QOS_NOT_AVAILABLE                  = 0x31,
    UC_CAUSE_FACILITY_NOT_SUBSCRIBED            = 0x32,
    UC_CAUSE_OUTGOING_CALL_BARRED               = 0x34,
    UC_CAUSE_INCOMING_CALL_BARRED               = 0x36,
    UC_CAUSE_BEARER_CAP_NOT_AUTHORIZED          = 0x39,
    UC_CAUSE_BEARER_CAP_NOT_AVAILABLE           = 0x3A,
    UC_CAUSE_SERVICE_NOT_AVAILABLE              = 0x3F,

    /* Service or option not implemented class */
    UC_CAUSE_BEARER_CAP_NOT_IMPLEMENTED         = 0x41,
    UC_CAUSE_CHAN_TYPE_NOT_IMPLEMENTED          = 0x42,
    UC_CAUSE_FACILITY_NOT_IMPLEMENTED           = 0x45,
    UC_CAUSE_RESTRICTED_DIGITAL_BEARER          = 0x46,
    UC_CAUSE_SERVICE_NOT_IMPLEMENTED            = 0x4F,

    /* Invalid message (e.g. bad parameter) class */
    UC_CAUSE_INVALID_CALL_REF                   = 0x51,
    UC_CAUSE_CHAN_DOES_NOT_EXIST                = 0x52,
    UC_CAUSE_SUSPENDED_CALL_BAD_CALL_REF        = 0x53,
    UC_CAUSE_CALL_REF_IN_USE                    = 0x54,
    UC_CAUSE_NO_CALL_SUSPENDED                  = 0x55,
    UC_CAUSE_CALL_REF_CLEARED                   = 0x56,
    UC_CAUSE_INCOMPATIBLE_DEST                  = 0x58,
    UC_CAUSE_INVALID_TRANSIT_NETWORK            = 0x5B,
    UC_CAUSE_INVALID_MSG_UNSPECIFIED            = 0x5F,

    /* Protocol error (e.g. unknown message) class */
    UC_CAUSE_MANDATORY_IE_MISSING               = 0x60,
    UC_CAUSE_NONEXISTANT_MSG                    = 0x61,
    UC_CAUSE_WRONG_MESSAGE                      = 0x62,
    UC_CAUSE_BAD_INFO_ELEM                      = 0x63,
    UC_CAUSE_INVALID_ELEM_CONTENTS              = 0x64,
    UC_CAUSE_WRONG_MSG_FOR_STATE                = 0x65,
    UC_CAUSE_TIMER_EXPIRY                       = 0x66,
    UC_CAUSE_MANDATORY_IE_LEN_ERROR             = 0x67,
    UC_CAUSE_PROTOCOL_ERROR                     = 0x6F,

    /* Interworking class */
    UC_CAUSE_INTERWORKING_UNSPECIFIED           = 0x7F
};

/* Call states */

enum
{
    /*! 
        UC_STATE_NULL is the initial state for a circuit after it has just been
        opened. The circuit may dwell in this state while things are initialised.
    */
    UC_STATE_NULL                               = 0x00000000,
    
    /*! 
        UC_STATE_IDLE means the line is free for a new call to be initiated
    */
    UC_STATE_IDLE                               = 0x00000001,
    
    /*! 
        UC_STATE_DETECTED means an incoming call has seized the circuit, but the
        initial exchange (DNIS, ANI, etc.) is in progress, so the call has not been
        offered yet.
    */
    UC_STATE_DETECTED                           = 0x00000002,
    
    /*! 
        UC_STATE_OFFERED means an incoming call has completed its initial
        exchange of information, and the application has not yet accepted,
        rejected, or otherwise determined how to handle the call.
    */
    UC_STATE_OFFERED                            = 0x00000004,
    
    /*! 
        UC_STATE_REQUESTMOREINFO means an incoming call has been offered, but
        additional information is now being gathered from the far end (e.g. the
        originating number).
    */
    UC_STATE_REQUESTMOREINFO                    = 0x00000008,
    
    /*! 
        UC_STATE_CALLROUTING means an incoming call is being routed towards its 
        final destination. This does not mean the call has been accepted. We
        may have routed on the basis of, say, a number block, without knowing
        if the specific number is valid, or the line is free.
    */
    UC_STATE_CALLROUTING                        = 0x00000010,
    
    /*! 
        UC_STATE_ACCEPTED means an incoming call has been accepted by the
        application, but has yet to be answered. Examples would be that the
        dialed number has been accepted as valid, but the related local
        extension the call is for is currently ringing.
    */
    UC_STATE_ACCEPTED                           = 0x00000020,
    
    /*! 
        UC_STATE_DIALING means the circuit has been seized for an outgoing call,
        and the initial exchange (DNIS, ANI, etc.) is in progress
    */
    UC_STATE_DIALING                            = 0x00000040,
    
    /*! 
        UC_STATE_SENDMOREINFO means an outgoing call has completed the initial
        exchange of information, and now more information has been requested by
        the far end (e.g. originating number).
    */
    UC_STATE_SENDMOREINFO                       = 0x00000080,
    
    /*! 
        UC_STATE_PROCEEDING means the initial exchange of information for an
        outgoing call is complete, and the far end is dealing with the call
        (perhaps routing it towards its destination).
    */
    UC_STATE_PROCEEDING                         = 0x00000100,
    
    /*! 
        UC_STATE_ALERTING means an outgoing call has reached the stage where
        the far party is ringing.
    */
    UC_STATE_ALERTING                           = 0x00000200,
    
    /*! 
        UC_STATE_CONNECTED means the two parties are connected.
    */
    UC_STATE_CONNECTED                          = 0x00000400,
    
    /*! 
        UC_STATE_FAR_DISCONNECTED means the far end (only) has disconnected.
    */
    UC_STATE_FAR_DISCONNECTED                   = 0x00000800,
    
    /*! 
        UC_STATE_CALL_DISCONNECTED means both the local and far ends have disconnected,
        but some clean-up (e.g. release guard time) is in progress.
    */
    UC_STATE_CALL_DISCONNECTED                  = 0x00001000,
    
    /*! 
        UC_STATE_ON_HOLD means the call has been put on hold by the resource to
        which it is currently routed.
    */
    UC_STATE_ON_HOLD                            = 0x00002000,
    
    /*! 
        UC_STATE_ACCEPTED_QUEUED means the call is queued waiting for a resource
        to become free (e.g. a free operator) for initial answering. The call has
        not actually been answered at this stage.
    */
    UC_STATE_ACCEPTED_QUEUED                    = 0x00004000,
    
    /*! 
        UC_STATE_ON_HOLD_QUEUED means the call is queued waiting for a resource
        to become free (e.g. a free operator) for call transfer. The call has been
        answered at this stage.
    */
    UC_STATE_ON_HOLD_QUEUED                     = 0x00008000,
    
    /*! 
        UC_STATE_OOS means the circuit is out of service by request (as opposed to
        by equipment failure).
    */
    UC_STATE_OOS                                = 0x10000000,
    
    /*! 
        UC_STATE_FAILED means a physical failure has occured on the data or signaling
        path - e.g. a dead E1
    */
    UC_STATE_FAILED                             = 0x20000000,
    
    /*! 
        UC_STATE_BLOCKED means the far end has blocked the circuit (perhaps for
        maintenance purposes).
    */
    UC_STATE_BLOCKED                            = 0x40000000
};

/*! Call events */
enum
{
/*!
    UC_EVENT_DEVICEFAIL indicates there has been an abnormal device condition.
*/
    UC_EVENT_DEVICEFAIL = 1,

/*! 
    UC_EVENT_PROTOCOLFAIL indicates there has been an error in the
    signaling protocol (e.g. an out of sequence message.)
*/
    UC_EVENT_PROTOCOLFAIL,

/*! 
    UC_EVENT_SIGCHANSTATUS indicates there has been a change is the status of
    the associated signaling channel. This event only occurs for common
    channel signaling protocols (e.g. ISDN).
*/
    UC_EVENT_SIGCHANSTATUS,

/*! 
    UC_EVENT_DETECTED is an informational event only. It does not affect the
    call state. It is provided as an early warning that a call is coming in,
    to reduce the frequency with which outgoing call attempts collide with
    incoming calls. At this point a CRN is assigned to the call.
*/
    UC_EVENT_DETECTED = 0x10,

/*! 
    UC_EVENT_OFFERED indicates a new incoming call's arrival, complete with all its
    information (e.g. the originating and destination numbers).
*/
    UC_EVENT_OFFERED,

/*! 
    UC_EVENT_REQUESTMOREINFO indicates the far end is requesting additional call setup
    information (e.g. the originating number, or additional dialed digits). The application
    can respond to this event, providing more information (e.g. dialed digits) than was available
    when the call was initiated. This provides support for overlapped dialing.
*/
    UC_EVENT_REQUESTMOREINFO,

/*! 
    UC_EVENT_ACCEPTED indicates completion of call acceptance. It is the
    normal response to a UC_OP_ACCEPTCALL. Note that is af a call clears
    during call acceptance, this event might never occur.
*/
    UC_EVENT_ACCEPTED,

    UC_EVENT_CALLINFO,

    UC_EVENT_FACILITY,

/*! 
    UC_EVENT_DIALEDNUMBER indicates more digits have been received from a channel
    supporting overlapped dial.(e.g. from an FXO device, where we need to check
    digit by digit whether a complete number has been dialed). The application can
    respond to this event, requesting more dialed digits or accepting what has been
    dialed so far.
*/
    UC_EVENT_DIALEDNUMBER,

/*!
    UC_EVENT_DIALTONE indicates that dialtone has been detected.
*/
    UC_EVENT_DIALTONE = 0x20,

/*!
    UC_EVENT_DIALING indicates that an outgoing call has begun, and is in the
    initial phase of transfering the call setup information.
*/
    UC_EVENT_DIALING,

/*!
    UC_EVENT_SENDMOREINFO indicates a request for additional call setup information
    (e.g. more dialed digits, or the originating number).
*/
    UC_EVENT_SENDMOREINFO,

/*! 
    UC_EVENT_PROCEEDING indicates completion of the transfer of call setup
    information (DNIS, ANI, etc.). Further action, such as call routing, may
    be needed before the call is accepted.
*/
    UC_EVENT_PROCEEDING,

/*! 
    UC_EVENT_ALERTING indicates the call has reached an available line,
    and is alerting (ringing) awaiting answer.
*/
    UC_EVENT_ALERTING,

/*! 
    UC_EVENT_CONNECTED indicates an outgoing call has reached the conversation
    phase.
*/
    UC_EVENT_CONNECTED = 0x30,

/*! 
    UC_EVENT_ANSWERED indicates an incoming call has reached the conversation
    phase.
*/
    UC_EVENT_ANSWERED,

/*! 
    UC_EVENT_FARDISCONNECTED indicates the far end has cleared the call first.
    UC_OP_DROPCALL should be used to clear the call from our end.
*/
    UC_EVENT_FARDISCONNECTED,

/*! 
    UC_EVENT_DROPCALL follows a call to UC_OP_DROPCALL. It indicates we cleared
    forward first, and the far end has now responded by clearing backwards.
    UC_OP_RELEASECALL should be used to release resources used by the call.
*/
    UC_EVENT_DROPCALL,

/*!
    UC_EVENT_RELEASECALL indicates the final release of resources has completed.
    The CRN is no longer valid. The call has truly ended. The call structure has
    been freed by the time this event is received. A pointer to the call structure
    is passed in the event parameters. This is just for reference (e.g. it can be
    matched against the previous call pointer). Do not attempt the access the
    contents of the call structure at this time.
*/
    UC_EVENT_RELEASECALL,

/*!
    UC_EVENT_FARBLOCKED indicates the far end has blocked the indicated channel.
*/
    UC_EVENT_FARBLOCKED = 0x40,

/*!
    UC_EVENT_FARUNBLOCKED indicates the far end has unblocked the indicated channel.
*/
    UC_EVENT_FARUNBLOCKED,

/*!
    UC_EVENT_LOCALBLOCKED indicates the local end has completed its blocking operation
    in response to a UC_OP_BLOCK call.
*/
    UC_EVENT_LOCALBLOCKED,

/*!
    UC_EVENT_LOCALUNBLOCKED indicates the local end has completed its unblocking operation
    in response to a UC_OP_BLOCK call.
*/
    UC_EVENT_LOCALUNBLOCKED,

/*!
    UC_EVENT_ALARM indicates that an alarm has be raised or has cleared.
*/
    UC_EVENT_ALARM,

    UC_EVENT_RESETLINEDEV,

    UC_EVENT_L2FRAME = 0x50,
    UC_EVENT_L2BUFFERFULL,
    UC_EVENT_L2NOBUFFER,
    UC_EVENT_USRINFO
};

enum
{
    UC_ECHO_CANCEL_OFF = 0,
    UC_ECHO_CANCEL_ON,
    UC_ECHO_CANCEL_NOTRAINING,
    UC_ECHO_CANCEL_TRAINING
};

enum
{
    UC_SWITCHING_FREE = 0,
    UC_SWITCHING_UNMUTE,
    UC_SWITCHING_MUTE,
    UC_SWITCHING_CONNECT
};

#define UC_MAXPHONENUMLEN           32
#define UC_MAXPHONENAMELEN          256

/*! Defines for the "type" field above */
enum
{
    /*! \brief Request more info */
    UC_CALLACK_SERVICE_INFO = 1,
    /*! \brief Proceed with the call */
    UC_CALLACK_SERVICE_PROC
};

/*! Defines for the "info_type" field above
    for UC_CALLACK_SERVICE_INFO */
enum
{
    /*! \brief Proceed with the current B channel. */
    UC_CALLACK_SAME_B_CHAN = 0,
    /*! \brief Proceed with a new B channel. */
    UC_CALLACK_NEW_B_CHAN,
    /*! \brief Setup ACK */  
    UC_CALLACK_SETUP_ACK
};

/*! Parameters for UC_CALLACK_SERVICE_PROC */
enum
{
    /*! \brief Request more DNIS digits */
    UC_CALLACK_DESTINATION_NUMBER = 0,
    /*! \brief Proceed with ANI digits */
    UC_CALLACK_ORIGINATING_NUMBER
};

/*! Type parameters for UC_OP_REQUESTMOREINFO */
enum
{
    /*! \brief More destination number digits */
    UC_REQUESTMOREINFO_DESTINATION_NUMBER = 0,
    /*! \brief More originating number digits */
    UC_REQUESTMOREINFO_ORIGINATING_NUMBER
};

/*! \brief Call reference number 
    This is a value between 1 and 32767, used to idenrify a particular call
    within a UniCall context. This range is chosen to match the range of Q.931
    calls. */
typedef int uc_crn_t;

/*!
    Call descriptor. This contains all the setup information for a call. In many
    cases most of the entries can be left at their default values.
 */
typedef struct uc_callparms_s
{
    /*! \brief */
    uint8_t bear_cap_transfer_cap;
    /*! \brief */
    uint8_t bear_cap_transfer_mode;
    /*! \brief */
    uint8_t bear_cap_transfer_rate;
    /*! \brief */
    uint8_t userinfo_layer1_protocol;
    /*! \brief */
    uint8_t user_rate;
    /*! \brief The type of number of the destination address */
    uint8_t destination_ton;
    /*! \brief The numbering plan indicator of the destination address */
    uint8_t destination_npi;
    /*! \brief The destination address */
    char destination_number[UC_MAXPHONENUMLEN + 1];
    /*! \brief The type of number of the destination sub-address */
    uint8_t destination_sub_addr_ton;
    /*! \brief The numbering plan indicator of the destination sub-address */
    uint8_t destination_sub_addr_npi;
    /*! \brief The destination sub-address */
    char destination_sub_addr_number[UC_MAXPHONENUMLEN + 1];
    /*! \brief The originating address presentation */
    uint8_t originating_pres;
    /*! \brief The type of number of the originating address */
    uint8_t originating_ton;
    /*! \brief The numbering plan indicator of the originating address */
    uint8_t originating_npi;
    /*! \brief The originating address */
    char originating_number[UC_MAXPHONENUMLEN + 1];
    /*! \brief The originating party's registered name */
    char originating_name[UC_MAXPHONENAMELEN + 1];
    /*! \brief The type of number of the originating sub-address */
    uint8_t originating_sub_addr_ton;
    /*! \brief The numbering plan indicator of the originating sub-address */
    uint8_t originating_sub_addr_npi;
    /*! \brief The originating sub-address */
    char originating_sub_addr_number[UC_MAXPHONENUMLEN + 1];
    /*! \brief The reason for call redirection */
    uint8_t redirecting_cause;
    /*! \brief The redirecting address presentation */
    uint8_t redirecting_pres;
    /*! \brief The type of number of the redirecting address */
    uint8_t redirecting_ton;
    /*! \brief The number plan indicator of the redirecting address */
    uint8_t redirecting_npi;
    /*! \brief The redirecting address */
    char redirecting_number[UC_MAXPHONENUMLEN + 1];
    /*! \brief The type of number of the redirecting sub-address */
    uint8_t redirecting_subaddr_ton;
    /*! \brief The numbering plan indicator of the redirecting sub-address */
    uint8_t redirecting_subaddr_npi;
    /*! \brief The redirecting sub-address */
    char redirecting_subaddr[UC_MAXPHONENUMLEN + 1];
    /*! \brief The type of number of the original called number */
    uint8_t original_called_number_ton;
    /*! \brief The numbering plan indicator of the original called number */
    uint8_t original_called_number_npi;
    /*! \brief The original called number */
    char original_called_number[UC_MAXPHONENUMLEN + 1];
     /*! \brief The calling party category, as used in MFC/R2 and SS7 */
    uint8_t calling_party_category;
    /*! \brief Call type indicator (free, charged, etc.) */
    uint8_t call_type;
    /*! \brief TRUE if there will be no more call information available later. FALSE
               if the UC_EVENT_REQUESTMOREINFO event should be used to request more later
               on. */
    uint8_t call_information_complete;
} uc_callparms_t;

/*!
    Call acknowledgement
*/
struct uc_callack_s
{
    /*! What the structure really contains */
    int type;
    int info_type;
    /*! A digit string length, or a new B channel to be used */    
    int info_value;
};

/*!
    Request more info
*/
struct uc_requestmoreinfo_s
{
    /*! What the structure really contains */
    int type;
    int minimum;
};

/*!
    Make call
*/
struct uc_makecall_s
{
    uc_callparms_t *callparms;
    uc_crn_t crn;
};

/*!
    User to user
*/
struct uc_usertouser_s
{
    int len;
    uint8_t *message;
};

/* Call types */
#define UC_CALL_TYPE_UNSPECIFIED        0
#define UC_CALL_TYPE_NO_CHARGE_CALL     1
#define UC_CALL_TYPE_CHARGED_CALL       2
#define UC_CALL_TYPE_SIT                3

/*!
    Event structure used for most UC_EVENT_xxxx reports. Other structures are used for
    reports which carry extended information.
*/
typedef struct uc_event_generic_s
{
    /* Events without special additional information fall in this category */
    /*! \brief The event type. */
    int e;
    /*! \brief The channel associated with the event. -1 if no channel is associated. */
    int channel;
    /*! \brief The call reference number. */
    int crn;
    /*! \brief A simple numeric parameter. The meaning of this varies with the event type. */
    int data;
    /*! \brief An opaque pointer, used when refering to the call related to the event. NULL if
        is no related call. */
    void *call;
} uc_event_generic_t;

/*!
    Event structure used for UC_EVENT_OFFERED and UC_EVENT_MOREDIGITS.
*/
typedef struct uc_event_offered_s
{
    /*! \brief The event type. */
    int e;
    /*! \brief The channel associated with the event. -1 if no channel is associated. */
    int channel;
    /*! \brief The call reference number. */
    int crn;
    /*! \brief TRUE if we flexible with our channel selection */
    int flexible;
    /*! An opaque pointer, used when refering the the call related to this event. */
    void *call;
    /*! The parameters describing the offered call. */
    uc_callparms_t parms;
    int useruserprotocoldisc;
    char useruserinfo[256];
} uc_event_offered_t;

/*!
    Event structure used for UC_EVENT_FARDISCONNECTED.
*/
typedef struct uc_event_fardisconnected_s
{
    /*! \brief The event type. */
    int e;
    /*! \brief The channel associated with the event. -1 if no channel is associated. */
    int channel;
    /*! \brief The call reference number. */
    int crn;
    /*! \brief The reason for the disconnection. These codes are based on the ITU Q.931
               codes. */
    int cause;
    /*! An opaque pointer, used when refering the the call related to this event. */
    void *call;
} uc_event_fardisconnected_t;

/*!
    Event structure used for UC_EVENT_ALARM.
*/
typedef struct uc_event_alarm_s
{
    /*! \brief The event type. */
    int e;
    /*! \brief The channel associated with the event. -1 if no channel is associated. */
    int channel;
    /*! \brief The types of alarm raised. */
    int raised;
    /*! \brief The types of alarm cleared. */
    int cleared;
} uc_event_alarm_t;

typedef struct uc_event_error_s
{
    /*! \brief The event type. */
    int e;
    /*! \brief The channel associated with the event. -1 if no channel is associated. */
    int channel;
    char err[256];
} uc_event_error_t;

typedef struct uc_event_sigchanstatus_s
{
    /*! \brief The event type. */
    int e;
    /*! \brief The channel associated with the event. -1 if no channel is associated. */
    int channel;
    int ok;
} uc_event_sigchanstatus_t;

typedef struct uc_event_facname_s
{
    /*! \brief The event type. */
    int e;
    /*! \brief The channel associated with the event. -1 if no channel is associated. */
    int channel;
    /*! \brief The call reference number. */
    int crn;
    /*! \brief An opaque pointer, used when refering the the call related to this event. */
    void *call;
    char originating_name[UC_MAXPHONENAMELEN];
    char originating_number[UC_MAXPHONENUMLEN];
} uc_event_facname_t;

/*!
    Event descriptor.
 */
typedef union
{
    int e;
    uc_event_generic_t          gen;
    uc_event_offered_t          offered;
    uc_event_fardisconnected_t  fardisconnected;
    uc_event_alarm_t            alarm;
    uc_event_error_t            error;
    uc_event_sigchanstatus_t    sigchanstatus;
    uc_event_facname_t          facname;
} uc_event_t;

/*!
    A UniCall control structure.
 */
typedef struct uc_s uc_t;

/*!
    UniCall call descriptor.
 */
typedef struct uc_call_s uc_call_t;

typedef struct uc_callack_s uc_callack_t;

typedef struct uc_requestmoreinfo_s uc_requestmoreinfo_t;

typedef struct uc_makecall_s uc_makecall_t;

typedef struct uc_usertouser_s uc_usertouser_t;

typedef void (uc_signaling_callback_t)(uc_t *uc, void *user_data, uc_event_t *event);
typedef int (uc_signaling_error_callback_t)(uc_t *uc, void *user_data, int cause);
typedef void (uc_channel_read_callback_t)(uc_t *uc, int ch, void *user_data, const uint8_t *buf, int len);
typedef int (uc_channel_write_callback_t)(uc_t *uc, int ch, void *user_data, uint8_t *buf, int len);
typedef int (uc_channel_error_callback_t)(uc_t *uc, int ch, void *user_data, int cause);
typedef int (uc_channel_read_codec_callback_t)(uc_t *uc, int ch, void *user_data, uint8_t *buf, int len);
typedef int (uc_channel_write_codec_callback_t)(uc_t *uc, int ch, void *user_data, uint8_t *buf, int len);

/*! \brief Initialise the UniCall library
   
    Initialise the UniCall library.
*/
extern int uc_start(void);

/*! \brief Get a call's state
    Get the current state of a call. Note this returns the current state of
    the call. It makes no allowance for events which might be queued. It is
    provided mostly as a debug aid.
    \param uc The UniCall context
    \param crn Call reference number
    \param state
    \return A UniCall completion code.
*/
extern int uc_get_call_state(uc_t *uc, uc_crn_t crn, int *state);


/*! \brief Call control
    \param uc The UniCall context.
    \param op The operation to perform.
    \param crn Call reference number.
    \param data A data structure, which varies with the operation being performed.
    \return A UniCall completion code.
*/
extern int uc_call_control(uc_t *uc, int op, uc_crn_t crn, void *data);

/*! Open a channel
    \return the file descriptor, or -1 if there is an error.
*/
int uc_channel_open(const char *protocol,
                    const char *variant,
                    int mode,
                    char *dev);

/*! Close a channel
    \return 0 for OK, or -1 if there is an error.
*/
int uc_channel_close(const char *protocol,
                     const char *variant,
                     int mode,
                     int fd);

/*! Set the receive and transmit gains for a channel
    \param uc The UniCall context
    \param ch The channel number
    \param rxgain
    \param txgain
    \return A UniCall completion code.
*/
int uc_channel_gains(uc_t *uc, int ch, float rxgain, float txgain);

/*! Control echo cancellation
    \param uc The UniCall context
    \param ch The channel number
    \param op Operation to perform
    \return A UniCall completion code.
*/
int uc_channel_echo_cancel(uc_t *uc, int ch, int op);

/*! Control circuit switching
    \param uc The UniCall context
    \param ch The channel number
    \param op Operation to perform
    \param dest Destination channel or group
    \param parms Parameters
    \return A UniCall completion code.
*/
int uc_channel_switching(uc_t *uc, int ch, int op, int dest, int parms);

/*! Flush a channel's audio buffers.
    \param uc The UniCall context
    \param ch The channel number
    \param which ???
    \return A UniCall completion code.
*/
extern int uc_channel_flush(uc_t *uc, int ch, int which);

/*! \brief Get the device handles associated with a channel. */
extern int uc_get_device_handles(uc_t *uc, int ch, int *fe, int *be);

/*! \brief Get a channel's native codec.
    \param uc The UniCall context
    \param ch The channel number */
extern int uc_channel_get_native_codec(uc_t *uc, int ch);

/*! \brief Get the codec currently used at the API (read/write) interface.
    \param uc The UniCall context
    \param ch The channel number */
extern int uc_channel_get_api_codec(uc_t *uc, int ch);

/*! \brief Set the codec to be used at the API (read/write) interface.
    \param uc The UniCall context
    \param ch The channel number
    \param codec The codec */
extern int uc_channel_set_api_codec(uc_t *uc, int ch, int codec);

/*! \brief Write to a channel. */
extern int uc_channel_write(uc_t *uc, int ch, uint8_t *buf, int len);

/*! \brief Check for events to be processed. */
extern uc_event_t *uc_check_event(uc_t *uc);

/*! \brief Create a new protocol context. */
extern uc_t *uc_new(int fe,
                    int be,
                    const char *protocol,
                    const char *variant,
                    int mode,
                    int outgoing_calls_allowed);

/*! \brief Delete a protocol context. */
extern int uc_delete(uc_t *uc);

/*! \brief Create or initialise a uc_callparms_t call parameter structure
    Create or initialise a uc_callparms_t structure. 
    Returns a pointer to the structure, or NULL if allocation failed.
    The free() function should be used to free any allocated memory when it is
    no longer required.
    \param parms If parms is NULL, a new structure will be allocated. If parms is
                 not NULL is should point to an existing uc_callparms_t structure
                 which is to be initialised for use.
    \return A pointer to the callparms structure, or NULL if one could not be
            created (e.g. if out of memory).
*/
extern uc_callparms_t *uc_new_callparms(uc_callparms_t *parms);
/*! \brief Set a call parameter. */
extern void uc_callparm_bear_cap_transfer_cap(uc_callparms_t *parms, int cap);
/*! \brief Set a call parameter. */
extern void uc_callparm_bear_cap_transfer_mode(uc_callparms_t *parms, int mode);
/*! \brief Set a call parameter. */
extern void uc_callparm_bear_cap_transfer_rate(uc_callparms_t *parms, int rate);
/*! \brief Set a call parameter. */
extern void uc_callparm_userinfo_layer1_protocol(uc_callparms_t *parms, int prot);
/*! \brief Set a call parameter. */
extern void uc_callparm_user_rate(uc_callparms_t *parms, int rate);
/*! \brief Set a call parameter. */
extern void uc_callparm_destination_npi(uc_callparms_t *parms, int npi);
/*! \brief Set a call parameter. */
extern void uc_callparm_destination_ton(uc_callparms_t *parms, int ton);
/*! \brief Set a call parameter. */
extern void uc_callparm_destination_number(uc_callparms_t *parms, const char *num);
/*! \brief Set a call parameter. */
extern void uc_callparm_destination_sub_addr_npi(uc_callparms_t *parms, int npi);
/*! \brief Set a call parameter. */
extern void uc_callparm_destination_sub_addr_ton(uc_callparms_t *parms, int ton);
/*! \brief Set a call parameter. */
extern void uc_callparm_destination_sub_addr_number(uc_callparms_t *parms, const char *num);
/*! \brief Set a call parameter. */
extern void uc_callparm_originating_presentation(uc_callparms_t *parms, int pres);
/*! \brief Set a call parameter. */
extern void uc_callparm_originating_ton(uc_callparms_t *parms, int ton);
/*! \brief Set a call parameter. */
extern void uc_callparm_originating_npi(uc_callparms_t *parms, int npi);
/*! \brief Set a call parameter. */
extern void uc_callparm_originating_number(uc_callparms_t *parms, const char *num);
/*! \brief Set a call parameter. */
extern void uc_callparm_originating_name(uc_callparms_t *parms, const char *name);
/*! \brief Set a call parameter. */
extern void uc_callparm_originating_sub_addr_ton(uc_callparms_t *parms, int ton);
/*! \brief Set a call parameter. */
extern void uc_callparm_originating_sub_addr_npi(uc_callparms_t *parms, int npi);
/*! \brief Set a call parameter. */
extern void uc_callparm_originating_sub_addr_number(uc_callparms_t *parms, const char *num);
/*! \brief Set a call parameter. */
extern void uc_callparm_redirecting_cause(uc_callparms_t *parms, int cause);
/*! \brief Set a call parameter. */
extern void uc_callparm_redirecting_presentation(uc_callparms_t *parms, int pres);
/*! \brief Set a call parameter. */
extern void uc_callparm_redirecting_ton(uc_callparms_t *parms, int ton);
/*! \brief Set a call parameter. */
extern void uc_callparm_redirecting_npi(uc_callparms_t *parms, int npi);
/*! \brief Set a call parameter. */
extern void uc_callparm_redirecting_number(uc_callparms_t *parms, const char *num);
/*! \brief Set a call parameter. */
extern void uc_callparm_redirecting_subaddr_ton(uc_callparms_t *parms, int ton);
/*! \brief Set a call parameter. */
extern void uc_callparm_redirecting_subaddr_npi(uc_callparms_t *parms, int npi);
/*! \brief Set a call parameter. */
extern void uc_callparm_redirecting_subaddr(uc_callparms_t *parms, const char *num);
/*! \brief Set a call parameter. */
extern void uc_callparm_original_called_number_ton(uc_callparms_t *parms, int ton);
/*! \brief Set a call parameter. */
extern void uc_callparm_original_called_number_npi(uc_callparms_t *parms, int npi);
/*! \brief Set a call parameter. */
extern void uc_callparm_original_called_number(uc_callparms_t *parms, const char *num);
/*! \brief Set a call parameter. */
extern void uc_callparm_calling_party_category(uc_callparms_t *parms, int category);

/*! \brief Specify a callback routine for signaling events. */
extern void uc_set_signaling_callback(uc_t *uc, uc_signaling_callback_t *func, void *user_data);

/*! \brief Specify a callback routine for signaling errors. */
extern void uc_set_signaling_error_callback(uc_t *uc, uc_signaling_error_callback_t *func, void *user_data);

/*! \brief Specify a callback routine. */
extern int uc_set_channel_read_callback(uc_t *uc, int ch, uc_channel_read_callback_t *func, void *user_data);

/*! \brief Specify a callback routine. */
extern int uc_set_channel_write_callback(uc_t *uc, int ch, uc_channel_write_callback_t *func, void *user_data);

/*! \brief Specify a callback routine. */
extern int uc_set_channel_error_callback(uc_t *uc, int ch, uc_channel_error_callback_t *func, void *user_data);

/*! \brief Specify a callback routine. */
extern int uc_set_channel_read_codec_callback(uc_t *uc, int ch, uc_channel_read_codec_callback_t *func, void *user_data);

/*! \brief Specify a callback routine. */
extern int uc_set_channel_write_codec_callback(uc_t *uc, int ch, uc_channel_write_codec_callback_t *func, void *user_data);

/*! \brief Specify the logging level for a channel, and a tag to be used in the
           log to identify the channel. */
extern void uc_set_logging(uc_t *uc, int level, int format, const char *tag);

extern const char *uc_error_message(uc_t *uc, int code);

extern void uc_set_message_handler(void (*func)(int level, const char *text));
extern void uc_set_error_handler(void (*func)(const char *text));
extern int uc_str_to_protocol_variant(int protocol_class, const char *variant);

/*! \brief Return a short text description of a UniCall return code. */
extern const char *uc_ret_to_str(int code);

/*! \brief Return a short string description of a UniCall state code. */
extern const char *uc_state_to_str(int state);

/*! \brief Return a short string description of a UniCall event code. */
extern const char *uc_event_to_str(int id);

/*! \brief Return a short string description of a disconnect cause code. */
extern const char *uc_cause_to_str(int cause);

/*! \brief Return a short string description of a protocol class. */
extern const char *uc_protocol_class_to_str(int protocol_class);

/*! \brief Return a very short string description of a protocol class. */
extern const char *uc_protocol_class_to_short_str(int protocol_class);

/*! \brief Return a short string description of a protocol variant. */
extern const char *uc_protocol_variant_to_str(int protocol_class, int protocol_variant);

/*! \brief Dump the contents of an event structure, for debug purposes. */
extern void uc_dump_event(uc_t *uc, uc_event_t *ev);

/*! \brief Simple built in scheduler functions for use where the programming environment
    does not offer such facilities. */
extern struct timeval *uc_schedule_next(uc_t *uc);
extern void uc_schedule_run(uc_t *uc);

/* Logging elements */
enum
{
    UC_LOG_SEVERITY_MASK            = 0x00FF,
    UC_LOG_SHOW_DATE                = 0x0100,
    UC_LOG_SHOW_SAMPLE_TIME         = 0x0200,
    UC_LOG_SHOW_SEVERITY            = 0x0400,
    UC_LOG_SHOW_PROTOCOL            = 0x0800,
    UC_LOG_SHOW_VARIANT             = 0x1000,
    UC_LOG_SHOW_TAG                 = 0x2000,
    UC_LOG_SUPPRESS_LABELLING       = 0x8000
};

/* Logging severity levels */
enum
{
    UC_LOG_NONE                     = 0,
    UC_LOG_ERROR                    = 1,
    UC_LOG_WARNING                  = 2,
    UC_LOG_PROTOCOL_ERROR           = 3,
    UC_LOG_PROTOCOL_WARNING         = 4,
    UC_LOG_INFO                     = 5,
    UC_LOG_FLOW                     = 6,
    UC_LOG_FLOW_2                   = 7,
    UC_LOG_FLOW_3                   = 8,
    UC_LOG_CAS                      = 9,
    UC_LOG_TONE                     = 10,
    UC_LOG_DEBUG_1                  = 11,
    UC_LOG_DEBUG_2                  = 12,
    UC_LOG_DEBUG_3                  = 13
};

/* Return codes for most of the library functions */
typedef enum
{
    UC_RET_OK                       = 0,
    UC_RET_WRONG_PROTOCOL           = 1,
    UC_RET_BAD_PARAMETER            = 2,
    UC_RET_BAD_DEVICE               = 3,
    UC_RET_BAD_STATE                = 4,
    UC_RET_DEVICE_ERROR             = 5,
    UC_RET_UNSUPPORTED              = 6,
    UC_RET_CRN_NOT_KNOWN            = 7,
    UC_RET_BAD_CHANNEL              = 8,
    UC_RET_MEMORY_ERROR             = 9,
    UC_RET_BLOCKED                  = 10,
    UC_RET_NOT_AVAILABLE            = 11
} uc_ret_t;

/* API level codecs */
typedef enum
{
    UC_CODEC_UNKNOWN                = 0,
    UC_CODEC_DEFAULT                = 1,
    UC_CODEC_LINEAR16               = 2,
    UC_CODEC_ALAW                   = 3,
    UC_CODEC_ULAW                   = 4
} uc_codec_t;

/* The following codes all match those in Q.931. They may, therefore
   be used directly in ISDN applications. */

/* Numbering plan identifier (NPI) */
typedef enum
{
    UC_NPI_UNKNOWN                                              = 0x0,
    UC_NPI_E163_E164                                            = 0x1,
    UC_NPI_X121                                                 = 0x3,
    UC_NPI_F69                                                  = 0x4,
    UC_NPI_NATIONAL                                             = 0x8,
    UC_NPI_PRIVATE                                              = 0x9,
    UC_NPI_RESERVED                                             = 0xF
} uc_npi_t;

/* Type of number (TON) */
typedef enum
{
    UC_TON_UNKNOWN                                              = 0x0,
    UC_TON_INTERNATIONAL                                        = 0x1,
    UC_TON_NATIONAL                                             = 0x2,
    UC_TON_NET_SPECIFIC                                         = 0x3,
    UC_TON_SUBSCRIBER                                           = 0x4,
    UC_TON_ALPHANUMERIC                                         = 0x5,
    UC_TON_ABBREVIATED                                          = 0x6,
    UC_TON_RESERVED                                             = 0x7
} uc_ton_t;

/* Redirection reasons */
typedef enum
{
    UC_REDIR_UNKNOWN                                            = 0x0,
    UC_REDIR_FORWARD_ON_BUSY                                    = 0x1,
    UC_REDIR_FORWARD_ON_NO_REPLY                                = 0x2,
    UC_REDIR_DEFLECTION                                         = 0x3,
    UC_REDIR_DTE_OUT_OF_ORDER                                   = 0x9,
    UC_REDIR_FORWARDED_BY_DTE                                   = 0xA,
    UC_REDIR_UNCONDITIONAL                                      = 0xF
} uc_redir_t;
    
/* Presentation */
typedef enum
{
    UC_PRES_ALLOWED_USER_NUMBER_NOT_SCREENED                    = 0x00,
    UC_PRES_ALLOWED_USER_NUMBER_PASSED_SCREEN                   = 0x01,
    UC_PRES_ALLOWED_USER_NUMBER_FAILED_SCREEN                   = 0x02,
    UC_PRES_ALLOWED_NETWORK_NUMBER                              = 0x03,
    UC_PRES_PROHIB_USER_NUMBER_NOT_SCREENED                     = 0x20,
    UC_PRES_PROHIB_USER_NUMBER_PASSED_SCREEN                    = 0x21,
    UC_PRES_PROHIB_USER_NUMBER_FAILED_SCREEN                    = 0x22,
    UC_PRES_PROHIB_NETWORK_NUMBER                               = 0x23,
    UC_PRES_NUMBER_NOT_AVAILABLE                                = 0x43
} uc_pres_t;

/* Bearer transfer capabilities */
typedef enum
{
    UC_TRANS_CAP_SPEECH                                         = 0x00,
    UC_TRANS_CAP_DIGITAL                                        = 0x08,
    UC_TRANS_CAP_RESTRICTED_DIGITAL                             = 0x09,
    UC_TRANS_CAP_3_1K_AUDIO                                     = 0x10,
    UC_TRANS_CAP_7K_AUDIO                                       = 0x11,
    UC_TRANS_CAP_VIDEO                                          = 0x18
} uc_trans_cap_t;

/* The 4ESS uses a different audio field */
#define UC_TRANS_CAP_AUDIO_4ESS                                 0x08

/* Bearer information transfer modes */
typedef enum
{
    UC_ITM_CIRCUIT                                              = 0x00,
    UC_ITM_PACKET                                               = 0x02
} uc_itm_t;

/* Bearer transfer modes */
typedef enum
{
    UC_TRANS_MODE_64_CIRCUIT                                    = 0x10,
    UC_TRANS_MODE_2x64_CIRCUIT                                  = 0x11,
    UC_TRANS_MODE_384_CIRCUIT                                   = 0x13,
    UC_TRANS_MODE_1536_CIRCUIT                                  = 0x15,
    UC_TRANS_MODE_1920_CIRCUIT                                  = 0x17,
    UC_TRANS_MODE_MULTIRATE                                     = 0x18,
    UC_TRANS_MODE_PACKET                                        = 0x40
} uc_trans_mode_t;

/* Bearer transfer rates */
typedef enum
{
    UC_BEAR_PACKET_MODE                                         = 0x00,
    UC_BEAR_RATE_64KBPS                                         = 0x10,
    UC_BEAR_RATE_128KBPS                                        = 0x11,
    UC_BEAR_RATE_384KBPS                                        = 0x13,
    UC_BEAR_RATE_1536KBPS                                       = 0x15,
    UC_BEAR_RATE_1920KBPS                                       = 0x17
} uc_bear_rate_t;

/* User info layer 1 types */
typedef enum
{
    UC_LAYER_1_ITU_RATE_ADAPT                                   = 0x21,
    UC_LAYER_1_ULAW                                             = 0x22,
    UC_LAYER_1_ALAW                                             = 0x23,
    UC_LAYER_1_G721                                             = 0x24,
    UC_LAYER_1_G722_G725                                        = 0x25,
    UC_LAYER_1_G7XX_384K                                        = 0x26,
    UC_LAYER_1_NON_ITU_ADAPT                                    = 0x27,
    UC_LAYER_1_V120_RATE_ADAPT                                  = 0x28,
    UC_LAYER_1_X31_RATE_ADAPT                                   = 0x29
} uc_layer_1_t;

/* User info layer 2 types */
typedef enum
{
    UC_LAYER_2_Q921                                             = 0x42,
    UC_LAYER_2_LAPB                                             = 0x46
} uc_layer_2_t;

/* User info layer 3 types */
typedef enum
{
    UC_LAYER_3_Q931                                             = 0x62,
    UC_LAYER_3_X25                                              = 0x66
} uc_layer_3_t;

/* ITU rate adaption types */
typedef enum
{
    UC_RATE_ADAPT_EBITS                                         = 0x00,
    UC_RATE_ADAPT_0K6                                           = 0x01,
    UC_RATE_ADAPT_1K2                                           = 0x02,
    UC_RATE_ADAPT_2K4                                           = 0x03,
    UC_RATE_ADAPT_3K6                                           = 0x04,
    UC_RATE_ADAPT_4K8                                           = 0x05,
    UC_RATE_ADAPT_7K2                                           = 0x06,
    UC_RATE_ADAPT_8K                                            = 0x07,
    UC_RATE_ADAPT_9K6                                           = 0x08,
    UC_RATE_ADAPT_14K4                                          = 0x09,
    UC_RATE_ADAPT_16K                                           = 0x0A,
    UC_RATE_ADAPT_19K2                                          = 0x0B,
    UC_RATE_ADAPT_32K                                           = 0x0C,
    UC_RATE_ADAPT_48K                                           = 0x0E,
    UC_RATE_ADAPT_56K                                           = 0x0F,
    UC_RATE_ADAPT_64K                                           = 0x10,
    UC_RATE_ADAPT_0K1345                                        = 0x15,
    UC_RATE_ADAPT_0K1                                           = 0x16,
    UC_RATE_ADAPT_0K075_1K2                                     = 0x17,
    UC_RATE_ADAPT_1K2_0K075                                     = 0x18,
    UC_RATE_ADAPT_0K05                                          = 0x19,
    UC_RATE_ADAPT_0K075                                         = 0x1A,
    UC_RATE_ADAPT_0K110                                         = 0x1B,
    UC_RATE_ADAPT_0K150                                         = 0x1C,
    UC_RATE_ADAPT_0K200                                         = 0x1D,
    UC_RATE_ADAPT_0K300                                         = 0x1E,
    UC_RATE_ADAPT_12K                                           = 0x1F
} uc_rate_adapt_t;

/* Location codes */
typedef enum
{
    UC_LOCATION_USER                                            = 0x0,
    UC_LOCATION_PRIV_NET_LOCAL_USER                             = 0x1,
    UC_LOCATION_PUB_NET_LOCAL_USER                              = 0x2,
    UC_LOCATION_TRANSIT_NET                                     = 0x3,
    UC_LOCATION_PUB_NET_REMOTE_USER                             = 0x4,
    UC_LOCATION_PRIV_NET_REMOTE_USER                            = 0x5,
    UC_LOCATION_INTERNATIONAL_NETWORK                           = 0x7,
    UC_LOCATION_NETWORK_BEYOND_INTERWORKING                     = 0xA
} uc_location_t;

typedef enum
{
    UC_CALLER_CATEGORY_NATIONAL_SUBSCRIBER_CALL                 = 0,
    UC_CALLER_CATEGORY_NATIONAL_PRIORITY_SUBSCRIBER_CALL,
    UC_CALLER_CATEGORY_NATIONAL_MAINTENANCE_CALL,
    UC_CALLER_CATEGORY_NATIONAL_OPERATOR_CALL,
    UC_CALLER_CATEGORY_NATIONAL_DATA_CALL,
    UC_CALLER_CATEGORY_NATIONAL_PAYPHONE_CALL,
    UC_CALLER_CATEGORY_INTERNATIONAL_SUBSCRIBER_CALL,
    UC_CALLER_CATEGORY_INTERNATIONAL_PRIORITY_SUBSCRIBER_CALL,
    UC_CALLER_CATEGORY_INTERNATIONAL_OPERATOR_CALL,
    UC_CALLER_CATEGORY_INTERNATIONAL_DATA_CALL,
    UC_CALLER_CATEGORY_UNKNOWN
} uc_caller_category_t;

enum
{
    /*! Make (initiate) a call 

        Initiate a call according to the specification (ANI, DNIS, etc.) in callparms.
        Requires a set of call parameters (ANI, DNIS, etc.)
    */
    UC_OP_MAKECALL = 1,
    
    /*! Request more call setup information.
    
        Request more call setup information for a call which has been offered. This may
        be used to get additional dialed digits, to request missing ANI information, etc.
    
        This should be used in response to a UC_EVENT_DIALED_NUMBER. If more digits are
        required, the new minimum length from the dialed number should be specified. If
        the number of complete, or incorrect, the minimum length should be set to -1. This
        tells the protocol software not to get any more digits, and to proceed with any
        other tasked needed to reach the UC_EVENT_OFFERED stage. Note that if the number was
        incorrect, a UC_EVENT_OFFERED event will occur, and the call should be rejected at
        that time.
    */
    UC_OP_REQUESTMOREINFO,
    
    /*! Send more call setup information.
    
        Send more call setup information, which was not available when the call was
        initiated. This may be used to send things like extra dialed digits, which were
        entered late.
    */
    UC_OP_SENDMOREINFO,
    
    /*! Acknowledge a call that has been offered in a UC_EVENT_OFFERED event.
    */
    UC_OP_CALLACK,
    
    /*! Accept a call that has been offered in a UC_EVENT_OFFERED (or a subsequent
        UC_EVENT_MOREDIGITS) event.
   
        Accept a call that has been offered in a UC_EVENT_OFFERED event. This means
        the call is to a valid destination, which is free to accept calls. The call
        is then effectively ringing.
    */
    UC_OP_ACCEPTCALL,
    
    /*! Answer a call that has been offered in a UC_EVENT_OFFERED (or a subsequent
        UC_EVENT_MOREDIGITS) event.
   
        Answer a call that has been offered in a UC_EVENT_OFFERED event. The call
        may have already been accepted, using uc_AcceptCall, or this step may have
        been skipped.  This should result in either a UC_EVENT_ANSWERED event, when
        the call has been fully answered, or a UC_EVENT_FARDISCONNECTED event, if
        the far end clears before answer.
    */
    UC_OP_ANSWERCALL,
    
    /*! Drop (clear) a call.

        Clear a call. Once the far end has also cleared the call, this should
        result in a UC_EVENT_DROPCALL event.
    */
    UC_OP_DROPCALL,
    
    /*! Release a call, and free any resources associated with it.
   
        Release a call, and free any resources associated with it. Once this
        operation is performed, the CRN will no longer be valid. This should
        result in a UC_EVENT_RELEASECALL event, when any slow activities related
        to releasing the call have completed.
    */
    UC_OP_RELEASECALL,
    
    /*! Unblock a trunk or trunks.
        Unblock a trunk. This is a somewhat protocol dependant operation. It will
        usually allow incoming calls, on a trunk capable of incoming calls. It
        may enable outgoing ones, too. This operation will result in a
        UC_EVENT_FARUNBLOCKED event. Once the trunk is unblocked.

        Requires the channel to unblock. -1 means all channels associated with the context.
    */
    UC_OP_UNBLOCK,
    
    /*! Block a trunk or trunks.
        Block a trunk. This is a somewhat protocol dependant operation. It will
        usually block incoming calls. It may blocking outgoing ones, too. This
        operation will result in a UC_EVENT_FARBLOCKED event. Once the trunk is
        blocked.
    
        Requires the channel to block. -1 means all channels associated with the context.
    */    
    UC_OP_BLOCK,
    
    /*! Exchange user to user information.
    */
    UC_OP_USERTOUSER
};

enum
{
    UC_PROTOCOL_FAIL_DEVICE_IO_ERROR            = 0x0001,
};

#if defined(__cplusplus)
extern "C"
{
#endif

/*! \brief Turn a presentation code to a string */
extern const char *uc_pres_to_str(int code);

/*! \brief Turn a call redirection cause code to a string */
const char *uc_redirection_reason_to_str(int code);

/*! \brief Turn type of number (TON) code to a string */
extern const char *uc_ton_to_str(int code);

/*! \brief Turn number plan indicator (NPI) code to a string */
extern const char *uc_npi_to_str(int code);

/*! \brief Turn a sub-address type to a string */
const char *uc_subaddrtype_to_str(int code);

/*! \brief Turn a call type to a string */
const char *uc_calltype_to_str(int codee);

/*! \brief Turn a transfer capability code to a string */
const char *uc_trans_cap_to_str(int code);

/*! \brief Turn a transfer mode code to a string */
const char *uc_trans_mode_to_str(int code);

/*! \brief Turn a rate adaption code to a string */
const char *uc_rate_adapt_to_str(int code);

/*! \brief Turn a layer 1 code to a string */
const char *uc_layer1_to_str(int code);

/*! \brief Turn a layer 2 code to a string */
const char *uc_layer2_to_str(int code);

/*! \brief Turn a layer 3 code to a string */
const char *uc_layer3_to_str(int code);

/*! \brief Turn a location code to a string */
const char *uc_location_to_str(int code);

/*! \brief Turn a calling party category code to a string */
const char *uc_calling_party_category_to_str(int code);

/*! \brief Generate a formatted dump of the contents of the passed
           callparms structure to the log. */
extern void uc_dump_callparms(uc_t *uc, uc_callparms_t *callparms, int verbose);

#if defined(__cplusplus)
}
#endif

#endif
/*- End of file ------------------------------------------------------------*/
