/*
 * protocol mfcr2 - a unicall protocol library for MFC/R2 signaling on E1 lines
 *
 * r2.h
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2002, 2003, 2005, 2007 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 2.1,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: r2.h,v 1.3 2008/04/17 17:22:21 steveu Exp $
 */

#if !defined(_R2_H_)
#define _R2_H_

/* Welcome to the weird and wonderful world of R2 signaling */

enum
{
    R2_IDLE                                     = 0,

    /* Call states for R2 protocol inbound (backward) */
    /*! Sending SEIZE ACK in response to SEIZE */
    R2_BACK_SEIZE_ACK                           = 101,
    /*! Sending ANSWER signal, after receiving SIGII_1 tone off */
    R2_BACK_ANSWER                              = 102,
    /*! Sending CLEAR BACK after ANSWER */
    R2_BACK_CLEAR_BACK                          = 103,
    /*! Got CLEAR FWD before CLEAR BACK */
    R2_BACK_CLEAR_FWD                           = 104,
    /*! Sending timed RELEASE GUARD to prevent early re-seize from our side. */
    R2_BACK_RELEASE_GUARD                       = 105,

    /* Call states for R2 protocol outbound (forward) */
    /*! Sending SEIZE */
    R2_FWD_SEIZE                                = 201,
    /*! Received SEIZE ACK in response to SEIZE */
    R2_FWD_SEIZE_ACK                            = 202,
    /*! Waiting for CAS ANSWER */
    R2_FWD_AWAIT_ANSWER                         = 203,
    /*! CAS ANSWER has preceeded completion of the tone signaling */
    R2_FWD_ANSWERED_BEFORE_TONES_COMPLETE       = 204,
    /*! The call is fully answered (tone signaling + CAS bits) */
    R2_FWD_ANSWERED                             = 205,
    /*! CLEAR BACK before ANSWER */
    R2_FWD_CLEAR_BACK_BEFORE_ANSWER             = 206,
    /*! CLEAR BACK after ANSWER */
    R2_FWD_CLEAR_BACK_AFTER_ANSWER              = 207,
    /*! Want to send CLEAR FWD before receiving SEIZE ACK */
    R2_FWD_CLEAR_FWD_BEFORE_SEIZE_ACK           = 208,
    /*! Sending CLEAR FWD before receiving ANSWER */
    R2_FWD_CLEAR_FWD_BEFORE_ANSWER              = 209,
    /*! Sending CLEAR FWD after receiving ANSWER */
    R2_FWD_CLEAR_FWD_AFTER_ANSWER               = 210,
    /*! Sending CLEAR FWD after receiving CLEAR BACK. Waiting for RELEASE GUARD */
    R2_FWD_CLEAR_FWD_AFTER_CLEAR_BACK           = 211
} r2_state_e;

typedef struct
{
    /*! \brief Mask to pick out the R2 signaling bits from amongst the ABCD bits. */
    uint8_t abcd_mask;
    /*! \brief The value to use on transmission for the ABCD bits which are not the
               R2 signaling bits. */
    uint8_t base_abcd_bits;
    /*! \brief The idle pattern for the ABCD bits. */
    uint8_t abcd_idle;
    /*! \brief The blocking pattern for the ABCD bits. */
    uint8_t abcd_blocking;

    /*! \brief The seize pattern for the ABCD bits. */
    uint8_t fwd_abcd_seize;
    /*! \brief The clear forward pattern for the ABCD bits. */
    uint8_t fwd_abcd_clear_fwd;

    /*! \brief The seize acknowledge pattern for the ABCD bits. */
    uint8_t back_abcd_seize_ack;
    /*! \brief The answer pattern for the ABCD bits. */
    uint8_t back_abcd_answer;
    /*! \brief The clear back pattern for the ABCD bits. */
    uint8_t back_abcd_clear_back;
    /*! \brief The billing pulse pattern fot the ABCD bits. */
    uint8_t abcd_billing_pulse;
    /*! \brief The duration of billing pulses. */
    int billing_pulse_duration;

    /*! \brief */
    int blocking_release_time;
    /*! \brief Zero may be OK */
    int release_guard_time;
    /*! ITU spec says 200ms */
    int max_seize_ack_wait;
    /*! \brief Clear back persistence check, in milliseconds. This can be zero for
               most purposes */
    int clear_back_persistence_check;
    /*! \brief The maximum time we will for the called party to answer, after the
               call setup is complete. 60s is common practice */
    int max_await_answer;
    /*! \brief 20ms should be OK */
    int inbound_cas_persistence_check;
    /*! \brief 20ms should be OK */
    int outbound_cas_persistence_check;

    /*! \brief The current state of the R2 line signalling. */
    int state;
    /*! \brief The current CAS ABCD pattern being received. */
    uint8_t current_abcd_in;

    /*! \brief Timer used to ensure we do not seize to aggressively as the far end unblocks. */
    int far_unblocking_timer;
    /*! \brief Timer used to ensure we do not seize to aggressively after we unblock. */
    int local_unblocking_timer;
    /*! \brief The seize acknowledge timeout, used to detect failure at the remote end. */
    int seize_ack_timer;
    /*! \brief The release guard timeout. */
    int release_guard_timer;
    /*! \brief A timer to handle the possibility that the called party never answers. */
    int await_answer_timer;
    /*! \brief Clear back persistence timer, used to differentiate between metering pulses
        and real clear back conditions. */
    int clear_back_persistence_timer;
    /*! \brief Timer for the duration of a billing pulse. */
    int billing_pulse_timer;

    /*! \brief TRUE is the far end is in the blocked state. */
    int far_blocked;
    /*! \brief TRUE is the local end is in the blocked state. */
    int local_blocked;
} r2_signaling_state_t;

#endif
