#include <zaptel/zaptel.h>

static int device_set_codec(uc_t *uc, int ch, int codec)
{
    int linear;
    int law;

    //uc_log(uc, UC_LOG_FLOW, "Channel set API codec - %d\n", codec);
    switch (codec)
    {
    case UC_CODEC_DEFAULT:
        linear = 0;
        if (ioctl(uc->chan[ch].handle, ZT_SETLINEAR, &linear))
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, UC_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return UC_RET_DEVICE_ERROR;
        }
        /*endif*/
        law = uc->chan[ch].native_law;
        if (law == ZT_LAW_ALAW)
            codec = UC_CODEC_ALAW;
        else if (law == ZT_LAW_MULAW)
            codec = UC_CODEC_ULAW;
        /*endif*/
        break;
    case UC_CODEC_LINEAR16:
        linear = 1;
        if (ioctl(uc->chan[ch].handle, ZT_SETLINEAR, &linear))
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, UC_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return UC_RET_DEVICE_ERROR;
        }
        /*endif*/
        law = uc->chan[ch].native_law;
        break;
    case UC_CODEC_ALAW:
        linear = 0;
        if (ioctl(uc->chan[ch].handle, ZT_SETLINEAR, &linear))
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, UC_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return UC_RET_DEVICE_ERROR;
        }
        /*endif*/
        law = ZT_LAW_ALAW;
        break;
    case UC_CODEC_ULAW:
        linear = 0;
        if (ioctl(uc->chan[ch].handle, ZT_SETLINEAR, &linear))
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, UC_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return UC_RET_DEVICE_ERROR;
        }
        /*endif*/
        law = ZT_LAW_MULAW;
        break;
    default:
        return UC_RET_UNSUPPORTED;
    }
    /*endswitch*/
    if (uc->chan[ch].current_law != law)
    {
        uc->chan[ch].current_law = law;
        if (ioctl(uc->chan[ch].handle, ZT_SETLAW, &law))
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, UC_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return UC_RET_DEVICE_ERROR;
        }
        /*endif*/
    }
    /*endif*/
    uc->chan[ch].api_codec = codec;
    return UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int device_flush_audio(uc_t *uc, int ch, int which)
{
    mfcr2_signaling_state_t *mfcr2;
    int x;

    /* ch = -1 means all voice channels */
    if (ch >= uc->num_channels)
        return UC_RET_BAD_PARAMETER;
    /*endif*/
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Channel flush\n");
    if ((which & UC_FLUSH_WRITE))
    {
        x = ZT_FLUSH_WRITE;
        if (ioctl(uc->chan[ch].handle, ZT_FLUSH, &x))
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, UC_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return UC_RET_DEVICE_ERROR;
        }
        /*endif*/
    }
    /*endif*/
    if ((which & UC_FLUSH_READ))
    {
        /* TODO: implement input flush */
    }
    /*endif*/
    return UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int device_set_gains(uc_t *uc, int ch, float rxgain, float txgain)
{
    mfcr2_signaling_state_t *mfcr2;
    struct zt_gains g;
    float gain;
    int j;
    int k;

    if (ch >= uc->num_channels)
        return UC_RET_BAD_PARAMETER;
    /*endif*/
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Channel gains\n");
    g.chan = 0;
    if (rxgain != 0.0f)
    {
        /* Calculate linear value of rx gain */
        gain = powf(10.0f, rxgain/20.0f);
        if (uc->chan[ch].current_law == UC_CODEC_ALAW)
        {
            for (j = 0;  j < 256;  j++)
            {
                k = fsaturate(((float) alaw_to_linear(j))*gain);
                g.rxgain[j] = linear_to_alaw(k);
            }
            /*endfor*/
        }
        else
        {
            for (j = 0;  j < 256;  j++)
            {
                k = fsaturate(((float) ulaw_to_linear(j))*gain);
                g.rxgain[j] = linear_to_ulaw(k);
            }
            /*endfor*/
        }
        /*endif*/
    }
    else
    {
        for (j = 0;  j < 256;  j++)
            g.rxgain[j] = j;
        /*endfor*/
    }
    /*endif*/
    if (txgain != 0.0f)
    {
        /* Calculate linear value of tx gain */
        gain = powf(10.0f, txgain/20.0f);
        if (uc->chan[ch].current_law == UC_CODEC_ALAW)
        {
            for (j = 0;  j < 256;  j++)
            {
                k = fsaturate(((float) alaw_to_linear(j))*gain);
                g.txgain[j] = linear_to_alaw(k);
            }
            /*endfor*/
        }
        else
        {
            for (j = 0;  j < 256;  j++)
            {
                k = fsaturate(((float) ulaw_to_linear(j))*gain);
                g.txgain[j] = linear_to_ulaw(k);
            }
            /*endfor*/
        }
        /*endif*/
    }
    else
    {
        for (j = 0;  j < 256;  j++)
            g.txgain[j] = j;
        /*endfor*/
    }
    /*endif*/
    if (ioctl(uc->chan[ch].handle, ZT_SETGAINS, &g))
    {
        if (uc->chan[ch].channel_error)
            uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, UC_PROTOCOL_FAIL_DEVICE_IO_ERROR);
        /*endif*/
        return UC_RET_DEVICE_ERROR;
    }
    /*endif*/
    return UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int device_set_echo_cancellation(uc_t *uc, int ch, int op)
{
    mfcr2_signaling_state_t *mfcr2;
    int x;

    if (ch >= uc->num_channels)
        return UC_RET_BAD_PARAMETER;
    /*endif*/
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Channel echo cancel\n");
    switch (op)
    {
    case UC_ECHO_CANCEL_OFF:
    case UC_ECHO_CANCEL_ON:
        x = op;
        if (ioctl(uc->chan[ch].handle, ZT_ECHOCANCEL, &x) < 0)
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, UC_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return UC_RET_DEVICE_ERROR;
        }
        /*endif*/
        break;
    case UC_ECHO_CANCEL_NOTRAINING:
    case UC_ECHO_CANCEL_TRAINING:
        x = op - UC_ECHO_CANCEL_NOTRAINING;
        if (ioctl(uc->chan[ch].handle, ZT_ECHOTRAIN, &x) < 0)
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, UC_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return UC_RET_DEVICE_ERROR;
        }
        /*endif*/
        break;
    default:
        return UC_RET_BAD_PARAMETER;
    }
    /*endswitch*/
    return UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int device_set_media_switching(uc_t *uc, int ch, int op, int dest, int parms)
{
    struct zt_confinfo zi;
    mfcr2_signaling_state_t *mfcr2;
    int x;

    if (ch >= uc->num_channels)
        return UC_RET_BAD_PARAMETER;
    /*endif*/
    mfcr2 = (mfcr2_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Channel switching\n");
    switch (op)
    {
    case UC_SWITCHING_FREE:
        memset(&zi, 0, sizeof(zi));
        if (ioctl(uc->chan[ch].handle, ZT_SETCONF, &zi))
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, UC_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return UC_RET_DEVICE_ERROR;
        }
        /*endif*/
        break;
    case UC_SWITCHING_UNMUTE:
    case UC_SWITCHING_MUTE:
        x = op - UC_SWITCHING_UNMUTE;
        if (ioctl(uc->chan[ch].handle, ZT_CONFMUTE, &x))
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, UC_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return UC_RET_DEVICE_ERROR;
        }
        /*endif*/
        break;
    case UC_SWITCHING_CONNECT:
        memset(&zi, 0, sizeof(zi));
        zi.confno = dest;
        zi.confmode = parms;
        if (ioctl(uc->chan[ch].handle, ZT_SETCONF, &zi))
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, UC_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return UC_RET_DEVICE_ERROR;
        }
        /*endif*/
        break;
    default:
        return UC_RET_BAD_PARAMETER;
    }
    /*endswitch*/
    return UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int device_set_abcd_bits(uc_t *uc, int ch, int abcd)
{
    int x;

    x = abcd;
    if (ioctl(uc->chan[ch].handle, ZT_SETTXBITS, &x))
    {
        if (errno != EINPROGRESS)
        {
            if (uc->chan[ch].channel_error)
                uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, UC_PROTOCOL_FAIL_DEVICE_IO_ERROR);
            /*endif*/
            return UC_RET_DEVICE_ERROR;
        }
        /*endif*/
    }
    /*endif*/
    return UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int device_get_abcd_bits(uc_t *uc, int ch, int *abcd)
{
    int x;

    if (ioctl(uc->chan[ch].handle, ZT_GETRXBITS, &x))
    {
        if (uc->chan[ch].channel_error)
            uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, UC_PROTOCOL_FAIL_DEVICE_IO_ERROR);
        /*endif*/
        return UC_RET_DEVICE_ERROR;
    }
    /*endif*/
    *abcd = x;
    return UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int device_check_alarms(uc_t *uc, int ch, int *alarms)
{
    ZT_SPANINFO zi;

    memset(&zi, 0, sizeof(zi));
    zi.spanno = 0;
    if (ioctl(uc->chan[ch].handle, ZT_SPANSTAT, &zi))
    {
        uc_error("Unable to get alarm status: %s\n", strerror(errno));
        if (uc->chan[ch].channel_error)
            uc->chan[ch].channel_error(uc, ch, uc->chan[ch].channel_error_user_data, UC_PROTOCOL_FAIL_DEVICE_IO_ERROR);
        /*endif*/
        return UC_RET_DEVICE_ERROR;
    }
    /*endif*/
    *alarms = zi.alarms;
    return UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
