/*
 * libunicall - a library for universal call handling on both analogue and
 *              digital telephone circuits.
 *
 * simplesched.c
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2002, 2003 Steve Underwood
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 2.1,
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 * $Id: simplesched.c,v 1.9 2008/04/06 09:28:48 steveu Exp $
 */

#include <stdio.h>
#include <inttypes.h>
#include <stdlib.h>
#include <sys/time.h>

#include "unicall.h"
#include "unicall/hashtable.h"
#include "unicall/unicall-private.h"

#define FALSE 0
#define TRUE (!FALSE)

int uc_schedule_event(uc_t *uc, int ms, sched_callback_func_t function, void *user_data)
{
    int x;
    struct timeval tv;

    for (x = 0;  x < uc->sched.max_to_date;  x++)
    {
        if (uc->sched.sched[x].callback == NULL)
            break;
        /*endif*/
    }
    /*endfor*/
    if (x >= uc->sched.allocated)
    {
        uc->sched.allocated += 5;
        uc->sched.sched = (uc_sched_t *) realloc(uc->sched.sched, sizeof(uc_sched_t)*uc->sched.allocated);
    }
    /*endif*/
    if (x >= uc->sched.max_to_date)
        uc->sched.max_to_date = x + 1;
    /*endif*/
    gettimeofday(&tv, NULL);
    tv.tv_sec += ms/1000;
    tv.tv_usec += (ms%1000)*1000;
    if (tv.tv_usec > 1000000)
    {
        tv.tv_usec -= 1000000;
        tv.tv_sec += 1;
    }
    /*endif*/
    uc->sched.sched[x].when = tv;
    uc->sched.sched[x].callback = function;
    uc->sched.sched[x].user_data = user_data;
    return x;
}
/*- End of function --------------------------------------------------------*/

struct timeval *uc_schedule_next(uc_t *uc)
{
    struct timeval *closest = NULL;
    int x;

    for (x = 0;  x < uc->sched.max_to_date;  x++)
    {
        if (uc->sched.sched[x].callback
            && 
                (!closest
                ||
                (closest->tv_sec > uc->sched.sched[x].when.tv_sec)
                ||
                    ((closest->tv_sec == uc->sched.sched[x].when.tv_sec)
                    && 
                    (closest->tv_usec > uc->sched.sched[x].when.tv_usec))))
        {
            closest = &uc->sched.sched[x].when;
        }
        /*endif*/
    }
    /*endfor*/
    return closest;
}
/*- End of function --------------------------------------------------------*/

void uc_schedule_run(uc_t *uc)
{
    struct timeval tv;
    int x;
    sched_callback_func_t callback;
    void *user_data;

    gettimeofday(&tv, NULL);
    for (x = 0;  x < uc->sched.max_to_date;  x++)
    {
        if (uc->sched.sched[x].callback
            &&
                ((uc->sched.sched[x].when.tv_sec < tv.tv_sec)
                ||
                    ((uc->sched.sched[x].when.tv_sec == tv.tv_sec)
                    &&
                    (uc->sched.sched[x].when.tv_usec <= tv.tv_usec))))
        {
            callback = uc->sched.sched[x].callback;
            user_data = uc->sched.sched[x].user_data;
            uc->sched.sched[x].callback = NULL;
            uc->sched.sched[x].user_data = NULL;
            callback(uc, user_data);
        }
        /*endif*/
    }
    /*endfor*/
}
/*- End of function --------------------------------------------------------*/

void uc_schedule_del(uc_t *uc, int id)
{
    if (id >= uc->sched.max_to_date
        ||
        id < 0
        ||
        uc->sched.sched[id].callback == NULL)
    {
        uc_log(uc,
               UC_LOG_PROTOCOL_ERROR,
               "Received request to delete non-existant schedule id %d\n",
               id);
        return;
    }
    /*endif*/
    uc->sched.sched[id].callback = NULL;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
