/*
 * libunicall - a library for universal call handling on both analogue and
 *              digital telephone circuits.
 *
 * unicall-private.h
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2002 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 2.1,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: unicall-private.h,v 1.14 2008/04/06 09:28:48 steveu Exp $
 */

/*! \file */

#if !defined(_UNICALL_PRIVATE_H_)
#define _UNICALL_PRIVATE_H_

typedef void (*sched_callback_func_t)(uc_t *uc, void *user_data);

enum
{
    UC_LOOP_EVENT_OFF_HOOK = 1,
    UC_LOOP_EVENT_ON_HOOK = 2,
    UC_LOOP_EVENT_RING_TRIP = 3,
    UC_LOOP_EVENT_RINGING = 4,
    UC_LOOP_EVENT_WINK = 5,
    UC_LOOP_EVENT_PULSE_DIAL_COMPLETE = 6,
    UC_LOOP_EVENT_HOOK_COMPLETE = 7,
    UC_LOOP_EVENT_RINGER_OFF = 8,
    UC_LOOP_EVENT_RINGER_ON = 9,
    UC_LOOP_EVENT_PULSE_START = 10,
    UC_LOOP_EVENT_REVERSAL = 11
};

enum
{
    UC_CHANNEL_MODE_UNSPECIFIED = 0,
    UC_CHANNEL_MODE_CCS = 1,
    UC_CHANNEL_MODE_CAS = 2,
    UC_CHANNEL_MODE_LS_CPE = 3,
    UC_CHANNEL_MODE_LS_CO = 4,
    UC_CHANNEL_MODE_GS_CPE = 5,
    UC_CHANNEL_MODE_GS_CO = 6,
    UC_CHANNEL_MODE_KS_CPE = 7,
    UC_CHANNEL_MODE_KS_CO = 8,
};

enum
{
    UC_FLUSH_WRITE = 1,
    UC_FLUSH_READ = 2
};

typedef struct uc_sched_s
{
    struct timeval when;
    sched_callback_func_t callback;
    void *user_data;
} uc_sched_t;

typedef struct
{
    /*! \brief Data for the simple scheduler built into UniCall */
    int allocated;
    int max_to_date;
    uc_sched_t *sched;
} uc_sched_state_t;

typedef struct uc_event_queue_s
{
    /*! This helps with tracking if things have been released back to the pool
        after use */
    int total_allocated;
    /*! The number of allocated entries in the event queue */
    int queue_allocated;
    /*! The number of currently in use entries in the event queue */
    int queue;
    /*! The event queue */
    uc_event_t **events;
    /*! The number of allocated entries in the pool of free event structures */
    int pool_allocated;
    /*! The number of currently in use entries in the pool of free event structures */
    int pool;
    /*! The pool of free event structures */
    uc_event_t **event_pool;
} uc_event_queue_t;

typedef struct uc_chan_s
{
    int logging_level;
    int logging_format;
    const char *logging_tag;

    /*! Callback function to read data. */
    uc_channel_read_callback_t *channel_read;
    void *channel_read_user_data;
    /*! Callback function to write data. */
    uc_channel_write_callback_t *channel_write;
    void *channel_write_user_data;
    /*! Callback function to handle channel errors. */
    uc_channel_error_callback_t *channel_error;
    void *channel_error_user_data;
    /*! Callback function to handle codec reads. */
    uc_channel_read_codec_callback_t *channel_read_codec;
    void *channel_read_codec_user_data;
    /*! Callback function to handle codec writes. */
    uc_channel_write_codec_callback_t *channel_write_codec;
    void *channel_write_codec_user_data;
    /*! The native codec for the channel. */
    int native_codec;
    /*! The current API level codec for the channel. */
    int api_codec;

    uc_channel_read_callback_t *current_read;
    uc_channel_write_callback_t *current_write;
    int test_write_buf_empty;

    /*! \brief Per channel event queue */
    uc_event_queue_t event_queue;

    /*! \brief Main call level channel state */
    int state;

    int current_law;
    int native_law;
    int audio_bufsize;

    /*! \brief A flag to indicate if MF6 tones, supervisory tones, voice or whatever is active */
    int active_rx;
    /*! \brief A flag to indicate if MF6 tones, supervisory tones, voice or whatever is active */
    int active_tx;

    /*! Channel device handle */
    int handle;
} uc_chan_t;

typedef struct
{
    int t301;
    int t302;
    int t303;
    int t304;
    int t305;
    int t306;
    int t307;
    int t308;
    int t309;
    int t310;
    int t311;
    int t312;
    int t313;
    int t314;
    int t315;
    int t316;
    int t317;
    int t318;
    int t319;
    int t320;
    int t321;
    int t322;
} uc_timeouts_t;

struct uc_call_s
{
    /*! Call reference */
    int crn;
    /*! Force inversion of call number even if 0 */
    int force_invert;
    /*! Slotmap specified (bitmap of channels 31/24 - 1) (channel identifier IE) (-1 means not specified) */
    uint32_t slotmap;
    /*! An explicit channel (channel identifier IE) (-1 means not specified) */
    int channelno;
    /*! An explicit DS1 (-1 means not specified) */
    int ds1no;
    /*! Channel flags (0 means none retrieved) */
    int chanflags;

    /*! Channel within the unicall context carrying this call */
    int chan;

    /*! Whether or not the call is alive */
    int alive;
    /*! Whether setup has been acked or not */
    int sent_setup_ack;
    /*! Whether or not we have sent disconnect */
    int sent_disconnect;
    /*! Whether we've sent a call proceeding / alerting */
    int sent_proceeding;

    /*! Restart indicator (Restart Indicator IE) */
    int restart_indicator;

    /* Bearer capability */
    int transcapability;
    int transmoderate;
    int transmultiple;
    int user_layer1;
    int user_layer2;
    int user_layer3;
    int rateadaption;

    int sentchannel;

    /*! Progress coding */
    int progcode;
    /*! Progress location */    
    int progloc;
    /*! Progress indicator */
    int progress;
    
    /*! Cause coding */
    int causecode;
    /*! Cause location */
    int causeloc;
    /*! Cause of clearing */
    int cause;

    /*! Call state of peer as reported */
    int callstate;
    /*! Our call state */
    int ourcallstate;

    /*! API level call state */
    int state;
    int disconnect_reason;
    int incoming_call;

    uc_callparms_t callparms;

    int nonisdn;
    int useruserprotocoldisc;
    char useruserinfo[256];

    uc_timeouts_t timeouts;
    
    /*! A reference point to allow extra, protocol specific, material to be placed at the 
        end of the call structure. */
    uint8_t extra[0];
};

/*! \brief A UniCall control structure.

    This is the top level structure for managing calls in the UniCall
    library. It references one protocol dependant signaling structure,
    and one or more protocol dependant channel structures. For CAS
    protocols there is one uc_t per voice/data channel. For CCS protocols
    there is one uc_t per signaling channel, handling all the voice/data
    channels under that signaling channel's duristiction.
*/
struct uc_s
{
    int protocol_class;
    int protocol_variant;
    int protocol_mode;

    /*! \brief Per signaling channel signaling callback */
    uc_signaling_callback_t *signaling_func;
    /*! \brief An opaque pointer passed to the signaling callback function. */
    void *signaling_user_data;
    uc_signaling_error_callback_t *signaling_error_func;
    void *signaling_error_user_data;
    
    int logging_level;
    int logging_format;
    const char *logging_tag;

    uc_sched_state_t sched;

    /*! \brief Per signaling channel event queue */
    uc_event_queue_t event_queue;

    int next_crn;

    /*! \brief Protocol dependant signaling management data */
    void *signaling;
    /*! \brief The calls currently associated with this context */
    hash_table_t calls;
    /*! \brief The number of voice/data channels associated with this context */
    int num_channels;
    /*! \brief The voice/data channels associated with this context */
    struct uc_chan_s *chan;

    uc_timeouts_t timeouts;    
};

/* These routines should only be used by signaling protocol modules - not by
   any application code. */
extern int uc_schedule_event(uc_t *uc, int ms, void (*function)(uc_t *uc, void *data), void *data);
extern void uc_schedule_del(uc_t *uc, int id);

typedef struct uc_protocol_s uc_protocol_t;

typedef uc_event_t *(uc_check_event_t)(uc_t *uc);
typedef uc_t *(uc_new_t)(int fd, const char *variant, int mode, int outgoing_calls_allowed);
typedef int (uc_call_control_t)(uc_t *uc, int op, uc_call_t *call, void *data);
typedef char *(uc_error_message_t)(uc_t *uc, int code);
typedef int (uc_get_device_handles_t)(uc_t *uc, int ch, int *fe, int *be);
typedef int (uc_channel_open_t)(const char *variant, char *dev);
typedef int (uc_channel_close_t)(int fd);
typedef int (uc_channel_set_api_codec_t)(uc_t *uc, int ch, int codec);
typedef int (uc_channel_write_t)(uc_t *uc, int ch, uint8_t *buf, int chan);
typedef int (uc_channel_flush_t)(uc_t *uc, int chan, int which);
typedef int (uc_channel_gains_t)(uc_t *uc, int chan, float rxgain, float txgain);
typedef int (uc_channel_echo_cancel_t)(uc_t *uc, int chan, int op);
typedef int (uc_channel_switching_t)(uc_t *uc, int chan, int op, int dest, int parms);

/*! Protocol descriptor */
struct uc_protocol_s
{
    const char *name;                   /* Unique ASCII name of the protocol family */
    const char *desc;                   /* Short description */
    const char *short_desc;             /* Very short description, suitable as a tag in logs */
    const char *variants;               /* Comma separated list of the variants supported */
    const char *variant_descriptions;   /* Comma separated list descriptions of the variants supported */

    uc_check_event_t *xx_check_event;
    uc_new_t *xx_new;
    uc_call_control_t *xx_call_control;
    uc_error_message_t *xx_error_message;
    uc_get_device_handles_t *xx_get_device_handles;
    uc_channel_open_t *xx_channel_open;
    uc_channel_close_t *xx_channel_close;
    uc_channel_set_api_codec_t *xx_channel_set_api_codec;
    uc_channel_write_t *xx_channel_write;
    uc_channel_flush_t *xx_channel_flush;
    uc_channel_gains_t *xx_channel_gains;
    uc_channel_echo_cancel_t *xx_channel_echo_cancel;
    uc_channel_switching_t *xx_channel_switching;
};

typedef struct
{
    void *lib;
    uc_protocol_t *protocol;
} protocol_control_t;

extern void uc_message(char *fmt, ...);
extern void uc_error(char *fmt, ...);
extern int uc_log(uc_t *uc, int level, const char *fmt, ...);
extern int uc_log_start(void);

extern uc_call_t *uc_createcall(uc_t *uc, int extra);
extern void uc_destroycall(uc_t *uc, int crn);
extern uc_call_t *uc_getcall(uc_t *uc, int crn, int extra);
extern uc_call_t *uc_findcall(uc_t *uc, int crn);

int uc_report_event(uc_t *uc, uc_event_t *ev, int len);

#endif
