/*
 * libunicall - a library for universal call handling on both analogue and
 *              digital telephone circuits.
 *
 * simplesched.c
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2002, 2003 Steve Underwood
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 2.1,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: simplesched.c,v 1.15 2008/07/06 11:01:26 steveu Exp $
 */

#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif

#include <stdlib.h>
#include <inttypes.h>
#include <stdio.h>
#include <sys/time.h>

#include "unicall.h"
#include "unicall/hashtable.h"
#include "unicall/unicall-private.h"

#define FALSE 0
#define TRUE (!FALSE)

int uc_schedule_event(uc_t *uc, int ms, sched_callback_func_t function, void *user_data)
{
    int i;
    struct timeval tv;
    uc_sched_state_t *s;

    s = &uc->sched;
    for (i = 0;  i < s->max_to_date;  i++)
    {
        if (s->sched[i].callback == NULL)
            break;
        /*endif*/
    }
    /*endfor*/
    if (i >= s->allocated)
    {
        s->allocated += 5;
        s->sched = (uc_sched_t *) realloc(s->sched, sizeof(uc_sched_t)*s->allocated);
    }
    /*endif*/
    if (i >= s->max_to_date)
        s->max_to_date = i + 1;
    /*endif*/
    gettimeofday(&tv, NULL);
    tv.tv_sec += ms/1000;
    tv.tv_usec += (ms%1000)*1000;
    if (tv.tv_usec > 1000000)
    {
        tv.tv_usec -= 1000000;
        tv.tv_sec += 1;
    }
    /*endif*/
    s->sched[i].when = tv;
    s->sched[i].callback = function;
    s->sched[i].user_data = user_data;
    return i;
}
/*- End of function --------------------------------------------------------*/

struct timeval *uc_schedule_next(uc_t *uc)
{
    int i;
    struct timeval *closest;
    uc_sched_state_t *s;

    s = &uc->sched;
    closest = NULL;
    for (i = 0;  i < s->max_to_date;  i++)
    {
        if (s->sched[i].callback
            && 
                (!closest
                ||
                (closest->tv_sec > s->sched[i].when.tv_sec)
                ||
                    ((closest->tv_sec == s->sched[i].when.tv_sec)
                    && 
                    (closest->tv_usec > s->sched[i].when.tv_usec))))
        {
            closest = &s->sched[i].when;
        }
        /*endif*/
    }
    /*endfor*/
    return closest;
}
/*- End of function --------------------------------------------------------*/

void uc_schedule_run(uc_t *uc)
{
    struct timeval tv;
    int i;
    sched_callback_func_t callback;
    void *user_data;
    uc_sched_state_t *s;

    s = &uc->sched;
    gettimeofday(&tv, NULL);
    for (i = 0;  i < s->max_to_date;  i++)
    {
        if (s->sched[i].callback
            &&
                ((s->sched[i].when.tv_sec < tv.tv_sec)
                ||
                    ((s->sched[i].when.tv_sec == tv.tv_sec)
                    &&
                    (s->sched[i].when.tv_usec <= tv.tv_usec))))
        {
            callback = s->sched[i].callback;
            user_data = s->sched[i].user_data;
            s->sched[i].callback = NULL;
            s->sched[i].user_data = NULL;
            callback(uc, user_data);
        }
        /*endif*/
    }
    /*endfor*/
}
/*- End of function --------------------------------------------------------*/

void uc_schedule_del(uc_t *uc, int i)
{
    uc_sched_state_t *s;

    s = &uc->sched;
    if (i >= s->max_to_date
        ||
        i < 0
        ||
        s->sched[i].callback == NULL)
    {
        uc_log(uc,
               UC_LOG_PROTOCOL_ERROR,
               "Received request to delete non-existant schedule id %d\n",
               i);
        return;
    }
    /*endif*/
    s->sched[i].callback = NULL;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
