/* Set the protocol to use */
#define PROTOCOL_CLASS      "mfcr2"
#define PROTOCOL_VARIANT    "mx,20,4"
#define PROTOCOL_END        UC_MODE_CPE

/* Set the test numbers to use */
#define BASE_ANI            987654321
#define BASE_DNIS           1236

/* Set the trunks to use */
#define FIRST_TRUNK_CIRCUIT 94
#define NUM_TRUNKS          15
#define FIRST_OG_TRUNK      0
#define NUM_OG_TRUNKS       2

/*
 * libunicall - a library for universal call handling on both analogue and
 *              digital telephone circuits.
 *
 * testcall.c
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2002 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: testcall.c,v 1.14 2005/02/05 13:05:45 steveu Exp $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <stdlib.h>
#include <stdint.h>
#include <unistd.h>
#include <sys/signal.h>
#include <sys/select.h>
#include <sys/wait.h>
#include <sys/resource.h>
#include <sys/time.h>
#include <linux/zaptel.h>
#include <pthread.h>
#if defined(HAVE_SPANDSP_H)
#include <tiffio.h>
#endif

#define FALSE 0
#define TRUE (!FALSE)

#if defined(HAVE_SPANDSP_H)
#include <spandsp.h>
#endif
#include "unicall.h"

struct
{
    pthread_t thread;
    int chan;
    int sig_fd;
    int fd;
    uc_call_t *call;
    uc_crn_t crn;
    uint8_t message[100000];
    int message_len;
    int message_ptr;
    int xxx;
    int cause;

#if defined(HAVE_SPANDSP_H)
    dtmf_rx_state_t dtmf_state;
    char dtmf[101];
    int dtmf_ptr;
#endif
    
    char *tag;
    
    char originating_number[32];
    char destination_number[32];

    int caller_mode;
    
    time_t last_call;
} chan_stuff[120];

void channel_read_file(uc_t *uc, int chan, void *user_data, uint8_t *buf, int len);
int channel_write_file(uc_t *uc, int chan, void *user_data, uint8_t *buf, int max_len);
int channel_error(uc_t *uc, int chan, void *user_data, int cause);
int signaling_error(uc_t *uc, void *user_data, int cause);

void channel_read_file(uc_t *uc, int chan, void *user_data, uint8_t *buf, int len)
{
    int i;
    int xlen;
    int16_t pcm_buf[1024];
    char *s;

    /* This routine is pretty useless for anything but illustration */
    
    for (i = 0;  i < len;  i++)
        pcm_buf[i] = alaw_to_linear(buf[i]);
    /*endfor*/
#if defined(HAVE_SPANDSP_H)
    dtmf_rx(&chan_stuff[chan].dtmf_state, pcm_buf, len);
    xlen = dtmf_get(&chan_stuff[chan].dtmf_state,
                    chan_stuff[chan].dtmf + chan_stuff[chan].dtmf_ptr,
                    100 - chan_stuff[chan].dtmf_ptr);
    if (xlen > 0)
    {
    	s = chan_stuff[chan].dtmf + chan_stuff[chan].dtmf_ptr;
        while (*s)
        {
            if (*s == '#')
            {
                uc_set_channel_read_callback(uc, 0, NULL, NULL);
                uc_set_channel_write_callback(uc, 0, NULL, NULL);
                if (uc_call_control(uc, UC_OP_DROPCALL, chan_stuff[chan].crn, (void *) UC_CAUSE_NORMAL_CLEARING))
            	    printf("A Drop Call failed\n");
                /*endif*/
                break;
            }
            /*endif*/
            s++;
        }
        /*endwhile*/
    	printf("Got '%s'\n", chan_stuff[chan].dtmf);
        chan_stuff[chan].dtmf_ptr += xlen;
    }
    /*endif*/
#endif
}

int channel_write_file(uc_t *uc, int chan, void *user_data, uint8_t *buf, int max_len)
{
    int len;
    int ret;

    chan = (int) user_data;
    /* This routine is pretty useless for anything but illustration */
    if (chan_stuff[chan].message_ptr >= chan_stuff[chan].message_len)
    {
        //close(user_data);
        uc_set_channel_read_callback(uc, 0, NULL, NULL);
        uc_set_channel_write_callback(uc, 0, NULL, NULL);
        if ((ret = uc_call_control(uc, UC_OP_DROPCALL, chan_stuff[chan].crn, (void *) UC_CAUSE_NORMAL_CLEARING)))
            printf("B Drop Call failed - %d\n", ret);
        /*endif*/
        return 0;
    }
    /*endif*/
    len = max_len;
    if (len > chan_stuff[chan].message_len - chan_stuff[chan].message_ptr)
        len = chan_stuff[chan].message_len - chan_stuff[chan].message_ptr;
    /*endif*/
    memcpy(buf, chan_stuff[chan].message + chan_stuff[chan].message_ptr, len);
    chan_stuff[chan].message_ptr += len;
    return len;
}

int channel_error(uc_t *uc, int chan, void *user_data, int cause)
{
    printf("Error %d\n", cause);
    return  0;
}

int signaling_error(uc_t *uc, void *user_data, int cause)
{
    printf("Error %d\n", cause);
    return  0;
}

static void initiate_call(uc_t *uc, int chan, uc_event_t *e)
{
    uc_makecall_t makecall;
    uc_callparms_t *callparms;
    int ret;

    printf ("Initiating call\n");
    if ((callparms = uc_new_callparms(NULL)) == NULL)
        return;
    /*endif*/
    uc_callparm_originating_number(callparms, chan_stuff[chan].originating_number);
    uc_callparm_destination_number(callparms, chan_stuff[chan].destination_number);
    makecall.callparms = callparms;
    makecall.crn = 0;
    if ((ret = uc_call_control(uc, UC_OP_MAKECALL, 0, (void *) &makecall)) != UC_RET_OK)
        fprintf(stderr, "Make Call failed - %d\n", ret);
    /*endif*/
    chan_stuff[chan].crn = makecall.crn;
    free(callparms);
}

static void handle_uc_event(uc_t *uc, void *user_data, uc_event_t *e)
{
    int file;
    time_t now;
    int chan;

    chan = (int) user_data;
    switch (e->e)
    {
    case UC_EVENT_DEVICEFAIL:
        break;
    case UC_EVENT_PROTOCOLFAIL:
        printf("%s -- Protocol failure on channel %d, cause (%d) %s\n", chan_stuff[chan].tag, e->gen.channel, e->gen.data, uc_error_message(uc, e->gen.data));
        fprintf(stderr, "%s -- Protocol failure on channel %d, cause (%d) %s\n", chan_stuff[chan].tag, e->gen.channel, e->gen.data, uc_error_message(uc, e->gen.data));
        break;
    case UC_EVENT_SIGCHANSTATUS:
        printf("%s -- Signalling channel status - %s\n", chan_stuff[chan].tag, e->sigchanstatus.ok  ?  "Up"  :  "Down");
        fprintf(stderr, "%s -- Signalling channel status - %s\n", chan_stuff[chan].tag, e->sigchanstatus.ok  ?  "Up"  :  "Down");
        break;
    case UC_EVENT_ALARM:
        printf("%s -- %s - 0x%X/0x%X\n", chan_stuff[chan].tag, uc_event2str(e->e), e->alarm.raised, e->alarm.cleared);
        fprintf(stderr, "%s -- %s - 0x%X\n", chan_stuff[chan].tag, uc_event2str(e->e), e->alarm.raised, e->alarm.cleared);
        break;
    case UC_EVENT_FARBLOCKED:
        printf("%s -- %s! :-(\n", chan_stuff[chan].tag, uc_event2str(e->e));
        fprintf(stderr, "%s -- %s! :-(\n", chan_stuff[chan].tag, uc_event2str(e->e));
        chan_stuff[chan].xxx &= ~1;
        break;
    case UC_EVENT_FARUNBLOCKED:
        printf("%s -- %s! :-)\n", chan_stuff[chan].tag, uc_event2str(e->e));
        fprintf(stderr, "%s -- %s! :-)\n", chan_stuff[chan].tag, uc_event2str(e->e));
        chan_stuff[chan].xxx |= 1;
        if (chan_stuff[chan].xxx == 3)
        {
            if (chan_stuff[chan].caller_mode)
                initiate_call(uc, chan, e);
            /*endif*/
        }
        /*endif*/
        break;
    case UC_EVENT_LOCALBLOCKED:
        printf("%s -- %s! :-(\n", chan_stuff[chan].tag, uc_event2str(e->e));
        fprintf(stderr, "%s -- %s! :-(\n", chan_stuff[chan].tag, uc_event2str(e->e));
        chan_stuff[chan].xxx &= ~2;
        break;
    case UC_EVENT_LOCALUNBLOCKED:
        printf("%s -- %s! :-)\n", chan_stuff[chan].tag, uc_event2str(e->e));
        fprintf(stderr, "%s -- %s! :-)\n", chan_stuff[chan].tag, uc_event2str(e->e));
        chan_stuff[chan].xxx |= 2;
        if (chan_stuff[chan].xxx == 3)
        {
            if (chan_stuff[chan].caller_mode)
                initiate_call(uc, chan, e);
            /*endif*/
        }
        /*endif*/
        break;
    case UC_EVENT_DIALING:
        printf("%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        fprintf(stderr, "%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        break;
    case UC_EVENT_PROCEEDING:
        printf("%s -- Proceeding on channel %d\n", chan_stuff[chan].tag, e->gen.channel);
        break;
    case UC_EVENT_ACCEPTED:
        printf("%s -- Accepted on channel %d\n", chan_stuff[chan].tag, e->gen.channel);
        fprintf(stderr, "%s -- Accepted on channel %d\n", chan_stuff[chan].tag, e->gen.channel);
        if (uc_call_control(uc, UC_OP_ANSWERCALL, e->offered.crn, NULL))
            fprintf(stderr, "Answer Call failed\n");
        /*endif*/
        break;
    case UC_EVENT_DETECTED:
        printf("%s -- %s on channel %d, CRN %d\n",
               chan_stuff[chan].tag, uc_event2str(e->e),
               e->gen.channel, e->gen.crn);
        fprintf(stderr, "%s -- %s on channel %d, CRN %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel, e->gen.crn);
        chan_stuff[chan].crn = e->gen.crn;
        break;
    case UC_EVENT_MOREDIGITS:
        printf("%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        fprintf(stderr, "%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        break;
    case UC_EVENT_ALERTING:
        printf("%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        fprintf(stderr, "%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        /* This is just a notification of call progress. We need take no action at this point. */
        break;
    case UC_EVENT_FARDISCONNECTED:
        printf("%s -- %s on channel %d\n",
               chan_stuff[chan].tag,
               uc_event2str(e->e),
               e->fardisconnected.channel);
        fprintf(stderr,
                "%s -- %s on channel %d\n",
                chan_stuff[chan].tag,
                uc_event2str(e->e),
                e->fardisconnected.channel);
        /* Kill any outstanding audio processing */
        uc_set_channel_read_callback(uc, 0, NULL, NULL);
        uc_set_channel_write_callback(uc, 0, NULL, NULL);
        if (uc_call_control(uc, UC_OP_DROPCALL, e->fardisconnected.crn, (void *) UC_CAUSE_NORMAL_CLEARING))
            fprintf(stderr, "C Drop Call failed\n");
        /*endif*/
        break;
    case UC_EVENT_DROPCALL:
        printf("%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        fprintf(stderr, "%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        if (uc_call_control(uc, UC_OP_RELEASECALL, e->gen.crn, NULL))
            fprintf(stderr, "Release Call failed\n");
        /*endif*/
        break;
    case UC_EVENT_RELEASECALL:
        printf("%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        fprintf(stderr, "%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        if (chan_stuff[chan].caller_mode)
            initiate_call(uc, chan, e);
        /*endif*/
        break;
    case UC_EVENT_OFFERED:
        printf("%s -- %s on channel %d, CRN %d (ANI: %s, DNIS: %s)\n",
               chan_stuff[chan].tag,
               uc_event2str(e->e),
               e->offered.channel,
               e->offered.crn,
               e->offered.parms.originating_number,
               e->offered.parms.destination_number);
        fprintf(stderr,
                "%s -- %s on channel %d, CRN %d (ANI: %s, DNIS: %s)\n",
                chan_stuff[chan].tag,
                uc_event2str(e->e),
                e->offered.channel,
                e->offered.crn,
                e->offered.parms.originating_number,
                e->offered.parms.destination_number);
        if (!chan_stuff[chan].caller_mode)
        {
            time(&now);
            switch (chan_stuff[chan].cause)
            {
            case 0:
                fprintf(stderr, "%s -- +++ ACCEPT (%d)\n", chan_stuff[chan].tag, now - chan_stuff[chan].last_call);
                if (uc_call_control(uc, UC_OP_ACCEPTCALL, e->offered.crn, NULL))
                    fprintf(stderr, "Accept Call failed\n");
                /*endif*/
                break;
            case 1:
                fprintf(stderr, "%s -- +++ ANSWER (%d)\n", chan_stuff[chan].tag, now - chan_stuff[chan].last_call);
                if (uc_call_control(uc, UC_OP_ANSWERCALL, e->offered.crn, NULL))
                    fprintf(stderr, "Answer Call failed\n");
                /*endif*/
                break;
            case 2:
                fprintf(stderr, "%s -- +++ BUSY (%d)\n", chan_stuff[chan].tag, now - chan_stuff[chan].last_call);
                if (uc_call_control(uc, UC_OP_DROPCALL, e->offered.crn, (void *) UC_CAUSE_USER_BUSY))
                    fprintf(stderr, "E Drop Call failed\n");
                /*endif*/
                break;
            case 3:
                fprintf(stderr, "%s -- +++ UNASSIGNED (%d)\n", chan_stuff[chan].tag, now - chan_stuff[chan].last_call);
                if (uc_call_control(uc, UC_OP_DROPCALL, e->offered.crn, (void *) UC_CAUSE_UNASSIGNED_NUMBER))
                    fprintf(stderr, "F Drop Call failed\n");
                /*endif*/
                break;
            case 4:
                fprintf(stderr, "%s -- +++ CONGESTED (%d)\n", chan_stuff[chan].tag, now - chan_stuff[chan].last_call);
                if (uc_call_control(uc, UC_OP_DROPCALL, e->offered.crn, (void *) UC_CAUSE_NETWORK_CONGESTION))
                    fprintf(stderr, "G Drop Call failed\n");
                /*endif*/
                break;
            case 5:
                fprintf(stderr, "%s -- +++ OUT OF SERVICE (%d)\n", chan_stuff[chan].tag, now - chan_stuff[chan].last_call);
                if (uc_call_control(uc, UC_OP_DROPCALL, e->offered.crn, (void *) UC_CAUSE_DEST_OUT_OF_ORDER))
                    fprintf(stderr, "H Drop Call failed\n");
                /*endif*/
                break;
            }
            /*endswitch*/
            chan_stuff[chan].last_call = now;
            if (++chan_stuff[chan].cause > 5)
                chan_stuff[chan].cause = 0;
            /*endif*/
        }
        /*endif*/
        break;
    case UC_EVENT_ANSWERED:
        printf("%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        fprintf(stderr, "%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        uc_set_channel_read_callback(uc, 0, channel_read_file, (void *) chan);
        file = open("audio", O_RDONLY);
        if (file < 0)
        {
            /* No audio file */
            printf("%s -- Outgoing audio file not found\n", chan_stuff[chan].tag);
            fprintf(stderr, "%s -- Outgoing audio file not found\n", chan_stuff[chan].tag);
            chan_stuff[chan].message_len = 0;
            chan_stuff[chan].message_ptr = 0;
        }
        else
        {
            chan_stuff[chan].message_len = read(file, chan_stuff[chan].message, 100000);
            chan_stuff[chan].message_ptr = 0;
            printf("%s -- %d bytes of audio to send\n", chan_stuff[chan].tag, chan_stuff[chan].message_len);
            fprintf(stderr, "%s -- %d bytes of audio to send\n", chan_stuff[chan].tag, chan_stuff[chan].message_len);
            close(file);
        }
        uc_set_channel_write_callback(uc, 0, channel_write_file, (void *) chan);
#if defined(HAVE_SPANDSP_H)
        dtmf_rx_init(&chan_stuff[chan].dtmf_state, NULL, NULL);
#endif
        break;
    case UC_EVENT_CONNECTED:
        printf("%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        fprintf(stderr, "%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        if (uc_call_control(uc, UC_OP_DROPCALL, e->offered.crn, (void *) UC_CAUSE_NORMAL_CLEARING))
            printf ("I Drop Call failed\n");
        /*endif*/
        break;
    default:
        printf ("%s -- %s\n", chan_stuff[chan].tag, uc_event2str(e->e));
        fprintf(stderr, "%s --!! Unknown signaling event %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->e);
        break;
    }
    /*endswitch*/
}

static void *run_uc(void *arg)
{
    uc_t *uc;
    uc_event_t *e;
    struct timeval tv = {0,0};
    struct timeval *next;
    fd_set rfds;
    fd_set wfds;
    fd_set efds;
    int res;
    int dfd;
    int chan;
    
    chan = *((int *) arg);

    dfd = chan_stuff[chan].fd;
    uc = uc_new(dfd, dfd, PROTOCOL_CLASS, PROTOCOL_VARIANT, PROTOCOL_END, 1);
    if (uc == NULL)
    {
        fprintf(stderr, "Unable to create instance\n");
        return NULL;
    }
    /*endif*/
    uc_set_signaling_callback(uc, handle_uc_event, (void *) chan);
    uc_set_signaling_error_callback(uc, signaling_error, (void *) chan);
    uc_set_channel_error_callback(uc, 0, channel_error, (void *) chan);
    uc_set_logging(uc, 0x7FFFFFFF, 1, chan_stuff[chan].tag);
    uc_call_control(uc, UC_OP_UNBLOCK, 0, (void *) -1);
    for (;;)
    {
        FD_ZERO(&rfds);
        FD_ZERO(&wfds);
        FD_ZERO(&efds);
        FD_SET(dfd, &rfds);
        FD_SET(dfd, &wfds);
        FD_SET(dfd, &efds);

        if ((next = uc_schedule_next(uc)))
        {
            gettimeofday(&tv, NULL);
            tv.tv_sec = next->tv_sec - tv.tv_sec;
            tv.tv_usec = next->tv_usec - tv.tv_usec;
            if (tv.tv_usec < 0)
            {
                tv.tv_usec += 1000000;
                tv.tv_sec -= 1;
            }
            /*endif*/
            if (tv.tv_sec < 0)
            {
                tv.tv_sec = 0;
                tv.tv_usec = 0;
            }
            /*endif*/
        }
        /*endif*/
        res = select(dfd + 1, &rfds, NULL, &efds, next  ?  &tv  :  NULL);
        if (res >= 0)
        {
            if (res == 0)
                uc_schedule_run(uc);
            /*endif*/
            while ((e = uc_check_event(uc)))
            {
                printf("Non-callback signaling event\n");
                handle_uc_event(uc, (void *) chan, e);
            }
            /*endwhile*/
        }
        else if (errno != EINTR)
        {
            fprintf(stderr, "Error (%d) on select: %s\n", errno, strerror(errno));
        }
        /*endif*/
    }
    /*endfor*/
    return NULL;
}

int main(int argc, char *argv[]) 
{
    pthread_attr_t attr;
    struct zt_bufferinfo b;
    struct zt_gains g;
    struct zt_spaninfo si;
    int chan;
    int chanx;
    char dev_name[20];
    int j;
    int x;
    ZT_PARAMS p;

    uc_start();    
    pthread_attr_init(&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    if (argc < 1)
    {
        fprintf(stderr, "Usage: testcall [call]\n");
        exit(1);
    }
    /*endif*/
    for (chan = 0;  chan < NUM_TRUNKS;  chan++)
    {
        chan_stuff[chan].sig_fd = open("/dev/zap/channel", O_RDWR | O_NONBLOCK);
        if (chan_stuff[chan].sig_fd < 0)
        {
            fprintf(stderr, "Failed to open channel: %s\n", strerror(errno));
            exit(1);
        }
        /*endif*/
        chan_stuff[chan].fd = chan_stuff[chan].sig_fd;
        
        /* Allow for the missing channel at TS16 */
        if (chan < 15)
            chanx = chan + FIRST_TRUNK_CIRCUIT;
        else if (chan < 45)
            chanx = chan + FIRST_TRUNK_CIRCUIT + 1;
        else
            chanx = chan + FIRST_TRUNK_CIRCUIT + 2;
        /*endif*/
        if (ioctl(chan_stuff[chan].fd, ZT_SPECIFY, &chanx))
        {
            fprintf(stderr, "Failed to specify channel %d: %s\n", chanx, strerror(errno));
            exit(1);
        }
        /*endif*/
        if (ioctl(chan_stuff[chan].fd, ZT_GET_BUFINFO, &b))
        {
            fprintf(stderr, "Unable to get buffer info on channel %d: %s\n", chanx, strerror(errno));
            exit(1);
        }
        /*endif*/
        printf ("%d %d %d %d %d %d\n",
                b.rxbufpolicy,
                b.txbufpolicy,
                b.numbufs,
                b.bufsize,
                b.readbufs,
                b.writebufs);
        b.rxbufpolicy = ZT_POLICY_IMMEDIATE;
        b.txbufpolicy = ZT_POLICY_IMMEDIATE;
        b.numbufs = 4;
        b.bufsize = 160;
        if (ioctl(chan_stuff[chan].fd, ZT_SET_BUFINFO, &b))
        {
            fprintf(stderr, "Unable to set buffer info on channel %d: %s\n", chanx, strerror(errno));
            exit(1);
        }
        /*endif*/
        if (ioctl(chan_stuff[chan].fd, ZT_GET_BUFINFO, &b))
        {
            fprintf(stderr, "Unable to get buffer info on channel %d: %s\n", chanx, strerror(errno));
            exit(1);
        }
        /*endif*/
        printf("%d %d %d %d %d %d\n",
               b.rxbufpolicy,
               b.txbufpolicy,
    	       b.numbufs,
               b.bufsize,
               b.readbufs,
               b.writebufs);
        memset(&p, 0, sizeof(p));
        if (ioctl(chan_stuff[chan].fd, ZT_GET_PARAMS, &p))
        {
            fprintf(stderr, "Unable to get params on channel %d: %s\n", chanx, strerror(errno));
            exit(1);
        }
        /*endif*/
        memset(&si, 0, sizeof(si));
        if (ioctl(chan_stuff[chan].fd, ZT_SPANSTAT, &si))
        {
            fprintf(stderr, "Unable to get spanstat on channel %d: %s\n", chanx, strerror(errno));
            exit(1);
        }
        /*endif*/
        x = 160;
        if (ioctl(chan_stuff[chan].fd, ZT_SET_BLOCKSIZE, &x))
        {
            fprintf(stderr, "Unable to set block size on channel %d: %s\n", chanx, strerror(errno));
            exit(1);
        }
        /*endif*/
#if 0
        x = ZT_LAW_ALAW;
        if (ioctl(chan_stuff[chan].fd, ZT_SETLAW, &x))
        {
            fprintf(stderr, "Unable to set law on channel %d: %s\n", chanx, strerror(errno));
            exit(1);
        }
        /*endif*/
        x = 1;
        if (ioctl(chan_stuff[chan].fd, ZT_SETLINEAR, &x))
        {
            fprintf(stderr, "Unable to set linear on channel %d: %s\n", chanx, strerror(errno));
            exit(1);
        }
        /*endif*/
#endif
        g.chan = 0;
        for (j = 0;  j < 256;  j++)
        {
            g.rxgain[j] = j;
            g.txgain[j] = j;
        }
        if (ioctl(chan_stuff[chan].fd, ZT_SETGAINS, &g))
        {
            fprintf(stderr, "Unable to set gains on channel %d: %s\n", chanx, strerror(errno));
            exit(1);
        }
        /*endif*/
        if (chan >= FIRST_OG_TRUNK  &&  chan < FIRST_OG_TRUNK + NUM_OG_TRUNKS)
            chan_stuff[chan].caller_mode = TRUE;
        else
            chan_stuff[chan].caller_mode = FALSE;
        /*endif*/
        chan_stuff[chan].chan = chan;
        sprintf(dev_name, "Chan %3d:", chanx);
        chan_stuff[chan].tag = strdup(dev_name);
        sprintf(chan_stuff[chan].originating_number, "%d", BASE_ANI + chan);
        sprintf(chan_stuff[chan].destination_number, "%d", BASE_DNIS + chan);
        printf("Thread for channel %d\n", chan);
        if (pthread_create(&chan_stuff[chan].thread, &attr, run_uc, &chan_stuff[chan].chan))
            exit(2);
        /*endif*/
    }
    /*endfor*/
    for (;;)
    {
        sleep(5);
        printf("Main thread\n");
    }
    /*endfor*/
    exit(0);

    return 0;
}
