/*
 * libt31 - A Unicall protocol module which emulates a T.31 compatible modem
 *
 * t31.c
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2004 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: t31.c,v 1.7 2005/01/28 23:29:46 steveu Exp $
 */

/*! \file libt31.h */

#define _ISOC9X_SOURCE        1
#define _ISOC99_SOURCE        1
#define _GNU_SOURCE

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <limits.h>
#include <stdio.h>
#include <stdarg.h>
#include <math.h>
#include <fcntl.h>
#include <stdint.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <time.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/ioctl.h>

#include <tiffio.h>
#include <spandsp.h>
#include <spandsp/t30_fcf.h>

#include <unicall.h>
#include <unicall/hashtable.h>
#include <unicall/unicall-private.h>

#include "libt31.h"
#include "t31.h"

#define FALSE 0
#define TRUE (!FALSE)

#define NUM_VOICE_CHANNELS_PER_CONTEXT  1

static uc_check_event_t check_event;
static uc_new_t create_new;
static uc_call_control_t call_control;
static uc_error_message_t error_message;
static uc_set_api_codec_t set_api_codec;
static uc_get_device_handles_t get_device_handles;
static uc_channel_write_t channel_write;
static uc_channel_flush_t channel_flush;

uc_protocol_t protocol_descriptor =
{
    "t31",
    "T.31 modem",
    "",
    "",

    check_event,
    create_new,
    call_control,
    error_message,
    set_api_codec,
    get_device_handles,
    channel_write,
    channel_flush
};

enum
{
    ACTIVE_NONE,
    ACTIVE_SPEECH
};

static int set_cas_abcd_signal(uc_t *uc, int ch, uint8_t abcd);
static int set_mf_signal(uc_t *uc, int ch, int forward_signal);

static void select_active_rxtx(uc_t *uc, int on);
static void far_unblocking_expired(uc_t *uc, void *data);
static void start_far_disconnected(uc_t *uc, uc_call_t *call);
static void start_connected(uc_t *uc, uc_call_t *call);
static void start_offered(uc_t *uc, uc_call_t *call);
static void start_detected(uc_t *uc, int ch);

static int variant_element(const char **variant, int which)
{
    int i;
    const char *s;
    const char *t;

    t = *variant - 1;
    for (i = -1;  i < which;  i++)
    {
        s = t + 1;
        if ((t = strchr(t + 1, ',')) == NULL)
        {
            t = s + strlen(s);
            break;
        }
        /*endif*/
    }
    /*endfor*/
    if (i >= which - 1)
    {
        *variant = s;
        return  t - s;
    }
    /*endif*/
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int protocol_variant(const char *variant)
{
    int i;
    int inlen;
    int len;
    char const *s;
    char const *t;

    inlen = variant_element(&variant, 0);
    s = protocol_descriptor.variants;
    for (i = 1;  *s;  i++)
    {
        if ((t = strchr(s, ',')))
            len = t - s;
        else
            len = strlen(s);
        /*endif*/
        if (len == inlen  &&  strncmp(variant, s, len) == 0)
            return  i;
        /*endif*/
        s += len;
        if (*s == ',')
            s++;
        /*endif*/
    }
    /*endfor*/
    return  -1;
}
/*- End of function --------------------------------------------------------*/

static void select_active_rxtx(uc_t *uc, int which)
{
    int ch;
    
    ch = 0;
    uc->chan[ch].active_rx =
    uc->chan[ch].active_tx = which;
}
/*- End of function --------------------------------------------------------*/

static void protocol_error_recovery(uc_t *uc, int ch, int cause)
{
    t31_signaling_state_t *t31;
    uc_event_t ev;

    t31 = (t31_signaling_state_t *) uc->signaling;
    uc_log(uc,
           UC_LOG_PROTOCOL_ERROR,
           "T.31 prot. err. [%d/%d/%8X/%3d] cause %d\n",
           uc->chan[ch].active_rx,
           uc->chan[ch].active_tx,
           uc->chan[ch].state,
           t31->call_state,
           cause);
    /* If we are blocked, a protocol problem isn't a good reason to unblock.
       However, for other starting conditions, going to the idle state seems OK. */
    if (uc->chan[ch].state != UC_STATE_BLOCKED)
        uc->chan[ch].state = UC_STATE_IDLE;
    t31->call_state = 0;
    
    /* TODO: We need to clear ourselves back thoroughly to a default condition, then
             try to pick up and carry on with fresh calls. The protocol error may have
             been a hiccup, rather than a hard failure. */

    ev.e = UC_EVENT_PROTOCOLFAIL;
    ev.gen.channel = ch;
    ev.gen.data = cause;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
}
/*- End of function --------------------------------------------------------*/

static int modem_call_control(t31_state_t *s, void *user_data, const char *num)
{
    uc_t *uc;
    t31_signaling_state_t *t31;
    int ch;
    
    uc = (uc_t *) user_data;
    t31 = (t31_signaling_state_t *) uc->signaling;
    ch = 0;
    uc_log(uc, UC_LOG_FLOW, "T.31 modem call control\n");
    /* The modem needs very simple call control. When it calls out it is with
       complete destination and (sometimes) originating numbers. Later it hangs up.
       When it answers a call, it just answers and later hangs up.
       A non-blank num means call that number.
       A blank num means answer.
       If num is NULL it means hangup. */ 
    if (num)
    {
        if (num[0])
        {
            /* Dialing */
            if (uc->chan[ch].state != UC_STATE_IDLE)
                return -1;
            /*endif*/
            start_detected(uc, 0);
            t31->call->chan = 0;
            strcpy(t31->call->callparms.originating_number, "");
            strcpy(t31->call->callparms.destination_number, num);
            start_offered(uc, t31->call);
        }
        else
        {
            /* Answering */
            if (uc->chan[ch].state != UC_STATE_ALERTING)
                return -1;
            /*endif*/
            select_active_rxtx(uc, ACTIVE_SPEECH);
            start_connected(uc, t31->call);
        }
        /*endif*/
    }
    else
    {
        /* Hang up */
        if (uc->chan[ch].state == UC_STATE_IDLE
            ||
            uc->chan[ch].state == UC_STATE_BLOCKED)
        {
            return -1;
        }
        /*endif*/
        select_active_rxtx(uc, ACTIVE_NONE);
        start_far_disconnected(uc, t31->call);
    }
    /*endif*/
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int at_tx_handler(t31_state_t *s, void *user_data, const uint8_t *buf, int len)
{
    t31_signaling_state_t *t31;
    uc_t *uc;
    
    uc = (uc_t *) user_data;
    t31 = (t31_signaling_state_t *) uc->signaling;
    return write(t31->pts_handle, buf, len);
}
/*- End of function --------------------------------------------------------*/

static void start_detected(uc_t *uc, int ch)
{
    t31_signaling_state_t *t31;
    uc_event_t ev;
    uc_call_t *call;

    t31 = (t31_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Detected\n");

    if ((call = uc_createcall(uc, 0)) == NULL)
        return;
    /*endif*/
    t31->call = call;

    /* Set a default for the final disconnection. */
    call->disconnect_reason = UC_CAUSE_UNSPECIFIED_CAUSE;
    call->incoming_call = TRUE;
    t31->incoming_call = TRUE;
    t31->cleared_fwd = FALSE;

    if (ch >= 0)
        uc->chan[ch].state = UC_STATE_DETECTED;
    /*endif*/
    call->state = UC_STATE_DETECTED;
    call->chan = ch;
    /* We need to choose the codec used by the card. */
    call->callparms.userinfo_layer1_protocol = UC_LAYER_1_ALAW;

    ev.e = UC_EVENT_DETECTED;
    ev.gen.channel = call->chan;
    ev.gen.crn = call->crn;
    ev.gen.call = call;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
}
/*- End of function --------------------------------------------------------*/

static void start_call_disconnected(uc_t *uc, uc_call_t *call)
{
    t31_signaling_state_t *t31;
    uc_event_t ev;

    t31 = (t31_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Call disconnected - state 0x%X\n", call->state);
    
    /* TODO: We probably need to take action, like stopping tone detectors and generators,
             at this point. */

    if (call->chan >= 0)
        uc->chan[call->chan].state = UC_STATE_CALL_DISCONNECTED;
    /*endif*/
    call->state = UC_STATE_CALL_DISCONNECTED;
    
    ev.fardisconnected.e = UC_EVENT_DROPCALL;
    ev.fardisconnected.channel = call->chan;
    ev.fardisconnected.cause = call->disconnect_reason;
    ev.fardisconnected.crn = call->crn;
    ev.fardisconnected.call = call;
    uc_report_event(uc, &ev, sizeof(ev));
}
/*- End of function --------------------------------------------------------*/

static void start_far_disconnected(uc_t *uc, uc_call_t *call)
{
    t31_signaling_state_t *t31;
    uc_event_t ev;

    t31 = (t31_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Far end disconnected - state 0x%X\n", call->state);

    if (call->chan >= 0)
        uc->chan[call->chan].state = UC_STATE_FAR_DISCONNECTED;
    /*endif*/
    call->state = UC_STATE_FAR_DISCONNECTED;

    ev.fardisconnected.e = UC_EVENT_FARDISCONNECTED;
    ev.fardisconnected.channel = call->chan;
    ev.fardisconnected.cause = call->disconnect_reason;
    ev.fardisconnected.crn = call->crn;
    ev.fardisconnected.call = call;
    uc_report_event(uc, &ev, sizeof(ev));
}
/*- End of function --------------------------------------------------------*/

static void start_idle(uc_t *uc, uc_call_t *call)
{
    t31_signaling_state_t *t31;
    uc_event_t ev;

    t31 = (t31_signaling_state_t *) uc->signaling;
    /* N.B. This is really only for starting the idle state from a call. From
       initialisation, unblocking, or error recovery this is not the right
       choice. */

    /* TODO: Make sure no junk is left behind. */
    t31->call_state = 0;

    /* TODO: We probably need to take action, like stopping tone detectors and generators,
             at this point. */

    if (call->chan >= 0)
        uc->chan[call->chan].state = UC_STATE_IDLE;
    /*endif*/
    call->state = UC_STATE_IDLE;

    ev.e = UC_EVENT_RELEASECALL;
    ev.gen.channel = call->chan;
    ev.gen.crn = call->crn;
    ev.gen.call = call;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));

    /* Although we destroy the call here, the event sent just above may not
       have been delivered. It is important, therefore, that the application
       only treat the call as a reference tag, and does not try to look at
       its contents. */
    uc_destroycall(uc, call->crn);
    t31->call = NULL;
}
/*- End of function --------------------------------------------------------*/

static void start_offered(uc_t *uc, uc_call_t *call)
{
    t31_signaling_state_t *t31;
    uc_event_t ev;
    
    t31 = (t31_signaling_state_t *) uc->signaling;
    if (call->chan >= 0)
        uc->chan[call->chan].state = UC_STATE_OFFERED;
    /*endif*/
    call->state = UC_STATE_OFFERED;

    /* Set a default for the final disconnection. */
    call->disconnect_reason = UC_CAUSE_NORMAL_CLEARING;

    memset (&ev.offered, '\0', sizeof(ev.offered));
    ev.offered.e = UC_EVENT_OFFERED;
    ev.offered.channel = call->chan;
    strcpy (ev.offered.parms.originating_number, call->callparms.originating_number);
    strcpy (ev.offered.parms.destination_number, call->callparms.destination_number);
    ev.offered.parms.calling_party_category = call->callparms.calling_party_category;
    ev.offered.parms.call_type = 0;
    ev.offered.parms.userinfo_layer1_protocol = call->callparms.userinfo_layer1_protocol;
    ev.offered.crn = call->crn;
    ev.offered.call = call;
    uc_report_event(uc, &ev, sizeof(ev));
}
/*- End of function --------------------------------------------------------*/

static void start_proceeding(uc_t *uc, uc_call_t *call)
{
    uc_event_t ev;

    if (call->chan >= 0)
        uc->chan[call->chan].state = UC_STATE_PROCEEDING;
    /*endif*/
    call->state = UC_STATE_PROCEEDING;

    ev.e = UC_EVENT_PROCEEDING;
    ev.gen.channel = call->chan;
    ev.gen.crn = call->crn;
    ev.gen.call = call;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
}
/*- End of function --------------------------------------------------------*/

static void start_dialing(uc_t *uc, uc_call_t *call)
{
    uc_event_t ev;

    if (call->chan >= 0)
        uc->chan[call->chan].state = UC_STATE_PROCEEDING;
    /*endif*/
    call->state = UC_STATE_PROCEEDING;

    ev.e = UC_EVENT_DIALING;
    ev.gen.channel = call->chan;
    ev.gen.crn = call->crn;
    ev.gen.call = call;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
}
/*- End of function --------------------------------------------------------*/

static void start_alerting(uc_t *uc, uc_call_t *call)
{
    uc_event_t ev;

    if (call->chan >= 0)
        uc->chan[call->chan].state = UC_STATE_ALERTING;
    /*endif*/
    call->state = UC_STATE_ALERTING;

    ev.e = UC_EVENT_ALERTING;
    ev.gen.channel = call->chan;
    ev.gen.crn = call->crn;
    ev.gen.call = call;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
}
/*- End of function --------------------------------------------------------*/

static void start_accepted(uc_t *uc, uc_call_t *call)
{
    uc_event_t ev;

    if (call->chan >= 0)
        uc->chan[call->chan].state = UC_STATE_ACCEPTED;
    /*endif*/
    call->state = UC_STATE_ACCEPTED;
    
    ev.e = UC_EVENT_ACCEPTED;
    ev.gen.channel = call->chan;
    ev.gen.crn = call->crn;
    ev.gen.call = call;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
}
/*- End of function --------------------------------------------------------*/

static void start_connected(uc_t *uc, uc_call_t *call)
{
    t31_signaling_state_t *t31;
    uc_event_t ev;

    t31 = (t31_signaling_state_t *) uc->signaling;

    /* Set a default for the final disconnection. */
    call->disconnect_reason = UC_CAUSE_NORMAL_CLEARING;

    /* TODO: We might need to take action, like opening the speech path, and clearing
             answer timeouts, at this point. */

    if (call->chan >= 0)
        uc->chan[call->chan].state = UC_STATE_CONNECTED;
    /*endif*/
    call->state = UC_STATE_CONNECTED;

    /* The event type depends on the direction of the call. Using different event
       types keeps the incoming and outgoing call event sequences distinct. */
    if (call->incoming_call)
    {
        t31_call_event(&t31->t31_state, T31_CALL_EVENT_CONNECTED);
        ev.e = UC_EVENT_ANSWERED;
    }
    else
    {
        t31_call_event(&t31->t31_state, T31_CALL_EVENT_ANSWERED);
        ev.e = UC_EVENT_CONNECTED;
    }
    /*endif*/
    ev.gen.channel = call->chan;
    ev.gen.crn = call->crn;
    ev.gen.call = call;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
}
/*- End of function --------------------------------------------------------*/

static void far_unblocking_expired(uc_t *uc, void *data)
{
    t31_signaling_state_t *t31;
    uc_event_t ev;
    int ch;
    
    ch = (int) data;
    t31 = (t31_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "far_unblocking_expired\n");
    t31->far_unblocking_timer = -1;

    t31->far_blocked = FALSE;

    if (uc->chan[ch].state == UC_STATE_BLOCKED)
        uc->chan[ch].state = UC_STATE_IDLE;
    /*endif*/

    ev.e = UC_EVENT_FARUNBLOCKED;
    ev.gen.channel = ch;
    ev.gen.crn = 0;
    ev.gen.call = NULL;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
}
/*- End of function --------------------------------------------------------*/

static void wait_for_answer_expired(uc_t *uc, void *data)
{
    t31_signaling_state_t *t31;
    uc_call_t *call;
    
    call = (uc_call_t *) data;    
    t31 = (t31_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "Timed out waiting for answer\n");
    /* Treat this as though the other end cleared the call. */
    call->disconnect_reason = UC_CAUSE_NO_ANSWER_FROM_USER;
    select_active_rxtx(uc, ACTIVE_NONE);
    start_far_disconnected(uc, call);
}
/*- End of function --------------------------------------------------------*/

static uc_event_t *check_event(uc_t *uc)
{
    char buf[1024];
    t31_signaling_state_t *t31;
    uc_event_t ev;
    int ch;
    int read_len;
    
    ch = 0;
    t31 = (t31_signaling_state_t *) uc->signaling;
    if ((read_len = read(t31->pts_handle, buf, 1024)) > 0)
    {
{
int i;
printf("Read %d bytes ", read_len);
for (i = 0;  i < read_len;  i++)
    printf("%02X ", buf[i] & 0xFF);
printf("\n");
}
        t31_at_rx(&t31->t31_state, buf, read_len);
    }
    /*endif*/
    return NULL;
}
/*- End of function --------------------------------------------------------*/

static int make_call(uc_t *uc, uc_makecall_t *makecall)
{
    t31_signaling_state_t *t31;
    uc_event_t ev;
    int ch;
    uc_call_t *call;
    const uc_callparms_t *callparms;

    ch = 0;
    t31 = (t31_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "T.31 make call\n");

    if (uc->chan[ch].state != UC_STATE_IDLE)
        return  UC_RET_BAD_STATE;
    /*endif*/
    if (!t31->outgoing_calls_allowed)
        return  UC_RET_BAD_STATE;
    /*endif*/

    if ((call = uc_createcall(uc, 0)) == NULL)
        return UC_RET_MEMORY_ERROR;
    /*endif*/
    makecall->crn = call->crn;
    callparms = makecall->callparms;

    t31->call = call;

    /* Gather information from callparms */
    strcpy(call->callparms.destination_number, callparms->destination_number);
    strcpy(call->callparms.originating_number, callparms->originating_number);
    if (callparms->calling_party_category == 0)
        call->callparms.calling_party_category = '3';
    else
        call->callparms.calling_party_category = callparms->calling_party_category;
    /*endif*/
    t31->call_type = 0;

    /* Assign a default cause for the eventual disconnection of this call. */
    call->disconnect_reason = UC_CAUSE_UNSPECIFIED_CAUSE;
    call->incoming_call = FALSE;
    t31->incoming_call = FALSE;
    t31->cleared_fwd = FALSE;
    
    call->chan = ch;
    /* The call has just begun, so we must be at the information transfer stage */
    start_dialing(uc, call);
    /* Alert the modem */
    t31_call_event(&t31->t31_state, T31_CALL_EVENT_ALERTING);
    start_alerting(uc, call);
    
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int send_more_info(uc_t *uc, uc_call_t *call, uc_sendmoreinfo_t *info)
{
    return  UC_RET_UNSUPPORTED;
}
/*- End of function --------------------------------------------------------*/

static int call_ack(uc_t *uc, uc_call_t *call, uc_callack_t *callack)
{
    return  UC_RET_UNSUPPORTED;
}
/*- End of function --------------------------------------------------------*/

static int accept_call(uc_t *uc, uc_call_t *call)
{
    t31_signaling_state_t *t31;
    int ch;

    ch = 0;
    t31 = (t31_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "T.31 accept call()\n");
    if (call == NULL)
        return  UC_RET_BAD_STATE;
    /*endif*/

    if (uc->chan[ch].state != UC_STATE_OFFERED)
        return  UC_RET_BAD_STATE;
    /*endif*/

    start_accepted(uc, call);

    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int answer_call(uc_t *uc, uc_call_t *call)
{
    t31_signaling_state_t *t31;

    t31 = (t31_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "T.31 answer call\n");
    if (call == NULL)
        return  UC_RET_BAD_STATE;
    /*endif*/
    switch (call->state)
    {
    case UC_STATE_OFFERED:
    case UC_STATE_ACCEPTED:
        /* Go directly to the answered state. */
        select_active_rxtx(uc, ACTIVE_SPEECH);
        start_connected(uc, call);
        break;
    default:
        return  UC_RET_BAD_STATE;
    }
    /*endif*/
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int drop_call(uc_t *uc, uc_call_t *call, int cause)
{
    t31_signaling_state_t *t31;
    int ch;
    
    ch = 0;
    t31 = (t31_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "T.31 drop call(cause=%d)\n", cause);
    if (call == NULL)
        return  UC_RET_BAD_STATE;
    /*endif*/
    
    //TODO: Check for further state conflicts
    if (call->state == UC_STATE_NULL
        ||
        call->state == UC_STATE_IDLE
        ||
        call->state == UC_STATE_CALL_DISCONNECTED)
    {
        return  UC_RET_BAD_STATE;
    }
    /*endif*/
    
    select_active_rxtx(uc, ACTIVE_NONE);
    if (t31->cleared_fwd)
    {
        /* Go to the call disconnected state, but do nothing else.
           release call will initiate the final release. */
        start_call_disconnected(uc, t31->call);
    }
    else
    {
        t31->cleared_fwd = TRUE;
        /* TODO: tell the modem the call dropped. */
    }
    /*endif*/
    return UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int release_call(uc_t *uc, uc_call_t *call)
{
    t31_signaling_state_t *t31;
    int ch;
    
    ch = 0;
    t31 = (t31_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "T.31 release call()\n");
    if (call == NULL)
        return  UC_RET_BAD_STATE;
    /*endif*/
    if (call->state != UC_STATE_CALL_DISCONNECTED)
        return  UC_RET_BAD_STATE;
    /*endif*/
    if (call->incoming_call)
    {
        select_active_rxtx(uc, ACTIVE_NONE);
        start_idle(uc, call);
    }
    else
    {
        /* There is nothing to do here. The call has completely cleared already. */
        start_idle(uc, call);
    }
    /*endif*/
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int unblock(uc_t *uc, int ch)
{
    t31_signaling_state_t *t31;
    uc_event_t ev;
  
    /* ch = -1 means all voice channels */
    if (ch >= uc->num_channels)
        return UC_RET_BAD_PARAMETER;
    /*endif*/
    t31 = (t31_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "T.31 unblock\n");
    
    /* TODO: unblock the channels */

    if (uc->chan[0].state == UC_STATE_BLOCKED)
        uc->chan[0].state = UC_STATE_IDLE;
    /*endif*/
    t31->local_blocked = FALSE;

    ev.e = UC_EVENT_LOCALUNBLOCKED;
    ev.gen.channel = ch;
    ev.gen.crn = 0;
    ev.gen.call = NULL;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));

    /* TODO: block until an application has connected. */
    t31->far_unblocking_timer =
        uc_schedule_event(uc, 60000, far_unblocking_expired, (void *) ch);
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int block(uc_t *uc, int ch)
{
    t31_signaling_state_t *t31;
    uc_event_t ev;

    /* ch = -1 means all voice channels */
    if (ch >= uc->num_channels)
        return UC_RET_BAD_PARAMETER;
    /*endif*/
    t31 = (t31_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "T.31 block()\n");

    /* TODO: Block the channels */

    t31->local_blocked = TRUE;

    ev.e = UC_EVENT_LOCALBLOCKED;
    ev.gen.channel = ch;
    ev.gen.crn = 0;
    ev.gen.call = NULL;
    uc_report_event(uc, &ev, sizeof(uc_event_generic_t));
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int call_control(uc_t *uc, int op, uc_call_t *call, void *data)
{
    t31_signaling_state_t *mfcr2;
    int ch;

    ch = 0;
    mfcr2 = (t31_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "T.31 call control(%d)\n", op);
    switch (op)
    {
    case UC_OP_MAKECALL:
        return make_call(uc, (uc_makecall_t *) data);
    case UC_OP_SENDMOREINFO:
        return send_more_info(uc, call, (uc_sendmoreinfo_t *) data);
    case UC_OP_CALLACK:
        return call_ack(uc, call, (uc_callack_t *) data);
    case UC_OP_ACCEPTCALL:
        return accept_call(uc, call);
    case UC_OP_ANSWERCALL:
        return answer_call(uc, call);
    case UC_OP_DROPCALL:
        return drop_call(uc, call, (int) data);
    case UC_OP_RELEASECALL:
        return release_call(uc, call);
    case UC_OP_UNBLOCK:
        return unblock(uc, (int) data);
    case UC_OP_BLOCK:
        return block(uc, (int) data);
    }
    /*endswitch*/
    return  UC_RET_UNSUPPORTED;
}
/*- End of function --------------------------------------------------------*/

uc_t *create_new(int fd, const char *variant, int mode, int outgoing_calls_allowed)
{
    uc_t *uc;
    t31_signaling_state_t *t31;
    uc_chan_t *chans;
    int variation;
    int ch;
    int value;
    int len;
    const char *s;
    char *t;
    
#if 0
    if ((variation = protocol_variant(variant)) < 0)
    {
        uc_error("Protocol variant '%s' unknown\n", variant);
        return  NULL;
    }
    /*endif*/
#endif
    /* The fd we have been given should be the already opened signaling channel
       we need to use. */
    
    /* TODO: Check the signaling channel is OK (a channel of the right type, in the
             correct mode, etc.), and perform any necessary initialisation of it. */

    if ((uc = malloc(sizeof(*uc))) == NULL)
    {
        uc_error("Failed to allocate buffers.\n");
        return  NULL;
    }
    /*endif*/
    memset(uc, '\0', sizeof(*uc));
    uc->protocol_variant = variation;
    uc->protocol_mode = mode;

    /* Allocate the protocol specific control structure */
    if ((t31 = malloc(sizeof(*t31))) == NULL)
    {
        uc_error("Failed to allocate buffers.\n");
        free(uc);
        return  NULL;
    }
    /*endif*/
    uc->signaling = t31;
    memset(t31, '\0', sizeof(*t31));
    t31->fe_handle = fd;
    
    /* Perform any protocol specific initialisation needed in the protocol
       specific control structure. */

    /* TODO: initialise */
    
    /* Allocate the array of per voice channel control structures */
    if ((chans = malloc(sizeof(*chans)*NUM_VOICE_CHANNELS_PER_CONTEXT)) == NULL)
    {
        uc_error("Failed to allocate buffers.\n");
        free(t31);
        free(uc);
        return  NULL;
    }
    /*endif*/
    memset(chans, '\0', sizeof(*chans));
    uc->num_channels = NUM_VOICE_CHANNELS_PER_CONTEXT;
    uc->chan = chans;
    
    for (ch = 0;  ch < NUM_VOICE_CHANNELS_PER_CONTEXT;  ch++)
    {
        uc->chan[ch].handle = fd;
        uc->chan[ch].state = UC_STATE_BLOCKED;
        uc->chan[ch].logging_level = 0;
        uc->chan[ch].native_codec =
        uc->chan[ch].api_codec = UC_CODEC_LINEAR16;
        uc->chan[ch].audio_bufsize = 160;
        uc->chan[ch].active_rx =
        uc->chan[ch].active_tx = ACTIVE_NONE;
    }
    /*endfor*/

    t31->outgoing_calls_allowed = outgoing_calls_allowed;
    if ((t31->pts_handle = open("/dev/ptmx", O_RDWR | O_NONBLOCK)) < 0)
        return NULL;
    /*endif*/
    if (grantpt(t31->pts_handle) < 0)
        return NULL;
    /*endif*/
    if (unlockpt(t31->pts_handle) < 0)
        return NULL;
    /*endif*/
    t31->pts_name = ptsname(t31->pts_handle);
    printf("pts name is '%s'\n", t31->pts_name);
    if (remove("/dev/ttyxxx"))
        printf("symlink removal failed\n");
    if (symlink(t31->pts_name, "/dev/ttyxxx"))
        printf("symlink create failed\n");
    t31->tty_name = ttyname(t31->pts_handle);
    printf("tty name is '%s'\n'", t31->tty_name);

    if (t31_init(&(t31->t31_state), at_tx_handler, (void *) uc, modem_call_control, (void *) uc) < 0)
        return NULL;
    /*endif*/
    hash_InitHashTable(&uc->calls, HASH_ONE_WORD_KEYS);
    return uc;
}
/*- End of function --------------------------------------------------------*/

static char *error_message(uc_t *uc, int code)
{
    switch (code)
    {
    case T31_PROTOCOL_FAIL_GENERAL:
        return "General failure";
    case T31_PROTOCOL_FAIL_DEVICE_IO_ERROR:
        return "Device I/O error";
    }
    /*endswitch*/
    return "Undefined";
}
/*- End of function --------------------------------------------------------*/

int set_api_codec(uc_t *uc, int ch, int codec)
{
    int linear;
    int law;

    //uc_log(uc, UC_LOG_FLOW, "t31_set_api_codec - %d\n", codec);
    switch (codec)
    {
    case UC_CODEC_DEFAULT:
        break;
    case UC_CODEC_LINEAR16:
        break;
    case UC_CODEC_ALAW:
        break;
    case UC_CODEC_ULAW:
        break;
    default:
        return  UC_RET_UNSUPPORTED;
    }
    /*endswitch*/
    uc->chan[ch].api_codec = codec;
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int get_device_handles(uc_t *uc, int ch, int *fe, int *be)
{
    t31_signaling_state_t *t31;

    t31 = (t31_signaling_state_t *) uc->signaling;
    if (fe)
        *fe = t31->pts_handle;
    /*endif*/
    if (be)
        *be = t31->pts_handle;
    /*endif*/
    return UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

static int channel_write(uc_t *uc, int ch, uint8_t *buf, int len)
{
    t31_signaling_state_t *t31;
    int xlen;

    if (uc->chan[ch].active_tx != ACTIVE_SPEECH)
        return len;
    /*endif*/
    if (uc->chan[ch].channel_write_codec)
        len = uc->chan[ch].channel_write_codec(uc, ch, uc->chan[ch].channel_write_codec_user_data, buf, len);
    /*endif*/
    t31 = (t31_signaling_state_t *) uc->signaling;
    xlen = t31_rx(&(t31->t31_state), (int16_t *) buf, len/sizeof(int16_t));
    if (uc->chan[ch].channel_read)
    {
        /* TODO: should not be reusing the caller's buffer */
        xlen = t31_tx(&(t31->t31_state), (int16_t *) buf, len/sizeof(int16_t));
        if (uc->chan[ch].channel_read_codec)
            xlen = uc->chan[ch].channel_read_codec(uc, ch, uc->chan[ch].channel_read_codec_user_data, buf, xlen);
        /*endif*/
        if (xlen*sizeof(int16_t) < len)
            memset(buf + xlen*sizeof(int16_t), 0, len - xlen*sizeof(int16_t));
        /*endif*/
        uc->chan[ch].channel_read(uc, ch, uc->chan[ch].channel_read_user_data, buf, len);
    }
    /*endif*/
    return  xlen;
}
/*- End of function --------------------------------------------------------*/

static int channel_flush(uc_t *uc, int ch, int which)
{
    t31_signaling_state_t *t31;
    int x;

    /* ch = -1 means all voice channels */
    if (ch >= uc->num_channels)
        return UC_RET_BAD_PARAMETER;
    /*endif*/
    t31 = (t31_signaling_state_t *) uc->signaling;
    uc_log(uc, UC_LOG_FLOW, "t31_Flush()\n");
    if ((which & 1))
    {
        /* TODO: implement output flush */
    }
    /*endif*/
    if ((which & 2))
    {
        /* TODO: implement input flush */
    }
    /*endif*/
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
