/*
 * libt31 - A Unicall protocol module which emulates a T.31 compatible modem
 *
 * t31_tests_app.c - Test application for the T.31 channel.
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2004 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: t31_test_app.c,v 1.5 2004/11/05 14:47:47 steveu Exp $
 */

#define	_ISOC9X_SOURCE	1
#define _ISOC99_SOURCE	1

#include <stdint.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <math.h>
#include <assert.h>
#include <termios.h>
#include <tiffio.h>
#include <sys/select.h>
#include <sys/time.h>

#include "spandsp.h"
#include "spandsp/t30_fcf.h"

#define MODEM_MAKER     "www.opencall.org"
#define MODEM_MODEL     "spandsp"

#define DLE 0x10
#define ETX 0x03
#define SUB 0x1A

int expect(int fd, char *s);
int expect_stuffed(int fd, char *s, int len);
int send(int fd, char *t, char *s);
int send_stuffed(int fd, uint8_t *t, int len, char *s);
int hdlc_header(uint8_t frame[], int last, int type);
int general_test(int fd);
int fax_send_test(int fd);
int fax_receive_test(int fd);

void print_string(char *s)
{
    while (*s)
    {
        if (*s == '\r')
            printf("\\r");
        else if (*s == '\n')
            printf("\\n");
        else if (*s < ' ')
            printf("\\x%02X", *s & 0xFF);
        else
            printf("%c", *s);
        /*endif*/
        s++;
    }
    /*endwhile*/
}

int get_timeout(int fd, uint8_t *buf, int max_len, int timeout)
{
    struct timeval tv = {0,0};
    struct timeval *next;
    fd_set rfds;
    fd_set wfds;
    fd_set efds;
    int res;
    int len;
    int total_len;

    total_len = 0;
    for (;;)
    {
        FD_ZERO(&rfds);
        FD_ZERO(&wfds);
        FD_ZERO(&efds);
        FD_SET(fd, &rfds);
        FD_SET(fd, &wfds);
        FD_SET(fd, &efds);

        tv.tv_sec = timeout;
        tv.tv_usec = 0;
        if ((res = select(fd + 1, &rfds, NULL, &efds, &tv)) >= 0)
        {
            if (res == 0)
                return total_len;
            /*endif*/
            len = read(fd, buf, max_len - total_len);
            if (len > 0)
            {
                buf += len;
                total_len += len;
            }
            if (total_len >= max_len)
                return total_len;
        }
        else if (errno != EINTR)
        {
            fprintf(stderr, "Error (%d) on select: %s\n", errno, strerror(errno));
        }
        /*endif*/
    }
    /*endfor*/
    return 0;
}

int expect(int fd, char *s)
{
    int len;
    uint8_t buf[132 + 1];
    char *t;

    len = 0;
    if (s)
    {
        if ((len = get_timeout(fd, buf, strlen(s), 10)) > 0)
        {
            buf[len] = '\0';
#if 0
            printf("Received '");
            print_string(buf);
            printf("' vs '");
            print_string(s);
            printf("' (%s)\n", (strcmp(buf, s) == 0)  ?  "OK"  :  "bad");
#else
            if (strcmp(buf, s) == 0)
            {
                printf("Received (OK)\n");
            }
            else
            {
                printf("Received '");
                print_string(buf);
                printf("' vs '");
                print_string(s);
                printf("' (bad)\n");
            }
#endif
        }
        else
        {
            printf("No receive '");
            print_string(s);
            printf("'\n");
        }
        /*endif*/
    }
    /*endif*/
    return len;
}

int expect_stuffed(int fd, char *s, int max_len)
{
    uint8_t x[1024];
    int i;
    int j;
    int len;
    int dled;
    
    i = 0;
    printf("Receive HDLC ");
    dled = FALSE;
    if ((len = get_timeout(fd, x, 1, 10)) == 1)
    {
        j = 0;
        for (;;)
        {
            if (dled)
            {
                dled = FALSE;
                if (x[j] == ETX)
                    break;
                /*endif*/
                printf("%02X ", x[j] & 0xFF);
                s[i++] = x[j];
            }
            else
            {
                if (x[j] == DLE)
                {
                    dled = TRUE;
                }
                else
                {
                    printf("%02X ", x[j] & 0xFF);
                    s[i++] = x[j];
                }
                /*endif*/
            }
            /*endif*/
            if (++j >= len)
            {
                len = get_timeout(fd, x, 1, 1);
                j = 0;
            }
            /*endif*/
        }
        /*endfor*/
    }
    /*endif*/
    printf("\n");
    return i;
}

int expect_any_stuffed(int fd)
{
    uint8_t x[1024];
    int j;
    int len;
    int dled;
    
    printf("Receive ");
    dled = FALSE;
    if ((len = get_timeout(fd, x, 1, 10)) == 1)
    {
        if (x[0] == DLE)
            dled = TRUE;
        /*endif*/
        for (;;)
        {
            if ((len = get_timeout(fd, x, 1024, 1)) <= 0)
            {
                printf("(timeout) %d bytes\n", len);
                return len;
            }
            for (j = 0;  j < len;  j++)
            {
                if (dled)
                {
                    dled = FALSE;
                    if (x[j] == ETX)
                    {
                        printf("%d bytes\n", len);
                        return len;
                    }
                }
                else
                {
                    if (x[j] == DLE)
                        dled = TRUE;
                }
                /*endif*/
            }
            /*endfor*/
        }
        /*endfor*/
    }
    /*endif*/
    return 0;
}

int send(int fd, char *t, char *s)
{
    char buf[132 + 1];
    char *u;
    
    printf("Send     '");
    print_string(t);
    printf("'\n");
    fflush(stdout);
    write(fd, t, strlen(t));
    if (s)
        expect(fd, s);
    /*endif*/
}

int send_stuffed(int fd, uint8_t *t, int len, char *s)
{
    uint8_t buf[20000];
    int i;
    int j;

    printf("Send stuffed (%d) ", len);
    for (i = 0, j = 0;  i < len;  i++)
    {
        //printf("%02X ", *t & 0xFF);
        if (*t == DLE)
            buf[j++] = DLE;
        /*endif*/
        buf[j++] = *t++;
    }
    /*endfor*/
    printf("\n");
    fflush(stdout);
    buf[j++] = DLE;
    buf[j++] = ETX;
    write(fd, buf, j);
    if (s)
        expect(fd, s);
    /*endif*/
}

int hdlc_header(uint8_t frame[], int last, int type)
{
    frame[0] = 0xFF;
    frame[1] = (last)  ?  0x13  :  0x03;
    frame[2] = type;
    return 3;
};

int general_test(int fd)
{
    printf("General AT command tests\n");
    send(fd, "ATZ\r", "ATZ\r\nOK\r\n");
    
    /* Test the basics */
    /* Upper and lower case AT should work. */
    send(fd, "atI0\r", "atI0\r\n" MODEM_MODEL "\r\n\r\nOK\r\n");
    send(fd, "ATI0\r", "ATI0\r\n" MODEM_MODEL "\r\n\r\nOK\r\n");
    /* Backspacing should work smoothly */
    send(fd, "A\bA\b\b\b\bAT\b\bATI0\r", "ATI0\r\n" MODEM_MODEL "\r\n\r\nOK\r\n");
    /* Split lines should work smoothly */
    send(fd, "ATI0", NULL);
    send(fd, "\r", "ATI0\r\n" MODEM_MODEL "\r\n\r\nOK\r\n");

    /* Test response control operations */
    send(fd, "ATQ1\r", "ATQ1");
    send(fd, "ATV0\r", "ATV0");
    send(fd, "ATQ0\r", "ATQ00\r");
    send(fd, "ATV0\r", "ATV00\r");
    send(fd, "ATV1\r", "ATV1\r\nOK\r\n");
    send(fd, "ATE0\r", "ATE0\r\nOK\r\n");
    send(fd, "ATE1\r", "\r\nOK\r\n");
    send(fd, "ATE0\r", "ATE0\r\nOK\r\n");

    /* Test S register operations */
    send(fd, "ATS10?\r", "\r\n000\r\n\r\nOK\r\n");
    send(fd, "ATS10=1\r", "\r\nOK\r\n");
    send(fd, "ATS10.5?\r", "\r\n0\r\n\r\nOK\r\n");
    send(fd, "ATS10.5=1\r", "\r\nOK\r\n");
    send(fd, "ATS10.5?\r", "\r\n1\r\n\r\nOK\r\n");
    send(fd, "ATS10?\r", "\r\n033\r\n\r\nOK\r\n");

    /* Test information reporting operations */
    send(fd, "ATI0\r", "\r\n" MODEM_MODEL "\r\n\r\nOK\r\n");
    send(fd, "ATI3\r", "\r\n" MODEM_MAKER "\r\n\r\nOK\r\n");
    send(fd, "ATI8\r", "\r\nNMBR = \r\n\r\nOK\r\n");
    send(fd, "ATI9\r", "\r\nNDID = \r\n\r\nOK\r\n");
    send(fd, "AT+FCLASS?\r", "\r\n0\r\n\r\nOK\r\n");
    send(fd, "AT+FCLASS=?\r", "\r\n0,1\r\n\r\nOK\r\n");
    send(fd, "AT+FRS?\r", "\r\n-1\r\n\r\nOK\r\n");
    send(fd, "AT+FRS=?\r", "\r\n0-255\r\n\r\nOK\r\n");
    send(fd, "AT+FRH?\r", "\r\n-1\r\n\r\nOK\r\n");
    send(fd, "AT+FRH=?\r", "\r\n3\r\n\r\nOK\r\n");
    send(fd, "AT+FRM?\r", "\r\n-1\r\n\r\nOK\r\n");
    send(fd, "AT+FRM=?\r", "\r\n24,48,72,73,74,96,97,98,121,122,145,146\r\n\r\nOK\r\n");
    send(fd, "AT+FTS?\r", "\r\n-1\r\n\r\nOK\r\n");
    send(fd, "AT+FTS=?\r", "\r\n0-255\r\n\r\nOK\r\n");
    send(fd, "AT+FTH?\r", "\r\n-1\r\n\r\nOK\r\n");
    send(fd, "AT+FTH=?\r", "\r\n3\r\n\r\nOK\r\n");
    send(fd, "AT+FTM?\r", "\r\n-1\r\n\r\nOK\r\n");
    send(fd, "AT+FTM=?\r", "\r\n24,48,72,73,74,96,97,98,121,122,145,146\r\n\r\nOK\r\n");
    

#if 0
    send(fd, "ATA\r", "ATA");
    //expect(fd, "\r\nOK\r\n");
    //expect(fd, "\r\nOK\r\n");
    //send(fd, "ATH\r", "ATH");
    //expect(fd, "\r\nOK\r\n");
    send(fd, "ATH1\r", "\r\nOK\r\n");
    send(fd, "ATH1\r", "\r\nOK\r\n");
    send(fd, "ATL\r", "\r\nOK\r\n");
    send(fd, "ATM\r", "\r\nOK\r\n");
    send(fd, "ATN\r", "\r\nOK\r\n");
    send(fd, "ATO\r", "\r\nOK\r\n");
    send(fd, "ATZ1\r", "\r\nOK\r\n");
    send(fd, "AT+FAA=0\r", "\r\nOK\r\n");
    send(fd, "AT+FAA=?\r", "\r\nOK\r\n");
    send(fd, "AT+FAA?\r", "\r\nOK\r\n");
    send(fd, "AT+FMFR?\r", "\r\nOK\r\n");
    send(fd, "AT+FMDL?\r", "\r\nOK\r\n");
    send(fd, "AT+FREV?\r", "\r\nOK\r\n");
    send(fd, "AT+FRH=1\r", "\r\nOK\r\n");
    send(fd, "AT+FRH=3\r", "\r\nOK\r\n");
    send(fd, "AT+FRM=96\r", "\r\nOK\r\n");
    send(fd, "AT+FRS=1\r", "\r\nOK\r\n");
    send(fd, "AT+FTH=3\r", "\r\nOK\r\n");
    send(fd, "AT+FTM=144\r", "\r\nOK\r\n");
    send(fd, "AT+FTS=1\r", "\r\nOK\r\n");
    send(fd, "AT+FREV?\r", "\r\nOK\r\n");
    send(fd, "AT+V\r", "\r\nOK\r\n");
    send(fd, "AT#CID=?\r", "\r\nOK\r\n");
    send(fd, "AT#CID=0\r", "\r\nOK\r\n");
    send(fd, "AT#CID=10\r", "\r\nOK\r\n");
    send(fd, "AT#CID?\r", "\r\nOK\r\n");
    send(fd, "AT&C1\r", "\r\nOK\r\n");
    send(fd, "AT&D2\r", "\r\nOK\r\n");
    send(fd, "AT&F\r", "\r\nOK\r\n");
    send(fd, "AT&H7\r", "\r\nOK\r\n");
    send(fd, "atE1\r", "\r\nOK\r\n");
    send(fd, "atL1\r", "\r\nOK\r\n");
    send(fd, "atM1\r", "\r\nOK\r\n");
    send(fd, "atN1\r", "\r\nOK\r\n");
    send(fd, "atO\r", "\r\nOK\r\n");

    /* Test dialing */
    send(fd, "ATDT123 456-7890PFVLD\r", 22, "\r\nCONNECT\r\n");
#endif

    /* Check all the patterns */
    //send(fd, "AT+ASTO=?\r", "\r\n+AST0:\r\n\r\nOK\r\n");
    //send(fd, "AT+DR=?\r", "\r\n+DR:(0,1)\r\n\r\nOK\r\n");
    //send(fd, "AT+DS=?\r", "\r\n+DS:(0,1)\r\n\r\nOK\r\n");
    //send(fd, "AT+EB=?\r", "\r\n+EB:(0,1)\r\n\r\nOK\r\n");
    send(fd, "AT+EFCS=?\r", "\r\n+EFCS:(0-2)\r\n\r\nOK\r\n");
    send(fd, "AT+EFRAM=?\r", "\r\n+EFRAM:(1-65535),(1-65535)\r\n\r\nOK\r\n");
    send(fd, "AT+ER=?\r", "\r\n+ER:(0,1)\r\n\r\nOK\r\n");
    //send(fd, "AT+ES=?\r", "\r\n+ES:(0,1)\r\n\r\nOK\r\n");
    //send(fd, "AT+ESR=?\r", "\r\n+ESR:(0,1)\r\n\r\nOK\r\n");
    send(fd, "AT+ETBM=?\r", "\r\n+ETBM:(0-2),(0-2),(0-30)\r\n\r\nOK\r\n");
    send(fd, "AT+EWIND=?\r", "\r\n+EWIND:(1-127),(1-127)\r\n\r\nOK\r\n");
    send(fd, "AT+FAR=?\r", "\r\n0\r\n\r\nOK\r\n");
    //send(fd, "AT+FCL=?\r", "\r\n0\r\n\r\nOK\r\n");
    send(fd, "AT+FCLASS=?\r", "\r\n0,1.0\r\n\r\nOK\r\n");
    send(fd, "AT+FDD=?\r", "\r\n(0,1)\r\n\r\nOK\r\n");
    //send(fd, "AT+FIT=?\r", "\r\n0\r\n\r\nOK\r\n");
    //send(fd, "AT+FLO=?\r", "\r\n0\r\n\r\nOK\r\n");
    send(fd, "AT+FMI=?\r", "\r\nwww.opencall.org\r\n\r\nOK\r\n");
    send(fd, "AT+FMM=?\r", "\r\nspandsp\r\n\r\nOK\r\n");
    send(fd, "AT+FMR=?\r", "\r\n0.0.2\r\n\r\nOK\r\n");
    send(fd, "AT+FPR=?\r", "\r\n115200\r\n\r\nOK\r\n");
    send(fd, "AT+FRH=?\r", "\r\n3\r\n\r\nOK\r\n");
    send(fd, "AT+FRM=?\r", "\r\n24,48,72,73,74,96,97,98,121,122,145,146\r\n\r\nOK\r\n");
    send(fd, "AT+FRS=?\r", "\r\n0-255\r\n\r\nOK\r\n");
    send(fd, "AT+FTH=?\r", "\r\n3\r\n\r\nOK\r\n");
    send(fd, "AT+FTM=?\r", "\r\n24,48,72,73,74,96,97,98,121,122,145,146\r\n\r\nOK\r\n");
    send(fd, "AT+FTS=?\r", "\r\n0-255\r\n\r\nOK\r\n");
    send(fd, "AT+GCAP=?\r", "\r\n+GCAP:+FCLASS\r\n\r\nOK\r\n");
    //send(fd, "AT+GCI=?\r", "\r\n+GCI:\r\n\r\nOK\r\n");
    send(fd, "AT+GMI=?\r", "\r\nwww.opencall.org\r\n\r\nOK\r\n");
    send(fd, "AT+GMM=?\r", "\r\nspandsp\r\n\r\nOK\r\n");
    send(fd, "AT+GMR=?\r", "\r\n0.0.2\r\n\r\nOK\r\n");
    send(fd, "AT+GOI=?\r", "\r\n42\r\n\r\nOK\r\n");
    send(fd, "AT+GSN=?\r", "\r\n42\r\n\r\nOK\r\n");
    send(fd, "AT+ICF=?\r", "\r\n+ICF:(0-6),(0-3)\r\n\r\nOK\r\n");
    send(fd, "AT+ICLOK=?\r", "\r\n+ICLOK:(0-2)\r\n\r\nOK\r\n");
    send(fd, "AT+IDSR=?\r", "\r\n+IDSR:(0-2)\r\n\r\nOK\r\n");
    //send(fd, "AT+IFC=?\r", "\r\n+IFC:(0,1)\r\n\r\nOK\r\n");
    //send(fd, "AT+ILRR=?\r", "\r\n+ILRR:(0,1)\r\n\r\nOK\r\n");
    send(fd, "AT+ILSD=?\r", "\r\n+ILSD:(0,1)\r\n\r\nOK\r\n");
    send(fd, "AT+IPR=?\r", "\r\n+IPR:(115200),(115200)\r\n\r\nOK\r\n");
    send(fd, "AT+IRTS=?\r", "\r\n+IRTS:(0,1)\r\n\r\nOK\r\n");
    //send(fd, "AT+MA=?\r", "\r\n+MA:(0,1)\r\n\r\nOK\r\n");
    //send(fd, "AT+MR=?\r", "\r\n+MR:(0,1)\r\n\r\nOK\r\n");
    //send(fd, "AT+MS=?\r", "\r\n+MS:(0,1)\r\n\r\nOK\r\n");
    send(fd, "AT+MSC=?\r", "\r\n+MSC:(0,1)\r\n\r\nOK\r\n");
    //send(fd, "AT+MV18AM=?\r", "\r\n+MV18AM:(0,1)\r\n\r\nOK\r\n");
    send(fd, "AT+MV18P=?\r", "\r\n+MV18P:(2-7)\r\n\r\nOK\r\n");
    send(fd, "AT+MV18R=?\r", "\r\n+MV18R:(0,1)\r\n\r\nOK\r\n");
    //send(fd, "AT+MV18S=?\r", "\r\n+MV18S:(0,1)\r\n\r\nOK\r\n");
    //send(fd, "AT+TADR=?\r", "\r\n+TADR:(0,1)\r\n\r\nOK\r\n");
    send(fd, "AT+TAL=?\r", "\r\n+TAL:(0,1),(0,1)\r\n\r\nOK\r\n");
    send(fd, "AT+TALS=?\r", "\r\n+TALS:(0-3)\r\n\r\nOK\r\n");
    send(fd, "AT+TDLS=?\r", "\r\n+TDLS:(0-4)\r\n\r\nOK\r\n");
    send(fd, "AT+TE140=?\r", "\r\n+TE140:(0,1)\r\n\r\nOK\r\n");
    send(fd, "AT+TE141=?\r", "\r\n+TE141:(0,1)\r\n\r\nOK\r\n");
    send(fd, "AT+TEPAL=?\r", "\r\n+TEPAL:(0,1)\r\n\r\nOK\r\n");
    send(fd, "AT+TEPDL=?\r", "\r\n+TEPDL:(0,1)\r\n\r\nOK\r\n");
    send(fd, "AT+TERDL=?\r", "\r\n+TERDL:(0,1)\r\n\r\nOK\r\n");
    send(fd, "AT+TLDL=?\r", "\r\n+TLDL:(0,1)\r\n\r\nOK\r\n");
    send(fd, "AT+TMODE=?\r", "\r\n+TMODE:(0,1)\r\n\r\nOK\r\n");
    //send(fd, "AT+TNUM=?\r", "\r\n+TNUM:(0,1)\r\n\r\nOK\r\n");
    send(fd, "AT+TRDL=?\r", "\r\n+TRDL:(0,1)\r\n\r\nOK\r\n");
    //send(fd, "AT+TRDLS=?\r", "\r\n+TRDLS:(0,1)\r\n\r\nOK\r\n");
    send(fd, "AT+TRES=?\r", "\r\n+TRES:(0-2)\r\n\r\nOK\r\n");
    send(fd, "AT+TSELF=?\r", "\r\n+TSELF:(0,1)\r\n\r\nOK\r\n");
    send(fd, "AT+TTER=?\r", "\r\n+TTER:(0-65535),(0-65535)\r\n\r\nOK\r\n");

    return 0;
}

int fax_send_test(int fd)
{
    uint8_t frame[10000];
    int i;

    printf("FAX send test from T.31 spec\n");
    send(fd, "ATE0\r", "ATE0\r\nOK\r\n");
    send(fd, "AT+FCLASS=?\r", "\r\n0,1.0\r\n\r\nOK\r\n");
    send(fd, "AT+FCLASS?\r", "\r\n0\r\n\r\nOK\r\n");
    send(fd, "AT+FCLASS=1.0\r", "\r\nOK\r\n");
    send(fd, "AT+FCLASS?\r", "\r\n1\r\n\r\nOK\r\n");
    send(fd, "ATD123456789\r", "\r\nCONNECT\r\n");
    //hdlc_header(frame, FALSE, T30_NSF);
    /* AT+FRH=3 is implied when dialing in AT+FCLASS=1.0 state */
    //expect_stuffed(fd, frame, 3);
    //expect(fd, "\r\nOK\r\n");
    //send(fd, "AT+FRH=3\r", "\r\nCONNECT\r\n");
    hdlc_header(frame, FALSE, T30_CSI);
    for (i = 3;  i < 3 + 21;  i++)
        frame[i] = ' ';
    /*endfor*/
    expect_stuffed(fd, frame, 21);
    expect(fd, "\r\nOK\r\n");
    send(fd, "AT+FRH=3\r", "\r\nCONNECT\r\n");
    hdlc_header(frame, TRUE, T30_DIS);
    frame[3] = 0x80;
    frame[4] = 0xE8;
    frame[5] = 0xCE;
    frame[6] = 0xF4;
    frame[7] = 0x80;
    frame[8] = 0x80;
    frame[9] = 0x81;
    frame[10] = 0x80;
    frame[11] = 0x80;
    frame[12] = 0x80;
    frame[13] = 0x18;
    frame[14] = 0x00;
    frame[15] = 0x00;
    expect_stuffed(fd, frame, 14);
    expect(fd, "\r\nOK\r\n");
    send(fd, "AT+FRH=3\r", "\r\nNO CARRIER\r\n");
    send(fd, "AT+FTH=3\r", "\r\nCONNECT\r\n");
    hdlc_header(frame, FALSE, T30_TSI);
    for (i = 3;  i < 3 + 21;  i++)
        frame[i] = ' ';
    /*endfor*/
    send_stuffed(fd, frame, i, "\r\nCONNECT\r\n");
    hdlc_header(frame, TRUE, T30_DCS);
    frame[3] = 0x83;
    frame[4] = 0xE8;
    frame[5] = 0xc6;
    frame[6] = 0x80;
    frame[7] = 0x80;
    frame[8] = 0x80;
    frame[9] = 0x00;
    send_stuffed(fd, frame, 10, "\r\nOK\r\n");
    send(fd, "AT+FTS=8;+FTM=96\r", "\r\nCONNECT\r\n");
    /* TCF */
    for (i = 0;  i < 1500;  i++)
        frame[i] = 0;
    /*endfor*/
    send_stuffed(fd, frame, 1500, "\r\nOK\r\n");
    send(fd, "AT+FRH=3\r", "\r\nCONNECT\r\n");
    hdlc_header(frame, TRUE, T30_CFR);
    expect_stuffed(fd, frame, 3);
    expect(fd, "\r\nOK\r\n");
    send(fd, "AT+FRH=3\r", "\r\nNO CARRIER\r\n");
    send(fd, "AT+FTM=96\r", "\r\nCONNECT\r\n");
    /* Image */
    for (i = 0;  i < 10;  i++)
        frame[i] = 0x0E + i;
    /*endfor*/
    send_stuffed(fd, frame, 10, "\r\nOK\r\n");
    send(fd, "AT+FTS=8;+FTH=3\r", "\r\nCONNECT\r\n");
    hdlc_header(frame, TRUE, T30_EOP);
    send_stuffed(fd, frame, 3, "\r\nOK\r\n");
    send(fd, "AT+FRH=3\r", "\r\nCONNECT\r\n");
    hdlc_header(frame, TRUE, T30_MCF);
    expect_stuffed(fd, frame, 3);
    expect(fd, "\r\nOK\r\n");
    send(fd, "AT+FRH=3\r", "\r\nNO CARRIER\r\n");
    send(fd, "AT+FTH=3\r", "\r\nCONNECT\r\n");
    hdlc_header(frame, TRUE, T30_DCN);
    send_stuffed(fd, frame, 3, "\r\nOK\r\n");
    send(fd, "ATH0\r", "\r\nOK\r\n");    
    return 0;
}

int fax_receive_test(int fd)
{
    uint8_t frame[132 + 1];
    int i;

    printf("FAX receive test from T.31 spec\n");
    send(fd, "ATE0\r", "ATE0\r\nOK\r\n");
    send(fd, "AT+FCLASS=1.0\r", "\r\nOK\r\n");
    /* Some time later, as a call arrives */
    expect(fd, "RING");
    send(fd, "ATA\r", "\r\nCONNECT\r\n");
    /* AT+FTH=3 is implied when answering in AT+FCLASS=1.0 state */
    hdlc_header(frame, FALSE, T30_CSI);
    for (i = 3;  i < 3 + 21;  i++)
        frame[i] = ' ';
    /*endfor*/
    send_stuffed(fd, frame, i, "\r\nCONNECT\r\n");
    hdlc_header(frame, TRUE, T30_DIS);
    for (i = 3;  i < 3 + 4;  i++)
        frame[i] = 0;
    /*endfor*/
    send_stuffed(fd, frame, 7, "\r\nOK\r\n");
    send(fd, "AT+FRH=3\r", "\r\nCONNECT\r\n");
    hdlc_header(frame, FALSE, T30_TSI);
    for (i = 3;  i < 3 + 21;  i++)
        frame[i] = ' ';
    /*endfor*/
    expect_stuffed(fd, frame, i);
    expect(fd, "\r\nOK\r\n");
    send(fd, "AT+FRH=3\r", "\r\nCONNECT\r\n");
    hdlc_header(frame, TRUE, T30_DCS);
    for (i = 3;  i < 3 + 4;  i++)
        frame[i] = 0;
    /*endfor*/
    expect_stuffed(fd, frame, 7);
    expect(fd, "\r\nOK\r\n");
    send(fd, "AT+FRH=3\r", "\r\nNO CARRIER\r\n");
    send(fd, "AT+FRM=96\r", "\r\nCONNECT\r\n");
    /* TCF */
    expect_any_stuffed(fd);
    expect(fd, "\r\nNO CARRIER\r\n");
    send(fd, "AT+FTH=3\r", "\r\nCONNECT\r\n");
    hdlc_header(frame, TRUE, T30_CFR);
    send_stuffed(fd, frame, 3, "\r\nOK\r\n");
    send(fd, "AT+FRM=96\r", "\r\nCONNECT\r\n");
    /* Image */
    expect_any_stuffed(fd);
    expect(fd, "\r\nNO CARRIER\r\n");
    send(fd, "AT+FRH=3\r", "\r\nCONNECT\r\n");
    hdlc_header(frame, TRUE, T30_EOP);
    expect_stuffed(fd, frame, 3);
    expect(fd, "\r\nOK\r\n");
    send(fd, "AT+FRH=3\r", "\r\nNO CARRIER\r\n");
    send(fd, "AT+FTH=3\r", "\r\nCONNECT\r\n");
    hdlc_header(frame, TRUE, T30_MCF);
    send_stuffed(fd, frame, 3, "\r\nOK\r\n");
    send(fd, "AT+FRH=3\r", "\r\nCONNECT\r\n");
    hdlc_header(frame, TRUE, T30_DCN);
    expect_stuffed(fd, frame, 3);
    expect(fd, "\r\nOK\r\n");
    send(fd, "AT+FRH=3\r", "\r\nNO CARRIER\r\n");
    send(fd, "ATH0\r", "\r\nOK\r\n");
    return 0;
}

int main(int argc, char *argv[])
{
    int i;
    int j;
    char buf[128 + 1];
    int fd;
    struct termios tios;
    
    if (argc < 2)
    {
        fprintf(stderr, "No port specified\n");
        exit(2);
    }
    /*endif*/
    if ((fd = open(argv[1], O_RDWR)) < 0)
    {
        fprintf(stderr, "Cannot open the port\n");
        exit(2);
    }
    /*endif*/
    if (tcgetattr(fd, &tios) == 0)
    {
        tios.c_cflag &= ~(CSIZE | CSTOPB | PARENB);
        tios.c_cflag |= CS8 | CREAD | CLOCAL;
        tios.c_iflag = IGNPAR;
        tios.c_oflag = 0;
        tios.c_lflag = 0;
        tcsetattr(fd, TCSAFLUSH, &tios);
    }
    /*endif*/

    //general_test(fd);
    fax_receive_test(fd);
    //fax_send_test(fd);
    return  0;
}
