/* Set the protocol to use */
#define PROTOCOL_CLASS      "t31"
#define PROTOCOL_VARIANT    ""
#define PROTOCOL_END        UC_MODE_CPE

/* Set the test numbers to use */
#define BASE_ANI            987654321
#define BASE_DNIS           1236

/* Set the trunks to use */
#define FIRST_TRUNK_CIRCUIT 94
#define NUM_TRUNKS          1
#define FIRST_OG_TRUNK      0
#define NUM_OG_TRUNKS       0

/*
 * libt31 - A Unicall protocol module which emulates a T.31 compatible modem
 *
 * t31_test_call.c - Tests for the T.31 command interpreter.
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2004 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: t31_test_call.c,v 1.6 2004/12/27 12:45:10 steveu Exp $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <stdlib.h>
#include <stdint.h>
#include <unistd.h>
#include <sys/signal.h>
#include <sys/select.h>
#include <sys/wait.h>
#include <sys/resource.h>
#include <sys/time.h>
#include <pthread.h>
#include <audiofile.h>
#include <tiffio.h>

#define FALSE 0
#define TRUE (!FALSE)

#include <spandsp.h>
#include <spandsp/t30_fcf.h>
#include "unicall.h"

#define SAMPLES_PER_CHUNK   160

struct
{
    pthread_t thread;
    int chan;
    int sig_fd;
    int fd;
    uc_call_t *call;
    uc_crn_t crn;
    uint8_t message[100000];
    int message_len;
    int message_ptr;
    int xxx;
    int cause;

    char *tag;
    
    char originating_number[32];
    char destination_number[32];

    int caller_mode;
    
    time_t last_call;

    int up;
    int16_t silence[SAMPLES_PER_CHUNK];
    int16_t t30_amp[SAMPLES_PER_CHUNK];
    t30_state_t t30_state;
} chan_stuff[4];

AFfilehandle handle_a;
AFfilehandle handle_b;

void channel_read_handler(uc_t *uc, int chan, void *user_data, uint8_t *buf, int len);
int channel_write_handler(uc_t *uc, int chan, void *user_data, uint8_t *buf, int max_len);
int channel_error(uc_t *uc, int chan, void *user_data, int cause);
int signaling_error(uc_t *uc, void *user_data, int cause);
void phase_b_handler(t30_state_t *s, void *user_data, int result);
void phase_d_handler(t30_state_t *s, void *user_data, int result);
void phase_e_handler(t30_state_t *s, void *user_data, int result);

void channel_read_handler(uc_t *uc, int chan, void *user_data, uint8_t *buf, int len)
{
    int outframes;

//printf("Channel read %d bytes\n", len);
    outframes = afWriteFrames(handle_b, AF_DEFAULT_TRACK, (int16_t *) buf, len/sizeof(int16_t));
    fax_rx_process(&chan_stuff[chan].t30_state, (int16_t *) buf, len/sizeof(int16_t));
}
/*- End of function --------------------------------------------------------*/

int channel_write_handler(uc_t *uc, int chan, void *user_data, uint8_t *buf, int max_len)
{
    int len;

printf("Channel write %d samples\n", max_len);
    len = fax_tx_process(&chan_stuff[chan].t30_state, (int16_t *) buf, max_len/sizeof(int16_t));
    if (len*sizeof(int16_t) < max_len)
        memset(buf + len*sizeof(int16_t), 0, max_len - len*sizeof(int16_t));
    return max_len;
}
/*- End of function --------------------------------------------------------*/

int channel_error(uc_t *uc, int chan, void *user_data, int cause)
{
    printf("Error %d\n", cause);
    return  0;
}
/*- End of function --------------------------------------------------------*/

int signaling_error(uc_t *uc, void *user_data, int cause)
{
    printf("Error %d\n", cause);
    return  0;
}
/*- End of function --------------------------------------------------------*/

void phase_b_handler(t30_state_t *s, void *user_data, int result)
{
    int i;
    
    i = (int) user_data;
    printf("Phase B handler on channel %d - 0x%X\n", i, result);
}
/*- End of function --------------------------------------------------------*/

void phase_d_handler(t30_state_t *s, void *user_data, int result)
{
    int i;
    t30_stats_t t;
    char ident[21];

    i = (int) user_data;
    printf("Phase D handler on channel %d - 0x%X\n", i, result);
    fax_get_transfer_statistics(s, &t);
    printf("Phase D: bit rate %d\n", t.bit_rate);
    printf("Phase D: pages transferred %d\n", t.pages_transferred);
    printf("Phase D: image size %d x %d\n", t.columns, t.rows);
    printf("Phase D: image resolution %d x %d\n", t.column_resolution, t.row_resolution);
    printf("Phase D: bad rows %d\n", t.bad_rows);
    printf("Phase D: longest bad row run %d\n", t.longest_bad_row_run);
    printf("Phase D: coding method %d\n", t.encoding);
    printf("Phase D: image size %d\n", t.image_size);
    fax_get_local_ident(s, ident);
    printf("Phase D: local ident '%s'\n", ident);
    fax_get_far_ident(s, ident);
    printf("Phase D: remote ident '%s'\n", ident);
}
/*- End of function --------------------------------------------------------*/

void phase_e_handler(t30_state_t *s, void *user_data, int result)
{
    int i;
    
    i = (int) user_data;
    printf("Phase E handler on channel %d\n", i);
}
/*- End of function --------------------------------------------------------*/

static void initiate_call(uc_t *uc, int chan, uc_event_t *e)
{
    uc_makecall_t makecall;
    uc_callparms_t *callparms;
    int ret;

    printf ("Initiating call on channel %d\n", chan);
    if ((callparms = uc_new_callparms(NULL)) == NULL)
        return;
    /*endif*/
    uc_callparm_originating_number(callparms, chan_stuff[chan].originating_number);
    uc_callparm_destination_number(callparms, chan_stuff[chan].destination_number);
    makecall.callparms = callparms;
    makecall.crn = 0;
    if ((ret = uc_call_control(uc, UC_OP_MAKECALL, 0, (void *) &makecall)) != UC_RET_OK)
        fprintf(stderr, "Make Call failed - %d\n", ret);
    /*endif*/
    chan_stuff[chan].crn = makecall.crn;
    free(callparms);
}
/*- End of function --------------------------------------------------------*/

static void start_fax(uc_t *uc, int chan, int caller)
{
    printf("Starting FAX - %d\n", caller);
    fax_init(&chan_stuff[chan].t30_state, caller, NULL);
    chan_stuff[chan].t30_state.verbose = 1;
    fax_set_local_ident(&chan_stuff[chan].t30_state, "11111111");
    if (caller)
        fax_set_tx_file(&chan_stuff[chan].t30_state, "t31_tx.tif");
    else
        fax_set_rx_file(&chan_stuff[chan].t30_state, "t31_rx.tif");
    fax_set_phase_b_handler(&chan_stuff[chan].t30_state, phase_b_handler, (void *) 0);
    fax_set_phase_d_handler(&chan_stuff[chan].t30_state, phase_d_handler, (void *) 0);
    fax_set_phase_e_handler(&chan_stuff[chan].t30_state, phase_e_handler, (void *) 0);
    memset(chan_stuff[chan].t30_amp, 0, sizeof(chan_stuff[chan].t30_amp));
    uc_set_channel_read_callback(uc, 0, channel_read_handler, (void *) chan);
    uc_set_channel_write_callback(uc, 0, channel_write_handler, (void *) chan);
}
/*- End of function --------------------------------------------------------*/

static void handle_uc_event(uc_t *uc, void *user_data, uc_event_t *e)
{
    int file;
    time_t now;
    int chan;

    chan = (int) user_data;
    switch (e->e)
    {
    case UC_EVENT_DEVICEFAIL:
        break;
    case UC_EVENT_PROTOCOLFAIL:
        printf("%s -- Protocol failure on channel %d, cause (%d) %s\n", chan_stuff[chan].tag, e->gen.channel, e->gen.data, uc_error_message(uc, e->gen.data));
        fprintf(stderr, "%s -- Protocol failure on channel %d, cause (%d) %s\n", chan_stuff[chan].tag, e->gen.channel, e->gen.data, uc_error_message(uc, e->gen.data));
        break;
    case UC_EVENT_SIGCHANSTATUS:
        printf("%s -- Signalling channel status - %s\n", chan_stuff[chan].tag, e->sigchanstatus.ok  ?  "Up"  :  "Down");
        fprintf(stderr, "%s -- Signalling channel status - %s\n", chan_stuff[chan].tag, e->sigchanstatus.ok  ?  "Up"  :  "Down");
        break;
    case UC_EVENT_ALARM:
        printf("%s -- %s - 0x%X/0x%X\n", chan_stuff[chan].tag, uc_event2str(e->e), e->alarm.raised, e->alarm.cleared);
        fprintf(stderr, "%s -- %s - 0x%X\n", chan_stuff[chan].tag, uc_event2str(e->e), e->alarm.raised, e->alarm.cleared);
        break;
    case UC_EVENT_FARBLOCKED:
        printf("%s -- %s! :-(\n", chan_stuff[chan].tag, uc_event2str(e->e));
        fprintf(stderr, "%s -- %s! :-(\n", chan_stuff[chan].tag, uc_event2str(e->e));
        chan_stuff[chan].xxx &= ~1;
        break;
    case UC_EVENT_FARUNBLOCKED:
        printf("%s -- %s! :-)\n", chan_stuff[chan].tag, uc_event2str(e->e));
        fprintf(stderr, "%s -- %s! :-)\n", chan_stuff[chan].tag, uc_event2str(e->e));
        chan_stuff[chan].xxx |= 1;
        if (chan_stuff[chan].xxx == 3)
        {
            if (chan_stuff[chan].caller_mode)
                initiate_call(uc, chan, e);
            /*endif*/
        }
        /*endif*/
        break;
    case UC_EVENT_LOCALBLOCKED:
        printf("%s -- %s! :-(\n", chan_stuff[chan].tag, uc_event2str(e->e));
        fprintf(stderr, "%s -- %s! :-(\n", chan_stuff[chan].tag, uc_event2str(e->e));
        chan_stuff[chan].xxx &= ~2;
        break;
    case UC_EVENT_LOCALUNBLOCKED:
        printf("%s -- %s! :-)\n", chan_stuff[chan].tag, uc_event2str(e->e));
        fprintf(stderr, "%s -- %s! :-)\n", chan_stuff[chan].tag, uc_event2str(e->e));
        chan_stuff[chan].xxx |= 2;
        if (chan_stuff[chan].xxx == 3)
        {
            if (chan_stuff[chan].caller_mode)
                initiate_call(uc, chan, e);
            /*endif*/
        }
        /*endif*/
        break;
    case UC_EVENT_DIALING:
        printf("%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        fprintf(stderr, "%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        break;
    case UC_EVENT_PROCEEDING:
        printf("%s -- Proceeding on channel %d\n", chan_stuff[chan].tag, e->gen.channel);
        break;
    case UC_EVENT_ACCEPTED:
        printf("%s -- Accepted on channel %d\n", chan_stuff[chan].tag, e->gen.channel);
        fprintf(stderr, "%s -- Accepted on channel %d\n", chan_stuff[chan].tag, e->gen.channel);
        if (uc_call_control(uc, UC_OP_ANSWERCALL, e->offered.crn, NULL))
            fprintf(stderr, "Answer Call failed\n");
        /*endif*/
        break;
    case UC_EVENT_DETECTED:
        printf("%s -- %s on channel %d, CRN %d\n",
               chan_stuff[chan].tag, uc_event2str(e->e),
               e->gen.channel, e->gen.crn);
        fprintf(stderr, "%s -- %s on channel %d, CRN %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel, e->gen.crn);
        chan_stuff[chan].crn = e->gen.crn;
        break;
    case UC_EVENT_MOREDIGITS:
        printf("%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        fprintf(stderr, "%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        break;
    case UC_EVENT_ALERTING:
        printf("%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        fprintf(stderr, "%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        /* This is just a notification of call progress. We need take no action at this point. */
        break;
    case UC_EVENT_FARDISCONNECTED:
        printf("%s -- %s on channel %d\n",
               chan_stuff[chan].tag,
               uc_event2str(e->e),
               e->fardisconnected.channel);
        fprintf(stderr,
                "%s -- %s on channel %d\n",
                chan_stuff[chan].tag,
                uc_event2str(e->e),
                e->fardisconnected.channel);
        /* Kill any outstanding audio processing */
        uc_set_channel_read_callback(uc, 0, NULL, NULL);
        uc_set_channel_write_callback(uc, 0, NULL, NULL);
        chan_stuff[chan].up = FALSE;
        if (uc_call_control(uc, UC_OP_DROPCALL, e->fardisconnected.crn, (void *) UC_CAUSE_NORMAL_CLEARING))
            fprintf(stderr, "C Drop Call failed\n");
        /*endif*/
        break;
    case UC_EVENT_DROPCALL:
        printf("%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        fprintf(stderr, "%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        chan_stuff[chan].up = FALSE;
        if (uc_call_control(uc, UC_OP_RELEASECALL, e->gen.crn, NULL))
            fprintf(stderr, "Release Call failed\n");
        /*endif*/
        break;
    case UC_EVENT_RELEASECALL:
        printf("%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        fprintf(stderr, "%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        if (chan_stuff[chan].caller_mode)
            initiate_call(uc, chan, e);
        /*endif*/
        break;
    case UC_EVENT_OFFERED:
        printf("%s -- %s on channel %d, CRN %d (ANI: %s, DNIS: %s)\n",
               chan_stuff[chan].tag,
               uc_event2str(e->e),
               e->offered.channel,
               e->offered.crn,
               e->offered.parms.originating_number,
               e->offered.parms.destination_number);
        fprintf(stderr,
                "%s -- %s on channel %d, CRN %d (ANI: %s, DNIS: %s)\n",
                chan_stuff[chan].tag,
                uc_event2str(e->e),
                e->offered.channel,
                e->offered.crn,
                e->offered.parms.originating_number,
                e->offered.parms.destination_number);
        if (!chan_stuff[chan].caller_mode)
        {
            time(&now);
            switch (chan_stuff[chan].cause)
            {
            case 0:
                fprintf(stderr, "%s -- +++ ACCEPT (%d)\n", chan_stuff[chan].tag, now - chan_stuff[chan].last_call);
                if (uc_call_control(uc, UC_OP_ACCEPTCALL, e->offered.crn, NULL))
                    fprintf(stderr, "Accept Call failed\n");
                /*endif*/
                break;
            case 1:
                fprintf(stderr, "%s -- +++ ANSWER (%d)\n", chan_stuff[chan].tag, now - chan_stuff[chan].last_call);
                if (uc_call_control(uc, UC_OP_ANSWERCALL, e->offered.crn, NULL))
                    fprintf(stderr, "Answer Call failed\n");
                /*endif*/
                break;
            }
            /*endswitch*/
            chan_stuff[chan].last_call = now;
            if (++chan_stuff[chan].cause > 1)
                chan_stuff[chan].cause = 0;
            /*endif*/
        }
        /*endif*/
        break;
    case UC_EVENT_ANSWERED:
        printf("%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        fprintf(stderr, "%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        start_fax(uc, chan, FALSE);
        chan_stuff[chan].up = TRUE;
        break;
    case UC_EVENT_CONNECTED:
        printf("%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        fprintf(stderr, "%s -- %s on channel %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->gen.channel);
        start_fax(uc, chan, TRUE);
        chan_stuff[chan].up = TRUE;
        break;
    default:
        printf ("%s -- %s\n", chan_stuff[chan].tag, uc_event2str(e->e));
        fprintf(stderr, "%s --!! Unknown signaling event %d\n", chan_stuff[chan].tag, uc_event2str(e->e), e->e);
        break;
    }
    /*endswitch*/
}
/*- End of function --------------------------------------------------------*/

static void *run_uc(void *arg)
{
    uc_t *uc;
    uc_event_t *e;
    struct timeval tv = {0,0};
    struct timeval *next;
    fd_set rfds;
    fd_set wfds;
    fd_set efds;
    int res;
    int dfd;
    int chan;
    int len;
    int outframes;
    int16_t amp[SAMPLES_PER_CHUNK];
    
    chan = *((int *) arg);

    uc = uc_new(-1, -1, PROTOCOL_CLASS, PROTOCOL_VARIANT, PROTOCOL_END, 1);
    if (uc == NULL)
    {
        fprintf(stderr, "Unable to create instance\n");
        return NULL;
    }
    /*endif*/
    uc_get_device_handles(uc, chan, &chan_stuff[chan].fd, NULL);
    uc_set_signaling_callback(uc, handle_uc_event, (void *) chan);
    uc_set_signaling_error_callback(uc, signaling_error, (void *) chan);
    uc_set_channel_error_callback(uc, 0, channel_error, (void *) chan);
    uc_set_logging(uc, 0x7FFFFFFF, 1, chan_stuff[chan].tag);
    uc_call_control(uc, UC_OP_UNBLOCK, 0, (void *) -1);
    dfd = chan_stuff[chan].fd; 
    for (;;)
    {
        FD_ZERO(&rfds);
        FD_ZERO(&wfds);
        FD_ZERO(&efds);
        FD_SET(dfd, &rfds);
        FD_SET(dfd, &wfds);
        FD_SET(dfd, &efds);

        if ((next = uc_schedule_next(uc)))
        {
            gettimeofday(&tv, NULL);
            tv.tv_sec = next->tv_sec - tv.tv_sec;
            tv.tv_usec = next->tv_usec - tv.tv_usec;
            if (tv.tv_usec < 0)
            {
                tv.tv_usec += 1000000;
                tv.tv_sec -= 1;
            }
            /*endif*/
            if (tv.tv_sec < 0)
            {
                tv.tv_sec = 0;
                tv.tv_usec = 0;
            }
            /*endif*/
        }
        /*endif*/
        if (next == NULL  ||  tv.tv_sec > 0  ||  tv.tv_usec > 20000)
        {
            next = (struct timeval *) 42;
            tv.tv_sec = 0;
            tv.tv_usec = 20000;
        }
        res = select(dfd + 1, &rfds, NULL, &efds, next  ?  &tv  :  NULL);
        if (res >= 0)
        {
            if (res == 0)
                uc_schedule_run(uc);
            /*endif*/
            while ((e = uc_check_event(uc)))
            {
                printf("Non-callback signaling event\n");
                handle_uc_event(uc, (void *) chan, e);
            }
            /*endwhile*/
            if (chan_stuff[chan].up)
            {
                len = fax_tx_process(&chan_stuff[chan].t30_state, amp, SAMPLES_PER_CHUNK);
                /* The receive side always expects a full block of samples, but the
                   transmit side may not be sending any when it doesn't need to. We
                   may need to pad with some silence. */
                if (len < SAMPLES_PER_CHUNK)
                    memset(amp + len, 0, sizeof(int16_t)*(SAMPLES_PER_CHUNK - len));
                outframes = afWriteFrames(handle_a, AF_DEFAULT_TRACK, amp, SAMPLES_PER_CHUNK);
                if (outframes != SAMPLES_PER_CHUNK)
                    break;
//printf("fax_tx_process %d samples\n", len);
                uc_channel_write(uc, chan, (uint8_t *) amp, sizeof(int16_t)*SAMPLES_PER_CHUNK);
            }
        }
        else if (errno != EINTR)
        {
            fprintf(stderr, "Error (%d) on select: %s\n", errno, strerror(errno));
        }
        /*endif*/
    }
    /*endfor*/
    return NULL;
}
/*- End of function --------------------------------------------------------*/

int main(int argc, char *argv[]) 
{
    pthread_attr_t attr;
    int chan;
    int chanx;
    char dev_name[20];
    int j;
    int x;
    AFfilesetup filesetup;

    filesetup = afNewFileSetup();
    if (filesetup == AF_NULL_FILESETUP)
    {
        fprintf(stderr, "    Failed to create file setup\n");
        exit(2);
    }
    afInitSampleFormat(filesetup, AF_DEFAULT_TRACK, AF_SAMPFMT_TWOSCOMP, 16);
    afInitRate(filesetup, AF_DEFAULT_TRACK, (float) SAMPLE_RATE);
    afInitFileFormat(filesetup, AF_FILE_WAVE);
    afInitChannels(filesetup, AF_DEFAULT_TRACK, 1);
    handle_a = afOpenFile("a.wav", "w", filesetup);
    handle_b = afOpenFile("b.wav", "w", filesetup);

    uc_start();    
    pthread_attr_init(&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    if (argc < 1)
    {
        fprintf(stderr, "Usage: t31_test_call [call]\n");
        exit(1);
    }
    /*endif*/
    for (chan = 0;  chan < NUM_TRUNKS;  chan++)
    {
        chan_stuff[chan].fd = -1;
#if 0
        if (chan >= FIRST_OG_TRUNK  &&  chan < FIRST_OG_TRUNK + NUM_OG_TRUNKS)
            chan_stuff[chan].caller_mode = TRUE;
        else
            chan_stuff[chan].caller_mode = FALSE;
        /*endif*/
#else
        chan_stuff[chan].caller_mode = TRUE;
#endif
        chan_stuff[chan].chan = chan;
        sprintf(dev_name, "Chan %3d:", chanx);
        chan_stuff[chan].tag = strdup(dev_name);
        sprintf(chan_stuff[chan].originating_number, "%d", BASE_ANI + chan);
        sprintf(chan_stuff[chan].destination_number, "%d", BASE_DNIS + chan);
        printf("Thread for channel %d\n", chan);
        if (pthread_create(&chan_stuff[chan].thread, &attr, run_uc, &chan_stuff[chan].chan))
            exit(2);
        /*endif*/
    }
    /*endfor*/
    for (;;)
    {
        sleep(5);
        printf("Main thread\n");
    }
    /*endfor*/
    exit(0);

    return 0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
