/*
 * libunicall - a library for universal call handling on both analogue and
 *              digital telephone circuits.
 *
 * unicall.c
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2002 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: unicall.c,v 1.18 2005/07/11 13:22:24 steveu Exp $
 */

/*! \file unicall.h unicall-private.h */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <limits.h>
#include <stdio.h>
#include <stdarg.h>
#include <fcntl.h>
#include <stdlib.h>
#include <stdint.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <time.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#define __USE_GNU
#include <dlfcn.h>

#include <linux/zaptel.h>

#include "unicall.h"
#include "unicall/hashtable.h"
#include "unicall/unicall-private.h"

#define FALSE 0
#define TRUE (!FALSE)

static int uc_initialised = FALSE;

static int max_protocol_spec = 0;
static protocol_control_t protocol_specs[20];

static int loader_lock = 0;

uc_protocol_t *uc_protocol_list_init(void);

static void (*__uc_error)(char *text);
static void (*__uc_message)(char *text);

static int uc_queue_event(uc_t *uc, uc_event_t *ev, int len)
{
    uc_event_t *evx;
    uc_event_t **evxx;
    uc_event_queue_t *queue;
    int i;
    
    queue = &uc->event_queue;
    if (queue->pool <= 0)
    {
        if (queue->pool_allocated < 5)
        {
            queue->pool_allocated += 5;
            if ((evxx = (uc_event_t **) realloc(queue->event_pool, sizeof(uc_event_t *)*queue->pool_allocated)) == NULL)
                return -1;
            /*endif*/
            queue->event_pool = evxx;
        }
        /*endif*/
        if ((evx = (uc_event_t *) malloc(5*sizeof(uc_event_t))) == NULL)
            return  -1;
        /*endif*/
        for (i = 0;  i < 5;  i++)
            queue->event_pool[i] = &evx[i];
        /*endif*/
        queue->pool += 5;
        queue->total_allocated += 5;
    }
    /*endif*/
    if (queue->queue >= queue->queue_allocated)
    {
        queue->queue_allocated += 5;
        if ((evxx = (uc_event_t **) realloc(queue->events, sizeof(uc_event_t *)*queue->queue_allocated)) == NULL)
            return -1;
        /*endif*/
        queue->events = evxx;
    }
    /*endif*/
    evx = queue->event_pool[--queue->pool];
    memcpy(evx, ev, (len <= sizeof(*evx))  ?  len  :  sizeof(*evx));
    queue->events[queue->queue++] = evx;
    return  0;
}
/*- End of function --------------------------------------------------------*/

static int uc_get_next_event(uc_t *uc, uc_event_t **ev)
{
    uc_event_queue_t *queue;

    queue = &uc->event_queue;
    if (queue->queue <= 0)
        return  -1;
    /*endif*/
    *ev = queue->events[0];
    memcpy(&queue->events[0], &queue->events[1], queue->queue*sizeof(uc_event_t *));
    queue->queue--;
    return  0;
}
/*- End of function --------------------------------------------------------*/

static int uc_release_event(uc_t *uc, uc_event_t *ev)
{
    uc_event_queue_t *queue;
    uc_event_t **evxx;

    queue = &uc->event_queue;
    if (queue->pool >= queue->pool_allocated)
    {
        queue->pool_allocated += 5;
        if ((evxx = (uc_event_t **) realloc(queue->event_pool, sizeof(uc_event_t *)*queue->pool_allocated)) == NULL)
            return -1;
        /*endif*/
        queue->event_pool = evxx;
    }
    /*endif*/
    queue->event_pool[queue->pool++] = ev;
}
/*- End of function --------------------------------------------------------*/

int uc_report_event(uc_t *uc, uc_event_t *ev, int len)
{
    return uc_queue_event(uc, ev, len);
}
/*- End of function --------------------------------------------------------*/

static int str2protocolclass(const char *type)
{
    int i;
    int j;
    void *lib;
    uc_protocol_t *p;
    char name[512];

    for (i = 0;  i <= max_protocol_spec;  i++)
    {
        if (protocol_specs[i].protocol  &&  strcasecmp(protocol_specs[i].protocol->name, type) == 0)
        {
            return  i;
        }
        /*endif*/
    }
    /*endif*/

    //TODO: Need to deal with threading races properly here
    for (j = 0;  j < 10;  j++)
    {
        loader_lock++;
        if (loader_lock == 1)
            break;
        loader_lock--;
        sleep(1);
        for (i = 0;  i <= max_protocol_spec;  i++)
        {
            if (protocol_specs[i].protocol  &&  strcasecmp(protocol_specs[i].protocol->name, type) == 0)
            {
                return  i;
            }
            /*endif*/
        }
        /*endif*/
    }
    if (j == 10)
	return  -1;

    printf("Loading protocol %s\n", type);
    /* We may need to load this protocol for the first time. */
    sprintf(name, PROTOCOLDIR "/protocols/protocol_%s.so", type);
    if ((lib = dlopen(name, RTLD_NOW | RTLD_LOCAL)) == NULL)
    {
	printf("No protocol module %s - %s\n", name, dlerror());
        loader_lock--;
        return  -1;
    }
    if ((p = dlsym(lib, "protocol_descriptor")) == NULL)
    {
	printf("No protocol_descriptor in protocol module %s\n", name);
	//uc_log(uc, UC_LOG_WARNING, "No protocol_descriptor in protocol module %s\n", name);
        dlclose(lib);
        loader_lock--;
        return  -1;
    }
    if (strcasecmp(p->name, type))
    {
	printf("The protocol names do not match: %s vs %s\n", p->name, type);
	//uc_log(uc, UC_LOG_WARNING, "The protocol names do not match: %s vs %s\n", p->name, type);
        dlclose(lib);
        loader_lock--;
        return  -1;
    }
    protocol_specs[max_protocol_spec].lib = lib;
    protocol_specs[max_protocol_spec].protocol = p;
    i = max_protocol_spec++;
    loader_lock--;
    return  i;
}
/*- End of function --------------------------------------------------------*/

int uc_str2protocol_variant(int protocol_class, char *variant)
{
    int i;
    int len;
    char const *s;

    if (protocol_class >= max_protocol_spec  ||  protocol_specs[protocol_class].protocol == NULL)
        return  -1;
    /*endif*/
    s = protocol_specs[protocol_class].protocol->variants;
    for (i = 1;  *s;  i++)
    {
        len = strlen(s);
        if (strncmp(variant, s, len) == 0)
            return  i;
        /*endif*/
        s += (len + 1);
    }
    /*endfor*/
    return  -1;
}
/*- End of function --------------------------------------------------------*/

const char *uc_protocol_class2str(int protocol_class)
{
    if (protocol_class >= max_protocol_spec  ||  protocol_specs[protocol_class].protocol == NULL)
        return  NULL;
    /*endif*/
    return  protocol_specs[protocol_class].protocol->desc;
}
/*- End of function --------------------------------------------------------*/

const char *uc_protocol_class2short_str(int protocol_class)
{
    if (protocol_class >= max_protocol_spec  ||  protocol_specs[protocol_class].protocol == NULL)
        return  NULL;
    /*endif*/
    return  protocol_specs[protocol_class].protocol->short_desc;
}
/*- End of function --------------------------------------------------------*/

const char *uc_protocol_variant2str(int protocol_class, int protocol_variant)
{
    char const *s;

    if (protocol_class >= max_protocol_spec  ||  protocol_specs[protocol_class].protocol == NULL)
        return  NULL;
    /*endif*/
    s = protocol_specs[protocol_class].protocol->variant_descriptions;
    while (*s  &&  --protocol_variant)
        s += (strlen(s) + 1);
    /*endwhile*/
    return  s;
}
/*- End of function --------------------------------------------------------*/

void uc_set_signaling_callback(uc_t *uc, uc_signaling_callback_t *func, void *user_data)
{
    uc->signaling_func = func;
    uc->signaling_user_data = user_data;
}
/*- End of function --------------------------------------------------------*/

void uc_set_signaling_error_callback(uc_t *uc, uc_signaling_error_callback_t *func, void *user_data)
{
    uc->signaling_error_func = func;
    uc->signaling_error_user_data = user_data;
}
/*- End of function --------------------------------------------------------*/

int uc_set_channel_read_callback(uc_t *uc, int ch, uc_channel_read_callback_t *func, void *user_data)
{
    if (ch >= uc->num_channels)
        return  UC_RET_BAD_CHANNEL;
    /*endif*/
    uc->chan[ch].channel_read = func;
    uc->chan[ch].channel_read_user_data = user_data;
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

int uc_set_channel_write_callback(uc_t *uc, int ch, uc_channel_write_callback_t *func, void *user_data)
{
    if (ch >= uc->num_channels)
        return  UC_RET_BAD_CHANNEL;
    /*endif*/
    uc->chan[ch].channel_write = func;
    uc->chan[ch].channel_write_user_data = user_data;
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

int uc_set_channel_error_callback(uc_t *uc, int ch, uc_channel_error_callback_t *func, void *user_data)
{
    if (ch >= uc->num_channels)
        return  UC_RET_BAD_CHANNEL;
    /*endif*/
    uc->chan[ch].channel_error = func;
    uc->chan[ch].channel_error_user_data = user_data;
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

char *uc_error_message(uc_t *uc, int code)
{
    uc_call_t *call;

    if (code >= 0x8000)
    {
        /* It is a protocol dependant code */
        if (uc->protocol_class < max_protocol_spec
            &&
            protocol_specs[uc->protocol_class].protocol)
        {
            return protocol_specs[uc->protocol_class].protocol->xx_error_message(uc, code);
        }
        /*endif*/
    }
    /*endif*/
    return  "Undefined";
}
/*- End of function --------------------------------------------------------*/

int uc_get_device_handles(uc_t *uc, int ch, int *fe, int *be)
{
    if (ch >= uc->num_channels)
        return  UC_RET_BAD_CHANNEL;
    /*endif*/
    if (uc->protocol_class < max_protocol_spec
        &&
        protocol_specs[uc->protocol_class].protocol)
    {
        return 
        protocol_specs[uc->protocol_class].protocol->xx_get_device_handles(uc, ch, fe, be);
    }
    /*endif*/
    return  UC_RET_WRONG_PROTOCOL;
}
/*- End of function --------------------------------------------------------*/

int uc_channel_open(const char *protocol,
                    const char *variant,
                    int mode,
                    char *dev)
{
    uc_t *uc;
    int protocol_class;

    if ((protocol_class = str2protocolclass(protocol)) < 0)
        return -1;
    /*endif*/
    return  protocol_specs[protocol_class].protocol->xx_channel_open(variant, dev);
}
/*- End of function --------------------------------------------------------*/

int uc_channel_close(const char *protocol,
                     const char *variant,
                     int mode,
                     int fd)
{
    uc_t *uc;
    int protocol_class;

    if ((protocol_class = str2protocolclass(protocol)) < 0)
        return -1;
    /*endif*/
    return  protocol_specs[protocol_class].protocol->xx_channel_close(fd);
}
/*- End of function --------------------------------------------------------*/

int uc_channel_set_api_codec(uc_t *uc, int ch, int codec)
{
    if (ch >= uc->num_channels)
        return  UC_RET_BAD_CHANNEL;
    /*endif*/
    if (uc->chan[ch].api_codec != codec)
    {
        if (uc->protocol_class < max_protocol_spec
            &&
            protocol_specs[uc->protocol_class].protocol)
        {
            return  protocol_specs[uc->protocol_class].protocol->xx_channel_set_api_codec(uc, ch, codec);
        }
        /*endif*/
    }
    /*endif*/
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

int uc_channel_get_native_codec(uc_t *uc, int ch)
{
    if (ch >= uc->num_channels)
        return -1;
    /*endif*/
    return  uc->chan[ch].native_codec;
}
/*- End of function --------------------------------------------------------*/

int uc_channel_get_api_codec(uc_t *uc, int ch)
{
    if (ch >= uc->num_channels)
        return -1;
    /*endif*/
    return  uc->chan[ch].api_codec;
}
/*- End of function --------------------------------------------------------*/

int uc_channel_write(uc_t *uc, int ch, uint8_t *buf, int len)
{
    if (ch >= uc->num_channels)
        return  -1;
    /*endif*/
    if (uc->protocol_class < max_protocol_spec
        &&
        protocol_specs[uc->protocol_class].protocol)
    {
        return  protocol_specs[uc->protocol_class].protocol->xx_channel_write(uc, ch, buf, len);
    }
    /*endif*/
    return  -1;
}
/*- End of function --------------------------------------------------------*/

int uc_channel_flush(uc_t *uc, int ch, int which)
{
    if (ch >= uc->num_channels)
        return  UC_RET_BAD_CHANNEL;
    /*endif*/
    if (uc->protocol_class < max_protocol_spec
        &&
        protocol_specs[uc->protocol_class].protocol)
    {
        return  protocol_specs[uc->protocol_class].protocol->xx_channel_flush(uc, ch, which);
    }
    /*endif*/
    return  UC_RET_WRONG_PROTOCOL;
}
/*- End of function --------------------------------------------------------*/

int uc_channel_gains(uc_t *uc, int ch, float rxgain, float txgain)
{
    if (ch >= uc->num_channels)
        return  UC_RET_BAD_CHANNEL;
    /*endif*/
    if (uc->protocol_class < max_protocol_spec
        &&
        protocol_specs[uc->protocol_class].protocol)
    {
        return  protocol_specs[uc->protocol_class].protocol->xx_channel_gains(uc, ch, rxgain, txgain);
    }
    /*endif*/
    return  UC_RET_WRONG_PROTOCOL;
}
/*- End of function --------------------------------------------------------*/

int uc_channel_echo_cancel(uc_t *uc, int ch, int op)
{
    if (ch >= uc->num_channels)
        return  UC_RET_BAD_CHANNEL;
    /*endif*/
    if (uc->protocol_class < max_protocol_spec
        &&
        protocol_specs[uc->protocol_class].protocol)
    {
        return  protocol_specs[uc->protocol_class].protocol->xx_channel_echo_cancel(uc, ch, op);
    }
    /*endif*/
    return  UC_RET_WRONG_PROTOCOL;
}
/*- End of function --------------------------------------------------------*/

int uc_channel_switching(uc_t *uc, int ch, int op, int dest, int parms)
{
    if (ch >= uc->num_channels)
        return  UC_RET_BAD_CHANNEL;
    /*endif*/
    if (uc->protocol_class < max_protocol_spec
        &&
        protocol_specs[uc->protocol_class].protocol)
    {
        return  protocol_specs[uc->protocol_class].protocol->xx_channel_switching(uc, ch, op, dest, parms);
    }
    /*endif*/
    return  UC_RET_WRONG_PROTOCOL;
}
/*- End of function --------------------------------------------------------*/

int uc_set_channel_read_codec_callback(uc_t *uc, int ch, uc_channel_read_codec_callback_t *func, void *user_data)
{
    if (ch >= uc->num_channels)
        return  UC_RET_BAD_CHANNEL;
    /*endif*/
    uc->chan[ch].channel_read_codec = func;
    uc->chan[ch].channel_read_codec_user_data = user_data;
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

int uc_set_channel_write_codec_callback(uc_t *uc, int ch, uc_channel_write_codec_callback_t *func, void *user_data)
{
    if (ch >= uc->num_channels)
        return  UC_RET_BAD_CHANNEL;
    /*endif*/
    uc->chan[ch].channel_write_codec = func;
    uc->chan[ch].channel_write_codec_user_data = user_data;
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

int uc_call_control(uc_t *uc, int op, uc_crn_t crn, void *data)
{
    uc_call_t *call;

    if (uc->protocol_class < max_protocol_spec
        &&
        protocol_specs[uc->protocol_class].protocol)
    {
        if (crn)
        {
            if ((call = uc_findcall(uc, crn)) == NULL)
                return  UC_RET_CRN_NOT_KNOWN;
            /*endif*/
        }
        else
        {
            call = NULL;
        }
        /*endif*/
        return  protocol_specs[uc->protocol_class].protocol->xx_call_control(uc, op, call, data);
    }
    /*endif*/
    return  UC_RET_WRONG_PROTOCOL;
}
/*- End of function --------------------------------------------------------*/

int uc_get_call_state(uc_t *uc, uc_crn_t crn, int *state)
{
    uc_call_t *call;

    if ((call = uc_findcall(uc, crn)) == NULL)
        return  UC_RET_CRN_NOT_KNOWN;
    /*endif*/
    if (state)
        *state = call->state;
    /*endif*/
    return  UC_RET_OK;
}
/*- End of function --------------------------------------------------------*/

uc_t *uc_new(int fe,
             int be,
             const char *protocol,
             const char *variant,
             int mode,
             int outgoing_calls_allowed)
{
    uc_t *uc;
    int protocol_class;

    if ((protocol_class = str2protocolclass(protocol)) < 0)
        return  NULL;
    /*endif*/
    uc = protocol_specs[protocol_class].protocol->xx_new(fe, variant, mode, outgoing_calls_allowed);
    if (uc)
        uc->protocol_class = protocol_class;
    /*endif*/
    return  uc;
}
/*- End of function --------------------------------------------------------*/

uc_callparms_t *uc_new_callparms(uc_callparms_t *parms)
{
    if (parms == NULL)
    {
        /* We need to allocate a new one */
        if ((parms = malloc (sizeof(*parms))) == NULL)
            return  NULL;
        /*endif*/
    }
    /*endif*/
    memset (parms, '\0', sizeof(*parms));
    parms->bear_cap_transfer_cap = UC_TRANS_CAP_SPEECH;
    parms->bear_cap_transfer_mode = UC_ITM_CIRCUIT;
    parms->bear_cap_transfer_rate = UC_TRANS_MODE_64_CIRCUIT;
    parms->userinfo_layer1_protocol = UC_LAYER_1_ULAW;
    parms->user_rate = UC_BEAR_RATE_64KBPS;
    parms->destination_ton = UC_TON_UNKNOWN;
    parms->destination_npi = UC_NPI_UNKNOWN;
    parms->destination_sub_addr_ton = UC_TON_UNKNOWN;
    parms->destination_sub_addr_npi = UC_NPI_UNKNOWN;
    parms->originating_pres = UC_PRES_ALLOWED_USER_NUMBER_NOT_SCREENED;
    parms->originating_ton = UC_TON_UNKNOWN;
    parms->originating_npi = UC_NPI_UNKNOWN;
    parms->originating_sub_addr_ton = UC_TON_UNKNOWN;
    parms->originating_sub_addr_npi = UC_NPI_UNKNOWN;
    parms->redirecting_pres = UC_PRES_NUMBER_NOT_AVAILABLE;
    parms->redirecting_cause = UC_REDIR_UNKNOWN;
    parms->redirecting_ton = UC_TON_UNKNOWN;
    parms->redirecting_npi = UC_NPI_UNKNOWN;
    parms->redirecting_subaddr_ton = UC_TON_UNKNOWN;
    parms->redirecting_subaddr_npi = UC_NPI_UNKNOWN;
    parms->original_called_number_ton = UC_TON_UNKNOWN;
    parms->original_called_number_npi = UC_NPI_UNKNOWN;
    return  parms;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_bear_cap_transfer_cap(uc_callparms_t *parms, int cap)
{
    parms->bear_cap_transfer_cap = cap;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_bear_cap_transfer_mode(uc_callparms_t *parms, int mode)
{
    parms->bear_cap_transfer_mode = mode;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_bear_cap_transfer_rate(uc_callparms_t *parms, int rate)
{
    parms->bear_cap_transfer_rate = rate;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_userinfo_layer1_protocol(uc_callparms_t *parms, int prot)
{
    parms->userinfo_layer1_protocol = prot;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_user_rate(uc_callparms_t *parms, int rate)
{
    parms->user_rate = rate;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_destination_ton(uc_callparms_t *parms, int ton)
{
    parms->destination_ton = ton;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_destination_npi(uc_callparms_t *parms, int npi)
{
    parms->destination_npi = npi;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_destination_number(uc_callparms_t *parms, char *num)
{
    if (num)
        strncpy(parms->destination_number, num, sizeof(parms->destination_number) - 1);
    else
        parms->destination_number[0] = '\0';
    /*endif*/
    parms->destination_number[sizeof(parms->destination_number) - 1] = '\0';
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_destination_sub_addr_ton(uc_callparms_t *parms, int ton)
{
    parms->destination_sub_addr_ton = ton;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_destination_sub_addr_npi(uc_callparms_t *parms, int npi)
{
    parms->destination_sub_addr_npi = npi;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_destination_sub_addr_number(uc_callparms_t *parms, char *num)
{
    if (num)
        strncpy(parms->destination_sub_addr_number, num, sizeof(parms->destination_sub_addr_number) - 1);
    else
        parms->destination_sub_addr_number[0] = '\0';
    /*endif*/
   parms->destination_sub_addr_number[sizeof(parms->destination_sub_addr_number) - 1] = '\0';
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_originating_presentation(uc_callparms_t *parms, int pres)
{
    parms->originating_pres = pres;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_originating_ton(uc_callparms_t *parms, int ton)
{
    parms->originating_ton = ton;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_originating_npi(uc_callparms_t *parms, int npi)
{
    parms->originating_npi = npi;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_originating_number(uc_callparms_t *parms, char *num)
{
    if (num)
        strncpy(parms->originating_number, num, sizeof(parms->originating_number) - 1);
    else
        parms->originating_number[0] = '\0';
    /*endif*/
    parms->originating_number[sizeof(parms->originating_number) - 1] = '\0';
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_originating_sub_addr_ton(uc_callparms_t *parms, int ton)
{
    parms->originating_sub_addr_ton = ton;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_originating_sub_addr_npi(uc_callparms_t *parms, int npi)
{
    parms->originating_sub_addr_npi = npi;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_originating_sub_addr_number(uc_callparms_t *parms, char *num)
{
    if (num)
        strncpy(parms->originating_sub_addr_number, num, sizeof(parms->originating_sub_addr_number) - 1);
    else
        parms->originating_sub_addr_number[0] = '\0';
    /*endif*/
    parms->originating_sub_addr_number[sizeof(parms->originating_sub_addr_number) - 1] = '\0';
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_redirecting_cause(uc_callparms_t *parms, int cause)
{
    parms->redirecting_cause = cause;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_redirecting_presentation(uc_callparms_t *parms, int pres)
{
    parms->redirecting_pres = pres;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_redirecting_ton(uc_callparms_t *parms, int ton)
{
    parms->redirecting_ton = ton;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_redirecting_npi(uc_callparms_t *parms, int npi)
{
    parms->redirecting_npi = npi;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_redirecting_number(uc_callparms_t *parms, char *num)
{
    if (num)
        strncpy(parms->redirecting_number, num, sizeof(parms->redirecting_number) - 1);
    else
        parms->redirecting_number[0] = '\0';
    /*endif*/
    parms->redirecting_number[sizeof(parms->redirecting_number) - 1] = '\0';
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_redirecting_subaddr_ton(uc_callparms_t *parms, int ton)
{
    parms->redirecting_subaddr_ton = ton;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_redirecting_subaddr_npi(uc_callparms_t *parms, int npi)
{
    parms->redirecting_subaddr_npi = npi;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_redirecting_subaddr(uc_callparms_t *parms, char *num)
{
    if (num)
        strncpy(parms->redirecting_subaddr, num, sizeof(parms->redirecting_number) - 1);
    else
        parms->redirecting_subaddr[0] = '\0';
    /*endif*/
    parms->redirecting_subaddr[sizeof(parms->redirecting_subaddr) - 1] = '\0';
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_original_called_number_ton(uc_callparms_t *parms, int ton)
{
    parms->original_called_number_ton = ton;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_original_called_number_npi(uc_callparms_t *parms, int npi)
{
    parms->original_called_number_npi = npi;
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_original_called_number(uc_callparms_t *parms, char *num)
{
    if (num)
        strncpy(parms->original_called_number, num, sizeof(parms->original_called_number) - 1);
    else
        parms->original_called_number[0] = '\0';
    /*endif*/
    parms->original_called_number[sizeof(parms->original_called_number) - 1] = '\0';
}
/*- End of function --------------------------------------------------------*/

void uc_callparm_calling_party_category(uc_callparms_t *parms, int category)
{
    parms->calling_party_category = category;
}
/*- End of function --------------------------------------------------------*/

void uc_set_logging(uc_t *uc, int level, int format, const char *tag)
{
    uc->logging_level = level;
    uc->logging_format = format;
    if (tag)
        uc->logging_tag = strdup(tag);
    else if (uc->logging_tag == NULL)
        uc->logging_tag = "";
    /*endif*/
}
/*- End of function --------------------------------------------------------*/

void uc_set_message_handler(void (*func)(char *text))
{
    __uc_message = func;
}
/*- End of function --------------------------------------------------------*/

void uc_set_error_handler(void (*func)(char *text))
{
    __uc_error = func;
}
/*- End of function --------------------------------------------------------*/

void uc_message(char *fmt, ...)
{
    char tmp[1024];
    va_list ap;

    va_start(ap, fmt);
    vsnprintf(tmp, sizeof(tmp), fmt, ap);
    va_end(ap);
    if (__uc_message)
        __uc_message(tmp);
    else
        fprintf(stdout, tmp);
    /*endif*/
}
/*- End of function --------------------------------------------------------*/

void uc_error(char *fmt, ...)
{
    char tmp[1024];
    va_list ap;

    va_start(ap, fmt);
    vsnprintf(tmp, sizeof(tmp), fmt, ap);
    va_end(ap);
    if (__uc_error)
        __uc_error(tmp);
    else
        fprintf(stderr, tmp);
    /*endif*/
}
/*- End of function --------------------------------------------------------*/

uc_event_t *uc_check_event(uc_t *uc)
{
    uc_event_t *ev;

    if (uc->protocol_class < max_protocol_spec
        &&
        protocol_specs[uc->protocol_class].protocol)
    {
        if (uc->signaling_func)
        {
            while (uc_get_next_event(uc, &ev) == 0)
            {
                uc->signaling_func(uc, uc->signaling_user_data, ev);
                uc_release_event(uc, ev);
            }
            /*endwhile*/
        }
        else
        {
            if (uc_get_next_event(uc, &ev) == 0)
                return ev;
            /*endwhile*/
        }
        /*endif*/
        protocol_specs[uc->protocol_class].protocol->xx_check_event(uc);
        if (uc->signaling_func)
        {
            while (uc_get_next_event(uc, &ev) == 0)
            {
                uc->signaling_func(uc, uc->signaling_user_data, ev);
                uc_release_event(uc, ev);
            }
            /*endwhile*/
        }
        else
        {
            if (uc_get_next_event(uc, &ev) == 0)
                return ev;
            /*endwhile*/
        }
        /*endif*/
    }
    /*endif*/
    return  NULL;
}
/*- End of function --------------------------------------------------------*/

int uc_start(void)
{
    void *lib;
    Dl_info info;

    if (uc_initialised)
        return  0;
    /*endif*/
    /* We need to find the name of this .so file, so we can globalise its
       symbols. Some of these will be needed by the protocol modules. */
    if (dladdr(uc_start, &info) == 0)
        return  -1;
    /*endif*/
    /* Now globalise them */
    if ((lib = dlopen(info.dli_fname, RTLD_NOW | RTLD_GLOBAL)) == NULL)
        return  -1;
    /*endif*/
    dlclose(lib);
    uc_initialised = TRUE;
    return  0;
}
/*- End of function --------------------------------------------------------*/

static void uc_call_init(uc_call_t *call)
{
    memset(call, 0, sizeof(*call));
    call->alive = 0;
    call->sent_disconnect = 0;
    call->force_invert = -1;    
    call->crn = -1;
    call->slotmap = -1;
    call->channelno = -1;
    call->ds1no = -1;
    call->chanflags = 0;
    call->sentchannel = 0;
    call->chan = -1;
}
/*- End of function --------------------------------------------------------*/

#if 0
static void uc_scan_calls(uc_t *uc)
{
    hash_search_t search;
    hash_entry_t *entry;
    uc_call_t *uc_call;
    int *key;
    
    for (entry = hash_first_entry(&uc->calls, &search);
         entry;
         entry = hash_next_entry(&search))
    {
        uc_call = (uc_call_t *) hash_get_value(entry);
        key = (int *) hash_get_key(&uc->calls, entry);
        uc_log(uc, UC_LOG_WARNING, "Found CRN %d/%d\n", uc_call->crn, *key);
    }
    /*endfor*/
}
/*- End of function --------------------------------------------------------*/
#endif

uc_call_t *uc_findcall(uc_t *uc, int crn)
{
    hash_entry_t *entry;

    if ((entry = hash_find_entry(&uc->calls, (const void *) (intptr_t) crn)) == NULL)
        return  NULL;
    /*endif*/
    return  (uc_call_t *) hash_get_value(entry);
}
/*- End of function --------------------------------------------------------*/

uc_call_t *uc_getcall(uc_t *uc, int crn, int extra)
{
    hash_entry_t *entry;
    uc_call_t *call;
    int new;

    entry = hash_create_entry(&uc->calls, (const void *) (intptr_t) crn, &new);
    if (!new)
        return  (uc_call_t *) hash_get_value(entry);
    /*endif*/

    /* No call exists with this CRN. Make a new one */
    uc_log(uc, UC_LOG_FLOW, "Making a new call with CRN %d\n", crn);
    if ((call = malloc(sizeof(uc_call_t) + extra)) == NULL)
        return  NULL;
    /*endif*/
    uc_call_init(call);
    call->crn = crn;
    /* Add it to the list */
    hash_set_value(entry, call);
    return  call;
}
/*- End of function --------------------------------------------------------*/

uc_call_t *uc_createcall(uc_t *uc, int extra)
{
    int crn;

    /* Allocate a unique call reference number. Use a generally sequential
       number, but check for clashes with old stuff. Long calls might genuinely
       leave a CRN in use for a whole cycle of the counter. */
    do
    {
        if (++uc->next_crn > 32767)
            uc->next_crn = 1;
        /*endif*/
        crn = uc->next_crn | 0x8000;
    }
    while (hash_find_entry(&uc->calls, (const void *) (intptr_t) crn));
    return  uc_getcall(uc, crn, extra);
}
/*- End of function --------------------------------------------------------*/

void uc_destroycall(uc_t *uc, int crn)
{
    hash_entry_t *entry;

    uc_log(uc, UC_LOG_FLOW, "Destroying call with CRN %d\n", crn);
    if ((entry = hash_find_entry(&uc->calls, (const void *) (intptr_t) crn)) == NULL)
    {
        uc_error("Can't find CRN %d to destroy its call!\n", crn);
        return;
    }
    /*endif*/
    free((uc_call_t *) hash_get_value(entry));
    hash_delete_entry(entry);
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
