/*
 * libmfcr2 - a library for MFC/R2 signaling on E1 lines
 *
 * mfcr2.h
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2002, 2003 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: mfcr2.h,v 1.24 2005/06/21 14:36:33 steveu Exp $
 */

#if !defined(_MFCR2_H_)
#define _MFCR2_H_

/* Welcome to the weird and wonderful world of MFC/R2 signaling */

/* MFC/R2 is specified by the ITU. MFC/R2 is not specified by the ITU.
   What does this mean? It means there are a bunch of R2 specs from the
   ITU, but all they do is define an outline of how things will work.
   Large scale national variations on the theme are used. Bascially,
   every nation that is dumb enough to keep using something as clunky
   as R2 is dumb enough to make themselves incompatible with the rest of
   the universe (a job creation scheme, maybe). */
   
/* R2 is rooted in the 1950s. It suited the needs of its day well - speeding
   up, and adding some bells and whistles to the pulse signaling then used
   on copper pairs. In the E1 PCM world it works like this.....

   A combination of wiggling the E1 CAS ABCD bits and dual tone signaling is
   used. The tone signaling is limited to the phases of a call up to the
   answer/connected state. The bit wiggling controls the rest (seize/clear/
   blocking/etc.
   
   The dual tone signaling is unlike DTMF. Here there are six forward direction
   tones, and six backward direction tone. Any two out of six tones represents
   a valid signal, and there are 15 combinations of two out of six - 15 valid
   signals. Using two separate groups of tones allowed the system to work
   without echo cancellation in the old days (in the 1950s echo cancellation
   was not technically feasible, and only crude echo supression was ever used).
   These days the use of two separate tone groups is largely irrelevant.
   
   Call processing goes something like this.....

   The calling end seizes the line by wiggling the E1 CAS ABCD bits in some
   nationally dependant manner. It also puts the tone pair for the first
   signal on the line.

   When the called end recognises the tone pair, it responds with its own
   tone pair.

   When the calling end recognises this it stops its tone pair.

   When the called end recognises the silence condition it stops its tone pair.
   
   When the calling end recognises the silence condition is starts sending
   its next tone pair.
   
   This proceeds until all information (ANI, DNIS, and a few optional bells
   and whistles) are exchanged. After that the rest of the call signaling
   (answer/clear down) is handled by more wiggling of the ABCD bits.
   
   One thing to note about MFC/R2: In most places an MFC/R2 is either incoming
   only or outgoing only. Only a few places support both-ways operation on
   an MFC/R2 E1, and this may have some impact on the code as support for such
   variants is added.
 */

/*
    Some China specific signals
    ---------------------------

    Calling party category group I (KA) signals

         Step by step switch           Crossbar or electronic switch
         -------------------           -----------------------------
    1  - regular, fixed delay          voice/fax/data, fixed delay
    2  - regular, immediate, meter     voice/fax/data, immediate, meter
    3  - regular, business             voice/fax/data, business
    4  - priority #1                   voice/fax/data, priority #1
    5  - free                          free
    6  - small PBX                     small PBX
    7  - priority #1, fixed delay      voice/fax/data, priority #1, fixed delay
    8  - priority #2, fixed delay      voice/fax/data, priority #2, fixed delay
    9  - privileged local              voice/fax/data, priority #1, business
    10 - non-privileged local          free
    11 - spare                         spare
    12 - spare                         spare
    13 - test                          test
    14 - spare                         spare
    15 - spare                         spare

    Calling party category group II (KD) signals (typical values)
    
    1  - Long distance operator, semi-automatic
    2  - Long distance, automatic switching
    3  - Local call, voice
    4  - Local call, fax and data
    5  - Semi-automatic, verify calling party's number
    6  - Test call
*/

/* There are only 15 real signals, but the ITU specs. give them different
   names in different contexts, so lets do the same. That was there is some
   correlation with the specs. */
   
/* R2 group I forward signals */
/* The meaning of these signals depends whether they are the first signal on
   an international circuit or not. */
#define R2_SIGI_1   '1'     /* First: Language digit - French */
                            /* Other: Digit 1 */
#define R2_SIGI_2   '2'     /* First: Language digit - English */
                            /* Other: Digit 2 */
#define R2_SIGI_3   '3'     /* First: Language digit - German */
                            /* Other: Digit 3 */
#define R2_SIGI_4   '4'     /* First: Language digit - Russian */
                            /* Other: Digit 4 */
#define R2_SIGI_5   '5'     /* First: Language digit - Spanish */
                            /* Other: Digit 5 */
#define R2_SIGI_6   '6'     /* First: Spare (language digit) */
                            /* Other: Digit 6 */
#define R2_SIGI_7   '7'     /* First: Spare (language digit) */
                            /* Other: Digit 7 */
#define R2_SIGI_8   '8'     /* First: Spare (language digit) */
                            /* Other: Digit 8 */
#define R2_SIGI_9   '9'     /* First: Spare (discriminating digit) */
                            /* Other: Digit 9 */
#define R2_SIGI_10  '0'     /* First: Discriminating digit */
                            /* Other: Digit 0 */
#define R2_SIGI_11  'A'     /* First: Outgoing half echo suppressor required */
                            /* Other: Operator code 11 */
#define R2_SIGI_12  'B'     /* First: No echo suppressor required */
                            /* Other: Operator code 12/request not accepted */
#define R2_SIGI_13  'C'     /* First: Test call indicator (call by ATE) */
                            /* Other: Code 13 (call to ATE) */
#define R2_SIGI_14  'D'     /* First: Incoming half echo suppressor required */
                            /* Other: Incoming half echo suppressor required */
#define R2_SIGI_15  'E'     /* First: Signal not used */
                            /* Other: End of pulsing (code 15) */

/* R2 group II forward signals */
#define R2_SIGII_1  '1'     /* National - subscriber initiated call */
#define R2_SIGII_2  '2'     /* National - subscriber with priority call */
#define R2_SIGII_3  '3'     /* National - maintenance equipment call */
#define R2_SIGII_4  '4'     /* National - spare */
#define R2_SIGII_5  '5'     /* National - operator initiated call */
#define R2_SIGII_6  '6'     /* National - data transmission call */
#define R2_SIGII_7  '7'     /* International - Subscriber (or operator without
                               forward transfer facility) */
#define R2_SIGII_8  '8'     /* International - data transmission call */
#define R2_SIGII_9  '9'     /* International - subscriber with priority */
#define R2_SIGII_10 '0'     /* International - operator with forward transfer
                               facility */
#define R2_SIGII_11 'A'     /* Spare for national use */
#define R2_SIGII_12 'B'     /* Spare for national use */
#define R2_SIGII_13 'C'     /* Spare for national use */
#define R2_SIGII_14 'D'     /* Spare for national use */
#define R2_SIGII_15 'E'     /* Spare for national use */

/* R2 group A backward signals */
#define R2_SIGA_1   '1'     /* Send next digit (DNIS) (n+1) */
#define R2_SIGA_2   '2'     /* Send last but one digit (DNIS) (n-1) */
#define R2_SIGA_3   '3'     /* Address complete. Change to group B signals */
#define R2_SIGA_4   '4'     /* Congestion in national network */
#define R2_SIGA_5   '5'     /* Send calling party's category (+ ANI) */
#define R2_SIGA_6   '6'     /* Address complete. Charge. Set-up speech conditions */
#define R2_SIGA_7   '7'     /* Send last but two digits (DNIS) (n-2) */
#define R2_SIGA_8   '8'     /* Send last but three digits (DNIS) (n-3) */
#define R2_SIGA_9   '9'     /* Spare for national use */
#define R2_SIGA_10  '0'     /* Spare for national use */
#define R2_SIGA_11  'A'     /* Send country code indicator */
#define R2_SIGA_12  'B'     /* Send language or discriminating digit */
#define R2_SIGA_13  'C'     /* Send location of outgoing R2 international register */
#define R2_SIGA_14  'D'     /* Is an incoming half echo suppressor required? */
#define R2_SIGA_15  'E'     /* International exchange congestion */

/* R2 group B backward signals */
#define R2_SIGB_1   '1'     /* Spare for national use */
#define R2_SIGB_2   '2'     /* Subscriber transferred - send SIT to the caller */
#define R2_SIGB_3   '3'     /* Subscriber line busy */
#define R2_SIGB_4   '4'     /* Congestion (encountered after change from
                               group A to group B). */
#define R2_SIGB_5   '5'     /* Unassigned number */
#define R2_SIGB_6   '6'     /* Subscriber line free - charge */
#define R2_SIGB_7   '7'     /* Subscriber line free - no charge */
#define R2_SIGB_8   '8'     /* Subscriber line out of order */
#define R2_SIGB_9   '9'     /* Spare for national use */
#define R2_SIGB_10  '0'     /* Spare for national use */
#define R2_SIGB_11  'A'     /* Spare for national use */
#define R2_SIGB_12  'B'     /* Spare for national use */
#define R2_SIGB_13  'C'     /* Spare for national use */
#define R2_SIGB_14  'D'     /* Spare for national use */
#define R2_SIGB_15  'E'     /* Spare for national use */

enum
{
    /* Call states for R2 protocol inbound (backward) */
    /*! Sending SEIZE ACK in response to SEIZE */
    R2_BACK_SEIZE_ACK                           = 101,
    /*! Sending group A signals */
    R2_BACK_GROUP_A                             = 102,
    /*! Sending group B signals */
    R2_BACK_GROUP_B                             = 103,
    /*! Sending group C signals (only used in a couple of countries) */
    R2_BACK_GROUP_C                             = 104,
    /*! Sending ANSWER signal, after receiving SIGII_1 tone off */
    R2_BACK_ANSWER                              = 105,
    /*! Sending CLEAR BACK after ANSWER */
    R2_BACK_CLEAR_BACK                          = 106,
    /*! Got CLEAR FWD before CLEAR BACK */
    R2_BACK_CLEAR_FWD                           = 107,
    /*! Sending timed RELEASE GUARD to prevent early re-seize from our side. */
    R2_BACK_RELEASE_GUARD                       = 108,

    /* Call states for R2 protocol outbound (forward) */
    /*! Sending SEIZE */
    R2_FWD_SEIZE                                = 201,
    /*! Proceeding with group I signals */
    R2_FWD_GROUP_I                              = 202,
    /*! Proceeding with group II signals */
    R2_FWD_GROUP_II                             = 203,
    /*! Proceeding with group III signals (only used in a couple of countries) */
    R2_FWD_GROUP_III                            = 204,
    /*! Waiting for CAS ANSWER after group B signal received */
    R2_FWD_AWAIT_ANSWER                         = 205,
    /*! CAS ANSWER has preceeded group B signal */
    R2_FWD_ANSWERED_BEFORE_GROUP_B_SIGNAL       = 206,
    /*! The call is fully answered (group B tone + CAS bits) */
    R2_FWD_ANSWERED                             = 207,
    /*! CLEAR BACK after ANSWER */
    R2_FWD_CLEAR_BACK                           = 208,
    /*! Want to send CLEAR FWD before receiving SEIZE ACK */
    R2_FWD_CLEAR_FWD_BEFORE_SEIZE_ACK           = 209,
    /*! Sending CLEAR FWD before receiving ANSWER */
    R2_FWD_CLEAR_FWD_BEFORE_ANSWER              = 210,
    /*! Sending CLEAR FWD after receiving ANSWER */
    R2_FWD_CLEAR_FWD_AFTER_ANSWER               = 211,
    /*! Sending CLEAR FWD after receiving CLEAR BACK. Waiting for RELEASE GUARD */
    R2_FWD_CLEAR_FWD_AFTER_CLEAR_BACK           = 212
} r2_phase_e;

/* Now some definitions for the states of the R2 protocol level state machine */
enum
{
    /* Group A states for R2 protocol inbound (backward). */
    R2_SENT_SEIZE_ACK                           = 301,
    /* Asking for calling party's category. */
    R2_SENT_CATEGORY_REQUEST                    = 302,
    /*! Asking for the next DNIS digit. */
    R2_SENT_DNIS_REQUEST                        = 303,
    /*! Asking for next ANI digit. */
    R2_SENT_ANI_REQUEST                         = 304,
    /*! Asking for change to group II. */
    R2_SENT_CHANGE_TO_GROUP_II                  = 305,

    /* Group B states for R2 protocol inbound (backward) */
    /*! Sent numbers complete and start ANSWER. Charge. */
    R2_SENT_ACCEPTED_CHARGE                     = 306,
    /*! Sent numbers complete and start ANSWER. No charge. */
    R2_SENT_ACCEPTED_NO_CHARGE                  = 307,
    /*! Sent numbers complete. Route congested. */
    R2_SENT_NETWORK_CONGESTION                  = 308,
    /*! Sent numbers complete. Destination number unassigned. */
    R2_SENT_UNASSIGNED_NUMBER                   = 309,
    /*! Sent numbers complete. Destination number busy. */
    R2_SENT_DEST_BUSY                           = 310,
    /*! Sent numbers complete. Destination number out of order. */
    R2_SENT_DEST_OUT_OF_ORDER                   = 311,

    /* Group I states for R2 protocol outbound (forward) */
    R2_SENT_CATEGORY                            = 401,
    R2_SENT_DNIS                                = 402,
    R2_SENT_ANI                                 = 403,
    R2_SENT_END_OF_DNIS                         = 404,
    R2_SENT_END_OF_ANI                          = 405,
    R2_SENT_NOTHING                             = 406
} r2_state_e;

enum
{
    MFCR2_PROT_ALGERIA = 1,
    MFCR2_PROT_ARGENTINA,
    MFCR2_PROT_BAHRAIN,
    MFCR2_PROT_BOLIVIA,
    MFCR2_PROT_BRAZIL,
    MFCR2_PROT_CHILE,
    MFCR2_PROT_CHINA,
    MFCR2_PROT_COLOMBIA_LAND,
    MFCR2_PROT_COLOMBIA_CELL,
    MFCR2_PROT_CZECH,
    MFCR2_PROT_DEMO_CONGO,
    MFCR2_PROT_EGYPT,
    MFCR2_PROT_GHANA,
    MFCR2_PROT_HONDURAS,
    MFCR2_PROT_INDIA,
    MFCR2_PROT_INDONESIA,
    MFCR2_PROT_IRAQ,
    MFCR2_PROT_ITU,
    MFCR2_PROT_KOREA,
    MFCR2_PROT_KUWAIT,
    MFCR2_PROT_MALAYSIA,
    MFCR2_PROT_MEXICO,
    MFCR2_PROT_NIGERIA,
    MFCR2_PROT_PANAMA,
    MFCR2_PROT_PHILIPINNES,
    MFCR2_PROT_ROMANIA,
    MFCR2_PROT_SAUDI_ARABIA,
    MFCR2_PROT_SINGAPORE,
    MFCR2_PROT_THAILAND,
    MFCR2_PROT_VENEZUELA,
    MFCR2_PROT_VIETNAM
};

/* This structure should contain everything to control an R2 channel. */
struct mfcr2_signaling_state_s
{
    /* Configuration parameters, to adapt to different countries. */
    /*! \brief The number of DNIS digits we expect for incoming calls. Some national
               variants do not handle variable length dialed digit strings well. In
               those cases it is important to set this correctly. For other variants,
               a large number is OK. */
    int dnis_max_rx_digits;
    /*! \brief The number of ANI digits we expect for incoming calls. */
    int ani_max_rx_digits;
    /*! \brief TRUE to play progress tones for busy, NU, congested, etc. */
    int play_progress_tones;
    /*! \brief TRUE to play the disconnect tone after the local end has disconnected. */
    int play_disconnect_tone;
    /*! \brief TRUE to play the ringback tone while waiting for answer. */
    int play_ringback_tone;
    /*! \brief TRUE to use immediate accept (i.e. skipping the group B/II stage). */
    int use_immediate_accept;
    /*! \brief TRUE to get the ANI digits after the DNIS is complete. FALSE to get ANI after
               the first DNIS digit. */
    int get_ani_after_dnis;
    /*! \brief Forward tone max. on time, in milliseconds */
    int t1;
    /*! \brief Backward pulse on time (non-compelled), in milliseconds */
    int t1a;
    /*! \brief Forward side max wait for the far end to timeout and recover, when
               the end of digits is indicated by no reponse from our end. In
               milliseconds. */
    int t1b;
    /*! \brief Forward tone max. off time, in milliseconds */
    int t2;
    /*! \brief Inbound side compelled cycle max. time, in milliseconds */
    int t3;
    /*! \brief */
    int blocking_release_time;
    /*! \brief */
    int answer_guard_time;
    /*! \brief Zero may be OK */
    int release_guard_time;
    /*! \brief 20ms should be OK */
    int inbound_cas_persistence_check;
    /*! \brief 20ms should be OK */
    int outbound_cas_persistence_check;
    /*! ITU spec says 200ms */
    int max_seize_ack_wait;
    /*! \brief The maximum time we will for the called party to answer, after the
               call setup is complete. 60s is common practice */
    int max_await_answer;
    /*! \brief The maximum time we will wait for the group B signal. 15s is common
               practice */
    int max_wait_for_group_b_signal;
    /*! \brief Clear back persistence check, in milliseconds. This can be zero for
               most purposes */
    int clear_back_persistence_check;
    /*! \brief All trunks are able to accept incoming calls. This flag is TRUE if the
               trunk is permitted to also make outgoing calls. */
    int outgoing_calls_allowed;

    /*! \brief Mask to pick out the R2 signaling bits from amongst the ABCD bits. */
    uint8_t abcd_mask;
    /*! \brief The value to use on transmission for the ABCD bits which are not the
               R2 signaling bits. */
    uint8_t base_abcd_bits;
    /*! \brief The idle pattern for the ABCD bits */
    uint8_t abcd_idle;
    /*! \brief The blocking pattern for the ABCD bits */
    uint8_t abcd_blocking;

    /*! \brief The seize pattern for the ABCD bits */
    uint8_t fwd_abcd_seize;
    /*! \brief The clear forward pattern for the ABCD bits */
    uint8_t fwd_abcd_clear_fwd;

    /*! \brief The seize acknowledge pattern for the ABCD bits */
    uint8_t back_abcd_seize_ack;
    /*! \brief The answer pattern for the ABCD bits */
    uint8_t back_abcd_answer;
    /*! \brief The clear back pattern for the ABCD bits */
    uint8_t back_abcd_clear_back;

    uint8_t abcd_billing_pulse;
    int billing_pulse_duration;

    /* Group I signals. */
    /*! \brief This is 0 (not supported) in some countries. */
    uint8_t group_i_end_of_DNIS;
    /*! \brief This is 0 (not supported) in some countries. */
    uint8_t group_i_end_of_ANI;
    /*! \brief This is 0 (not supported) in some countries. */
    uint8_t group_i_end_of_ANI_restricted;

    /* Group II signals. */
    uint8_t group_ii[10];

    /* Group A signals. */
    /*! \brief Request the next digit of the DNIS
        
        If no more DNIS is available the outgoing side may
        send an end of DNIS signal, or remain quiet until a
        timeout. This depends on the variant of MFC/R2 in use.
    */
    uint8_t group_a_send_next_DNIS_digit;
    /*! \brief Request the last digit of the DNIS be repeated. */
    uint8_t group_a_send_previous_DNIS_digit;
    /*! \brief This is the end of the transfer of numeric information. */
    uint8_t group_a_switch_to_group_ii;
    /*! \brief Congestion has been detected. */
    uint8_t group_a_network_congestion;
    /*! \brief Send the calling party category. */
    uint8_t group_a_send_calling_party_category;
    /*! \brief Immediately accept the call, without the group B/II step. */
    uint8_t group_a_immediate_accept;
    /*! \brief Request the DNIS digit 2 digits back be repeated.
    
        This is an ITU specified feature, but a number of variants
        do not support it.
    */
    uint8_t group_a_send_DNIS_digit_n_minus_2;
    /*! \brief Request the DNIS digit 3 digits back be repeated.
    
        This is an ITU specified feature, but a number of variants
        do not support it.
    */
    uint8_t group_a_send_DNIS_digit_n_minus_3;
    /*! \brief Restart sending DNIS from the first digit.
    
        This is not an ITU specified feature, but a number of
        countries use it.
    */
    uint8_t group_a_repeat_all_DNIS_digits;
    /*! \brief The group A signal to request the next ANI digit. */
    uint8_t group_a_send_next_ANI_digit;
    /*! \brief The group A signal to switch to group C (only used in a
        couple of countries). */
    uint8_t group_a_send_calling_party_category_and_switch_to_group_c;    

    /* Group B signals. */
    /*! \brief The group B signal to send SIT. */
    uint8_t group_b_send_special_information_tone;
    /*! \brief The group B signal to indicate the called line is busy. */
    uint8_t group_b_user_busy;
    /*! \brief The group B signal to indicate the called line cannot be reached, due to
               network congestion. */
    uint8_t group_b_network_congestion;
    /*! \brief The group B signal to indicate the called number is an unassigned number. */
    uint8_t group_b_unassigned_number;
    /*! \brief The group B signal to indicate the called line is free, and a charged call
               will occur if the called party answers. */
    uint8_t group_b_line_free_charge;
    /*! \brief The group B signal to indicate the called line is free, and a free call
               will occur if the called party answers. */
    uint8_t group_b_line_free_no_charge;
    /*! \brief The group B signal to indicate the called line is out of order. */
    uint8_t group_b_line_out_of_order;
    
    /* Group C signals (only used in a couple of countries). */
    uint8_t group_c_send_next_ANI_digit;
    uint8_t group_c_repeat_all_DNIS_digits_and_switch_to_group_a;
    uint8_t group_c_send_next_DNIS_digit_and_switch_to_group_a;
    uint8_t group_c_send_previous_DNIS_digit_and_switch_to_group_a;
    uint8_t group_c_switch_to_group_ii;

    /* Special things for some MFC/R2 variants */
    /*! \brief This flag determines is a timeout waiting for a DNIS digit on an
               incoming call should be treated as the end of the DNIS. */
    int group_b_on_DNIS_timeout;

    /*! \brief The current call information. */
    uc_call_t *call;
    
    /*! \brief Alarms */
    int alarms;

    /*! \brief The current phase of the R2 protocol. */
    int r2_phase;
    /*! \brief The current compelled signalling state of the R2 protocol. */
    int r2_state;
    /*! \brief A flag used to differentiate calling party category requests from
               ANI digit requests. They use the same tone in many places. */
    int calling_party_category_exchanged;
    /*! \brief A flag which indicates all DNIS digits have been received. */
    int dnis_complete;
    /*! \brief The total length of the DNIS string for outgoing calls. A progressively
               incrementing count of the DNIS digits for incoming calls. */
    int dnis_len;
    /*! \brief The total length of the ANI string for outgoing calls. A progressively
               incrementing count of the ANI digits for incoming calls. */
    int ani_len;
    /*! \brief The DNIS digits sent so far for outgoing calls. Not used for incoming calls. */
    int dnis_cnt;
    /*! \brief The ANI digits sent so far for outgoing calls. Not used for incoming calls. */
    int ani_cnt;
    /*! \brief The ANI is restricted. */
    int ani_restricted;
    /*! \brief The current CAS ABCD pattern being received. */
    uint8_t current_abcd_in;
    /*! \brief Timer used to ensure we do not seize to aggressively as the far end unblocks. */
    int far_unblocking_timer;
    /*! \brief Timer used to ensure we do not seize to aggressively after we unblock. */
    int local_unblocking_timer;
    /*! \brief The seize acknowledge timeout, used to detect failure at the remote end. */
    int seize_ack_timer;
    /*! \brief The release guard timeout. */
    int release_guard_timer;
    /*! \brief Optional minimum delay between sending the group B signal and
        answering.
    
        If we answer too soon after sending the group B signal, the
        outgoing side might see the answer before its tone detector has
        found the group B signal. This timer allows some delay to be
        introduced, to prevent such overlap conditions. */
    int answer_guard_timer;
    /*! \brief A timer to handle the possibility that the called party never answers. */
    int await_answer_timer;
    /*! \brief A timer to handle the possibility that the group B signal never arrives. */
    int wait_for_group_b_signal_timer;
    /*! \brief Clear back persistence timer, used to differentiate between metering pulses
        and real clear back conditions. */
    int clear_back_persistence_timer;
    /*! \brief Timer for the duration of a billing pulse. */
    int billing_pulse_timer;
    /*! \brief The MF tone on timer, at the outgoing side. */
    int t1_timer;
    /*! \brief The MF tone pulse timer, at the incoming side. This is only used
               for variants which use non-compelled MF6 tone pulses to restart
               activities after certain timeouts. */
    int t1a_timer;
    /*! \brief A backstop timer, to ensure things do not get stuck when
               an outgoing call is relying on the other end moving on
               when a timeout occurs. */
    int t1b_timer;
    /*! \brief The MF tone off timer, at the outgoing side. */
    int t2_timer;
    /*! \brief The compelled cycle timer, at the incoming side. */
    int t3_timer;
    /*! \brief TRUE is the far end is in the blocked state. */
    int far_blocked;
    /*! \brief TRUE is the local end is in the blocked state. */
    int local_blocked;
    /*! \brief The MF receiver context. */
    r2_mf_rx_state_t mf_rx_state;
    /*! \brief The MF transmitter context. */
    tone_gen_state_t mf_gen_state;
    /*! \brief The current MF signal being received. Zero means no signal being received. */
    int mf_rx_signal;
    /*! \brief The current MF signal being sent. Zero means no signal being sent. */
    int mf_tx_signal;
    /*! \brief Supervisory tone generator context. */
    super_tone_tx_state_t super_tone_tx_state;
    /*! \brief Supervisory tone set. */
    super_tone_set_t *super_tones;

    /*! \brief Debug audio receive log file handle. */
    int audio_rx_log;
    /*! \brief Debug audio transmit log file handle. */
    int audio_tx_log;
};

#endif
