/*
 * An implementation of hash tables based on a section of TCL which is:
 *
 * Copyright (c) 1991-1993 The Regents of the University of California.
 * Copyright (c) 1994 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * $Id: hashtable.c,v 1.4 2005/07/11 13:16:30 steveu Exp $
 */

#include <stdlib.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include "unicall/hashtable.h"

/*
 * When there are this many entries per bucket, on average, rebuild
 * the hash table to make it larger.
 */
#define REBUILD_MULTIPLIER        3

/*
 * The following macro takes a preliminary integer hash value and
 * produces an index into a hash tables bucket list.  The idea is
 * to make it so that preliminary values that are arbitrarily similar
 * will end up in different buckets.  The hash function was taken
 * from a random-number generator.
 */
#define RANDOM_INDEX(tablePtr, i) \
    (((((long) (i))*1103515245) >> (tablePtr)->downShift) & (tablePtr)->mask)

/* Procedure prototypes for static procedures in this file: */
static hash_HashEntry_t *ArrayFind(hash_HashTable_t *tablePtr, const char *key);
static hash_HashEntry_t *ArrayCreate(hash_HashTable_t *tablePtr, const char *key, int *newPtr);
static hash_HashEntry_t *BogusFind(hash_HashTable_t *tablePtr, const char *key);
static hash_HashEntry_t *BogusCreate(hash_HashTable_t *tablePtr, const char *key, int *newPtr);
static unsigned int      HashString(const char *string);
static void              RebuildTable(hash_HashTable_t *tablePtr);
static hash_HashEntry_t *StringFind(hash_HashTable_t *tablePtr, const char *key);
static hash_HashEntry_t *StringCreate(hash_HashTable_t *tablePtr, const char *key, int *newPtr);
static hash_HashEntry_t *OneWordFind(hash_HashTable_t *tablePtr, const char *key);
static hash_HashEntry_t *OneWordCreate(hash_HashTable_t *tablePtr, const char *key, int *newPtr);

static const unsigned int primes[] =
{
    11,
    19,
    37,
    73,
    109,
    163,
    251,
    367,
    557,
    823,
    1237,
    1861,
    2777,
    4177,
    6247,
    9371,
    14057,
    21089,
    31627,
    47431,
    71143,
    106721,
    160073,
    240101,
    360163,
    540217,
    810343,
    1215497,
    1823231,
    2734867,
    4102283,
    6153409,
    9230113,
    13845163
};

static const int nprimes = (int) (sizeof (primes) / sizeof (primes[0]));

/*
 *----------------------------------------------------------------------
 *----------------------------------------------------------------------
 */

static unsigned int hash_ClosestPrime(unsigned int num)
{
    int i;

    for (i = 0;  i < nprimes;  i++)
    {
        if (primes[i] > num)
            return primes[i];
    }
    return primes[nprimes - 1];
}

/*
 *----------------------------------------------------------------------
 *
 * hash_InitHashTable --
 *
 *        Given storage for a hash table, set up the fields to prepare
 *        the hash table for use.
 *
 * Results:
 *        None.
 *
 * Side effects:
 *        TablePtr is now ready to be passed to hash_FindHashEntry and
 *        hash_CreateHashEntry.
 *
 *----------------------------------------------------------------------
 */

void hash_InitHashTable(hash_HashTable_t *tablePtr, int keyType)
{
    tablePtr->buckets = tablePtr->staticBuckets;
    tablePtr->staticBuckets[0] =
    tablePtr->staticBuckets[1] =
    tablePtr->staticBuckets[2] =
    tablePtr->staticBuckets[3] = 0;
    tablePtr->numBuckets = HASH_SMALL_HASH_TABLE;
    tablePtr->numEntries = 0;
    tablePtr->rebuildSize = HASH_SMALL_HASH_TABLE*REBUILD_MULTIPLIER;
    tablePtr->downShift = 28;
    tablePtr->mask = 3;
    tablePtr->keyType = keyType;
    if (keyType == HASH_STRING_KEYS)
    {
        tablePtr->findProc = StringFind;
        tablePtr->createProc = StringCreate;
    }
    else if (keyType == HASH_ONE_WORD_KEYS)
    {
        tablePtr->findProc = OneWordFind;
        tablePtr->createProc = OneWordCreate;
    }
    else
    {
        tablePtr->findProc = ArrayFind;
        tablePtr->createProc = ArrayCreate;
    };
}

/*
 *----------------------------------------------------------------------
 *
 * hash_DeleteHashEntry --
 *
 *        Remove a single entry from a hash table.
 *
 * Results:
 *        None.
 *
 * Side effects:
 *        The entry given by entryPtr is deleted from its table and
 *        should never again be used by the caller.  It is up to the
 *        caller to free the clientData field of the entry, if that
 *        is relevant.
 *
 *----------------------------------------------------------------------
 */

void hash_DeleteHashEntry(hash_HashEntry_t *entryPtr)
{
    hash_HashEntry_t *prevPtr;

    if (*entryPtr->bucketPtr == entryPtr)
    {
        *entryPtr->bucketPtr = entryPtr->nextPtr;
    }
    else
    {
        for (prevPtr = *entryPtr->bucketPtr;  ;  prevPtr = prevPtr->nextPtr)
        {
            if (prevPtr == NULL)
                fprintf(stderr, "malformed bucket chain in hash_DeleteHashEntry");
            if (prevPtr->nextPtr == entryPtr)
            {
                prevPtr->nextPtr = entryPtr->nextPtr;
                break;
            }
        }
    }
    entryPtr->tablePtr->numEntries--;
    free((char *) entryPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * hash_DeleteHashTable --
 *
 *        Free up everything associated with a hash table except for
 *        the record for the table itself.
 *
 * Results:
 *        None.
 *
 * Side effects:
 *        The hash table is no longer useable.
 *
 *----------------------------------------------------------------------
 */

void hash_DeleteHashTable(hash_HashTable_t *tablePtr)
{
    hash_HashEntry_t *hPtr;
    hash_HashEntry_t *nextPtr;
    int i;

    /* Free up all the entries in the table. */

    for (i = 0;  i < tablePtr->numBuckets;  i++)
    {
        for (hPtr = tablePtr->buckets[i];  hPtr;  hPtr = nextPtr)
        {
            nextPtr = hPtr->nextPtr;
            free((char *) hPtr);
        }
    }

    /* Free up the bucket array, if it was dynamically allocated. */

    if (tablePtr->buckets != tablePtr->staticBuckets)
        free((char *) tablePtr->buckets);

    /*
     * Arrange for panics if the table is used again without
     * re-initialization.
     */

    tablePtr->findProc = BogusFind;
    tablePtr->createProc = BogusCreate;
}

/*
 *----------------------------------------------------------------------
 *
 * hash_FirstHashEntry --
 *
 *        Locate the first entry in a hash table and set up a record
 *        that can be used to step through all the remaining entries
 *        of the table.
 *
 * Results:
 *        The return value is a pointer to the first entry in tablePtr,
 *        or NULL if tablePtr has no entries in it.  The memory at
 *        *searchPtr is initialized so that subsequent calls to
 *        hash_NextHashEntry will return all the entries in the table,
 *        one at a time.
 *
 * Side effects:
 *        None.
 *
 *----------------------------------------------------------------------
 */

hash_HashEntry_t *hash_FirstHashEntry(hash_HashTable_t *tablePtr,
                                      hash_HashSearch_t *searchPtr)
{
    searchPtr->tablePtr = tablePtr;
    searchPtr->nextIndex = 0;
    searchPtr->nextEntryPtr = NULL;
    return hash_NextHashEntry(searchPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * hash_NextHashEntry --
 *
 *        Once a hash table enumeration has been initiated by calling
 *        hash_FirstHashEntry, this procedure may be called to return
 *        successive elements of the table.
 *
 * Results:
 *        The return value is the next entry in the hash table being
 *        enumerated, or NULL if the end of the table is reached.
 *
 * Side effects:
 *        None.
 *
 *----------------------------------------------------------------------
 */

hash_HashEntry_t *hash_NextHashEntry(hash_HashSearch_t *searchPtr)
{
    hash_HashEntry_t *hPtr;

    while (searchPtr->nextEntryPtr == NULL)
    {
        if (searchPtr->nextIndex >= searchPtr->tablePtr->numBuckets)
            return NULL;
        searchPtr->nextEntryPtr =
            searchPtr->tablePtr->buckets[searchPtr->nextIndex];
        searchPtr->nextIndex++;
    }
    hPtr = searchPtr->nextEntryPtr;
    searchPtr->nextEntryPtr = hPtr->nextPtr;
    return hPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * hash_HashStats --
 *
 *        Return statistics describing the layout of the hash table
 *        in its hash buckets.
 *
 * Results:
 *        The return value is a malloc-ed string containing information
 *        about tablePtr.  It is the caller's responsibility to free
 *        this string.
 *
 * Side effects:
 *        None.
 *
 *----------------------------------------------------------------------
 */

char *hash_HashStats(hash_HashTable_t *tablePtr)
{
#define NUM_COUNTERS 10
    int count[NUM_COUNTERS];
    int overflow;
    int i;
    int j;
    double average;
    double tmp;
    hash_HashEntry_t *hPtr;
    char *result;
    char *p;

    /* Compute a histogram of bucket usage. */

    for (i = 0; i < NUM_COUNTERS; i++)
        count[i] = 0;
    overflow = 0;
    average = 0.0;
    for (i = 0;  i < tablePtr->numBuckets;  i++)
    {
        j = 0;
        for (hPtr = tablePtr->buckets[i];  hPtr;  hPtr = hPtr->nextPtr)
            j++;
        if (j < NUM_COUNTERS)
            count[j]++;
        else
            overflow++;
        tmp = j;
        average += (tmp+1.0)*(tmp/tablePtr->numEntries)/2.0;
    }

    /* Print out the histogram and a few other pieces of information. */

    result = (char *) malloc((unsigned) ((NUM_COUNTERS*60) + 300));
    if (result)
    {
        sprintf(result,
                "%d entries in table, %d buckets\n",
                tablePtr->numEntries,
                tablePtr->numBuckets);
        p = result + strlen(result);
        for (i = 0;  i < NUM_COUNTERS;  i++)
        {
            sprintf(p,
                    "number of buckets with %d entries: %d\n",
                    i,
                    count[i]);
            p += strlen(p);
        }
        sprintf(p,
                "number of buckets with %d or more entries: %d\n",
                NUM_COUNTERS,
                overflow);
        p += strlen(p);
        sprintf(p, "average search distance for entry: %.1f", average);
    }
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * HashString --
 *
 *        Compute a one-word summary of a text string, which can be
 *        used to generate a hash index.
 *
 * Results:
 *        The return value is a one-word summary of the information in
 *        string.
 *
 * Side effects:
 *        None.
 *
 *----------------------------------------------------------------------
 */

static unsigned int HashString(const char *string)
{
    unsigned int result;
    char c;

    /*
     * I tried a zillion different hash functions and asked many other
     * people for advice.  Many people had their own favorite functions,
     * all different, but no-one had much idea why they were good ones.
     * I chose the one below (multiply by 9 and add new character)
     * because of the following reasons:
     *
     * 1. Multiplying by 10 is perfect for keys that are decimal strings,
     *    and multiplying by 9 is just about as good.
     * 2. Times-9 is (shift-left-3) plus (old).  This means that each
     *    character's bits hang around in the low-order bits of the
     *    hash value for ever, plus they spread fairly rapidly up to
     *    the high-order bits to fill out the hash value.  This seems
     *    works well both for decimal and non-decimal strings.
     */

    for (result = 0;  ;  result += (result << 3) + c)
    {
        c = *string++;
        if (c == '\0')
            break;
    }
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * StringFind --
 *
 *        Given a hash table with string keys, and a string key, find
 *        the entry with a matching key.
 *
 * Results:
 *        The return value is a token for the matching entry in the
 *        hash table, or NULL if there was no matching entry.
 *
 * Side effects:
 *        None.
 *
 *----------------------------------------------------------------------
 */

static hash_HashEntry_t *StringFind(hash_HashTable_t *tablePtr, const char *key)
{
    hash_HashEntry_t *hPtr;
    const char *p1;
    const char *p2;
    int index;

    index = HashString(key) & tablePtr->mask;

    /* Search all the entries in the appropriate bucket. */
    for (hPtr = tablePtr->buckets[index];
         hPtr != NULL;
         hPtr = hPtr->nextPtr)
    {
        for (p1 = key, p2 = hPtr->key.string;  ;  p1++, p2++)
        {
            if (*p1 != *p2)
                break;
            if (*p1 == '\0')
                return hPtr;
        }
    }
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * StringCreate --
 *
 *        Given a hash table with string keys, and a string key, find
 *        the entry with a matching key.  If there is no matching entry,
 *        then create a new entry that does match.
 *
 * Results:
 *        The return value is a pointer to the matching entry.  If this
 *        is a newly-created entry, then *newPtr will be set to a non-zero
 *        value;  otherwise *newPtr will be set to 0.  If this is a new
 *        entry the value stored in the entry will initially be 0.
 *
 * Side effects:
 *        A new entry may be added to the hash table.
 *
 *----------------------------------------------------------------------
 */

static hash_HashEntry_t *StringCreate(hash_HashTable_t *tablePtr,
                                      const char *key,
                                      int *newPtr)
{
    hash_HashEntry_t *hPtr;
    const char *p1;
    const char *p2;
    int index;

    index = HashString(key) & tablePtr->mask;

    /* Search all the entries in this bucket. */
    for (hPtr = tablePtr->buckets[index];
         hPtr != NULL;
         hPtr = hPtr->nextPtr)
    {
        for (p1 = key, p2 = hPtr->key.string;  ;  p1++, p2++)
        {
            if (*p1 != *p2)
                break;
            if (*p1 == '\0')
            {
                if (newPtr)
                    *newPtr = 0;
                return hPtr;
            }
        }
    }

    /* Entry not found.  Add a new one to the bucket. */

    if (newPtr)
        *newPtr = 1;
    if ((hPtr = (hash_HashEntry_t *) malloc((unsigned) (sizeof(hash_HashEntry_t) + strlen((char *) key) - (sizeof(hPtr->key) - 1)))) == NULL)
        return  NULL;
    hPtr->tablePtr = tablePtr;
    hPtr->bucketPtr = &(tablePtr->buckets[index]);
    hPtr->nextPtr = *hPtr->bucketPtr;
    hPtr->clientData = NULL;
    strcpy(hPtr->key.string, (char *) key);
    *hPtr->bucketPtr = hPtr;
    tablePtr->numEntries++;

    /*
     * If the table has exceeded a decent size, rebuild it with many
     * more buckets.
     */
    if (tablePtr->numEntries >= tablePtr->rebuildSize)
        RebuildTable(tablePtr);
    return hPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * OneWordFind --
 *
 *        Given a hash table with one-word keys, and a one-word key, find
 *        the entry with a matching key.
 *
 * Results:
 *        The return value is a token for the matching entry in the
 *        hash table, or NULL if there was no matching entry.
 *
 * Side effects:
 *        None.
 *
 *----------------------------------------------------------------------
 */

static hash_HashEntry_t *OneWordFind(hash_HashTable_t *tablePtr, const char *key)
{
    hash_HashEntry_t *hPtr;
    int index;

    index = RANDOM_INDEX(tablePtr, key);

    /* Search all the entries in the appropriate bucket. */
    for (hPtr = tablePtr->buckets[index];
         hPtr;
         hPtr = hPtr->nextPtr)
    {
        if (hPtr->key.oneWordValue == key)
            return hPtr;
    }
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * OneWordCreate --
 *
 *        Given a hash table with one-word keys, and a one-word key, find
 *        the entry with a matching key.  If there is no matching entry,
 *        then create a new entry that does match.
 *
 * Results:
 *        The return value is a pointer to the matching entry.  If this
 *        is a newly-created entry, then *newPtr will be set to a non-zero
 *        value;  otherwise *newPtr will be set to 0.  If this is a new
 *        entry the value stored in the entry will initially be 0.
 *
 * Side effects:
 *        A new entry may be added to the hash table.
 *
 *----------------------------------------------------------------------
 */

static hash_HashEntry_t *OneWordCreate(hash_HashTable_t *tablePtr,
                                       const char *key,
                                       int *newPtr)
{
    hash_HashEntry_t *hPtr;
    int index;
    index = RANDOM_INDEX(tablePtr, key);

    /* Search all the entries in this bucket. */
    for (hPtr = tablePtr->buckets[index];
         hPtr;
         hPtr = hPtr->nextPtr)
    {
        if (hPtr->key.oneWordValue == key)
        {
            if (newPtr)
                *newPtr = 0;
            return hPtr;
        }
    }

    /* Entry not found.  Add a new one to the bucket. */

    if (newPtr)
        *newPtr = 1;
    if ((hPtr = (hash_HashEntry_t *) malloc(sizeof(hash_HashEntry_t))) == NULL)
        return  NULL;
    hPtr->tablePtr = tablePtr;
    hPtr->bucketPtr = &(tablePtr->buckets[index]);
    hPtr->nextPtr = *hPtr->bucketPtr;
    hPtr->clientData = NULL;
    hPtr->key.oneWordValue = (char *) key;        /* CONST XXXX */
    *hPtr->bucketPtr = hPtr;
    tablePtr->numEntries++;

    /*
     * If the table has exceeded a decent size, rebuild it with many
     * more buckets.
     */
    if (tablePtr->numEntries >= tablePtr->rebuildSize)
        RebuildTable(tablePtr);
    return hPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * ArrayFind --
 *
 *        Given a hash table with array-of-int keys, and a key, find
 *        the entry with a matching key.
 *
 * Results:
 *        The return value is a token for the matching entry in the
 *        hash table, or NULL if there was no matching entry.
 *
 * Side effects:
 *        None.
 *
 *----------------------------------------------------------------------
 */

static hash_HashEntry_t *ArrayFind(hash_HashTable_t *tablePtr, const char *key)
{
    hash_HashEntry_t *hPtr;
    int *arrayPtr;
    int *iPtr1;
    int *iPtr2;
    int index;
    int count;

    arrayPtr = (int *) key;
    for (index = 0, count = tablePtr->keyType, iPtr1 = arrayPtr;
         count > 0;
         count--, iPtr1++)
    {
        index += *iPtr1;
    }
    index = RANDOM_INDEX(tablePtr, index);

    /* Search all the entries in the appropriate bucket. */
    for (hPtr = tablePtr->buckets[index];
         hPtr;
         hPtr = hPtr->nextPtr)
    {
        for (iPtr1 = arrayPtr, iPtr2 = hPtr->key.words, count = tablePtr->keyType;
             ;
             count--, iPtr1++, iPtr2++)
        {
            if (count == 0)
                return hPtr;
            if (*iPtr1 != *iPtr2)
                break;
        }
    }
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * ArrayCreate --
 *
 *        Given a hash table with one-word keys, and a one-word key, find
 *        the entry with a matching key.  If there is no matching entry,
 *        then create a new entry that does match.
 *
 * Results:
 *        The return value is a pointer to the matching entry.  If this
 *        is a newly-created entry, then *newPtr will be set to a non-zero
 *        value;  otherwise *newPtr will be set to 0.  If this is a new
 *        entry the value stored in the entry will initially be 0.
 *
 * Side effects:
 *        A new entry may be added to the hash table.
 *
 *----------------------------------------------------------------------
 */

static hash_HashEntry_t *ArrayCreate(hash_HashTable_t *tablePtr,
                                     const char *key,
                                     int *newPtr)
{
    hash_HashEntry_t *hPtr;
    int *arrayPtr;
    int *iPtr1;
    int *iPtr2;
    int index;
    int count;

    arrayPtr = (int *) key;
    for (index = 0, count = tablePtr->keyType, iPtr1 = arrayPtr;
         count > 0;
         count--, iPtr1++)
    {
        index += *iPtr1;
    }
    index = RANDOM_INDEX(tablePtr, index);

    /* Search all the entries in the appropriate bucket. */
    for (hPtr = tablePtr->buckets[index];
         hPtr != NULL;
         hPtr = hPtr->nextPtr)
    {
        for (iPtr1 = arrayPtr, iPtr2 = hPtr->key.words, count = tablePtr->keyType;
             ;
             count--, iPtr1++, iPtr2++)
        {
            if (count == 0)
            {
                if (newPtr)
                    *newPtr = 0;
                return hPtr;
            }
            if (*iPtr1 != *iPtr2)
                break;
        }
    }

    /* Entry not found.  Add a new one to the bucket. */

    if (newPtr)
        *newPtr = 1;
    if ((hPtr = (hash_HashEntry_t *) malloc((unsigned) (sizeof(hash_HashEntry_t) + (tablePtr->keyType*sizeof(int)) - 4))) == NULL)
        return  NULL;
    hPtr->tablePtr = tablePtr;
    hPtr->bucketPtr = &(tablePtr->buckets[index]);
    hPtr->nextPtr = *hPtr->bucketPtr;
    hPtr->clientData = NULL;
    for (iPtr1 = arrayPtr, iPtr2 = hPtr->key.words, count = tablePtr->keyType;
         count > 0;
         count--, iPtr1++, iPtr2++)
    {
        *iPtr2 = *iPtr1;
    }
    *hPtr->bucketPtr = hPtr;
    tablePtr->numEntries++;

    /*
     * If the table has exceeded a decent size, rebuild it with many
     * more buckets.
     */
    if (tablePtr->numEntries >= tablePtr->rebuildSize)
        RebuildTable(tablePtr);
    return hPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * BogusFind --
 *
 *        This procedure is invoked when an hash_FindHashEntry is called
 *        on a table that has been deleted.
 *
 * Results:
 *        If panic returns (which it shouldn't) this procedure returns
 *        NULL.
 *
 * Side effects:
 *        Generates a panic.
 *
 *----------------------------------------------------------------------
 */

static hash_HashEntry_t *BogusFind(hash_HashTable_t *tablePtr, const char *key)
{
    fprintf(stderr, "called hash_FindHashEntry on deleted table");
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * BogusCreate --
 *
 *        This procedure is invoked when an hash_CreateHashEntry is called
 *        on a table that has been deleted.
 *
 * Results:
 *        If panic returns (which it shouldn't) this procedure returns
 *        NULL.
 *
 * Side effects:
 *        Generates a panic.
 *
 *----------------------------------------------------------------------
 */

static hash_HashEntry_t *BogusCreate(hash_HashTable_t *tablePtr,
                                  const char *key,
                                  int *newPtr)
{
    fprintf(stderr, "called hash_CreateHashEntry on deleted table");
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * RebuildTable --
 *
 *        This procedure is invoked when the ratio of entries to hash
 *        buckets becomes too large.  It creates a new table with a
 *        larger bucket array and moves all the entries into the
 *        new table.
 *
 * Results:
 *        None.
 *
 * Side effects:
 *        Memory gets reallocated and entries get re-hashed to new
 *        buckets.
 *
 *----------------------------------------------------------------------
 */

static void RebuildTable(hash_HashTable_t *tablePtr)
{
    int oldSize;
    int count;
    int index;
    hash_HashEntry_t **oldBuckets;
    hash_HashEntry_t **oldChainPtr;
    hash_HashEntry_t **newChainPtr;
    hash_HashEntry_t *hPtr;

    oldSize = tablePtr->numBuckets;
    oldBuckets = tablePtr->buckets;

    /*
     * Allocate and initialize the new bucket array, and set up
     * hashing constants for new array size.
     */
    tablePtr->numBuckets = hash_ClosestPrime(tablePtr->numEntries);
    /* TODO: report the memory failure some way. Don't just bomb out */
    if ((tablePtr->buckets = (hash_HashEntry_t **) malloc((unsigned) (tablePtr->numBuckets * sizeof(hash_HashEntry_t *)))) == NULL)
        return;
    for (count = tablePtr->numBuckets, newChainPtr = tablePtr->buckets;
         count > 0;
         count--, newChainPtr++)
    {
        *newChainPtr = NULL;
    }
    tablePtr->rebuildSize *= 4;
    tablePtr->downShift -= 2;
    tablePtr->mask = (tablePtr->mask << 2) + 3;

    /* Rehash all the existing entries into the new bucket array. */
    for (oldChainPtr = oldBuckets; oldSize > 0; oldSize--, oldChainPtr++)
    {
        for (hPtr = *oldChainPtr; hPtr != NULL; hPtr = *oldChainPtr)
        {
            *oldChainPtr = hPtr->nextPtr;
            if (tablePtr->keyType == HASH_STRING_KEYS)
            {
                index = HashString(hPtr->key.string) & tablePtr->mask;
            }
            else if (tablePtr->keyType == HASH_ONE_WORD_KEYS)
            {
                index = RANDOM_INDEX(tablePtr, hPtr->key.oneWordValue);
            }
            else
            {
                int *iPtr;
                int count;

                for (index = 0, count = tablePtr->keyType,
                     iPtr = hPtr->key.words; count > 0;
                     count--, iPtr++)
                {
                    index += *iPtr;
                }
                index = RANDOM_INDEX(tablePtr, index);
            }
            hPtr->bucketPtr = &(tablePtr->buckets[index]);
            hPtr->nextPtr = *hPtr->bucketPtr;
            *hPtr->bucketPtr = hPtr;
        }
    }

    /* Free up the old bucket array, if it was dynamically allocated. */

    if (oldBuckets != tablePtr->staticBuckets)
        free((char *) oldBuckets);
}

#if 0
int main(int srgc, char *argv[])
{
    int stuff[1000000];
    hash_HashTable_t fred;
    hash_HashEntry_t *x;
    int new;
    int r;
    int i;

    hash_InitHashTable(&fred, HASH_ONE_WORD_KEYS);

    for (i = 0;  i < 1000000;  i++)
    {
        r = rand();
        stuff[i] = r;
        x = hash_CreateHashEntry(&fred, (const char *) stuff[i], &new);
        if (!new)
            printf("Clash at %d\n", r);
        hash_SetHashValue(x, i);
    }

    printf("Filled\n");
    for (i = 0;  i < 1000000;  i++)
    {
        x = hash_FindHashEntry(&fred, (const char *) stuff[i]);
        if ((int) hash_GetHashValue(x) != i)
            printf("At %d - %x %p\n", i, stuff[i], x->clientData);
    }
    printf("Checked\n");
    return  0;
}
#endif
