/*
 * libunicall - a library for universal call handling on both analogue and
 *              digital telephone circuits.
 *
 * logging.c
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2002 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: logging.c,v 1.10 2005/05/22 05:45:32 steveu Exp $
 */

/*! \file uni-call.h */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <limits.h>
#include <stdio.h>
#include <stdarg.h>
#include <fcntl.h>
#include <stdlib.h>
#include <stdint.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <time.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/ioctl.h>

#include <linux/zaptel.h>

#include "unicall.h"
#include "unicall/hashtable.h"
#include "unicall/unicall-private.h"

#define FALSE 0
#define TRUE (!FALSE)

typedef struct msgtype_s
{
    int msgnum;
    const char *name;
} msgtype_t;

int uc_log(uc_t *uc, int level, const char *format, ...)
{
    char msg[1024 + 1];
    va_list arg_ptr;
    int len;
    char timestr[17 + 1];
    struct tm *tim;
    time_t now;

    if (uc  &&  uc->chan  &&  (uc->logging_level & UC_LOG_SEVERITY_MASK) >= level)
    {
        va_start(arg_ptr, format);
        len = 0;
        if ((uc->logging_level & UC_LOG_SHOW_DATE))
        {
            time(&now);
            tim = gmtime(&now);
            sprintf(msg + len,
                    "%04d/%02d/%02d %02d:%02d:%02d ",
                    tim->tm_year + 1900,
                    tim->tm_mon + 1,
                    tim->tm_mday,
                    tim->tm_hour,
                    tim->tm_min,
                    tim->tm_sec);
            len += strlen(msg + len);
        }
        /*endif*/
        if ((uc->logging_level & UC_LOG_SHOW_PROTOCOL))
        {
            strcpy(msg + len, uc_protocol_class2short_str(uc->protocol_class));
            len += strlen(msg + len);
            strcpy(msg + len, " ");
            len += strlen(msg + len);
        }
        /*endif*/
        if ((uc->logging_level & UC_LOG_SHOW_VARIANT))
        {
            strcpy(msg + len, uc_protocol_variant2str(uc->protocol_class, uc->protocol_variant));
            len += strlen(msg + len);
            strcpy(msg + len, " ");
            len += strlen(msg + len);
        }
        /*endif*/
        if ((uc->logging_level & UC_LOG_SHOW_TAG)  &&  uc->logging_tag)
        {
            strcpy(msg + len, uc->logging_tag);
            len += strlen(msg + len);
            strcpy(msg + len, " ");
            len += strlen(msg + len);
        }
        /*endif*/
        len += vsnprintf(msg + len, 1024 - len, format, arg_ptr);
        uc_error(msg);
        va_end(arg_ptr);
        return  1;
    }
    /*endif*/
    return  0;
}
/*- End of function --------------------------------------------------------*/

static const char *code2str(int code, msgtype_t *codes, int max)
{
    int x;
    
    for (x = 0;  x < max;  x++)
    {
        if (codes[x].msgnum == code)
            return codes[x].name;
        /*endif*/
    }
    /*endfor*/
    return "Unknown";
}
/*- End of function --------------------------------------------------------*/

const char *uc_ret2str(int code)
{
    static msgtype_t codes[] =
    {
        { UC_RET_OK, "OK" },
        { UC_RET_WRONG_PROTOCOL, "Wrong protocol" },
        { UC_RET_BAD_PARAMETER, "Bad parameter" },
        { UC_RET_BAD_DEVICE, "Bad device" },
        { UC_RET_BAD_STATE, "Bad state" },
        { UC_RET_DEVICE_ERROR, "Device error" },
        { UC_RET_UNSUPPORTED, "Unsupported operation" },
        { UC_RET_CRN_NOT_KNOWN, "CRN not known" },
        { UC_RET_BAD_CHANNEL, "Bad channel" },
        { UC_RET_MEMORY_ERROR, "Memory error" },
        { UC_RET_BLOCKED, "Blocked" },
    };
    return code2str(code, codes, sizeof(codes)/sizeof(codes[0]));
}
/*- End of function --------------------------------------------------------*/

const char *uc_event2str(int id)
{
    static msgtype_t events[] =
    {
        { UC_EVENT_DEVICEFAIL, "Device failure" },
        { UC_EVENT_PROTOCOLFAIL , "Protocol failure" },
        { UC_EVENT_SIGCHANSTATUS, "Signaling channel status" },
        { UC_EVENT_DETECTED, "Detected" },
        { UC_EVENT_OFFERED, "Offered" },
        { UC_EVENT_GETMOREINFO, "Get more info" },
        { UC_EVENT_ACCEPTED, "Accepted" },
        { UC_EVENT_CALLINFO, "Call info" },
        { UC_EVENT_FACILITY, "Facility" },
        { UC_EVENT_MOREDIGITS, "More digits" },
        { UC_EVENT_DIALTONE, "Dial tone" },
        { UC_EVENT_DIALING, "Dialing" },
        { UC_EVENT_SENDMOREINFO, "Send more info" },
        { UC_EVENT_PROCEEDING, "Proceeding" },
        { UC_EVENT_ALERTING, "Alerting" },
        { UC_EVENT_CONNECTED, "Connected" },
        { UC_EVENT_ANSWERED, "Answered" },
        { UC_EVENT_FARDISCONNECTED, "Far end disconnected" },
        { UC_EVENT_DROPCALL, "Drop call" },
        { UC_EVENT_RELEASECALL, "Release call" },
        { UC_EVENT_FARBLOCKED, "Far end blocked" },
        { UC_EVENT_FARUNBLOCKED, "Far end unblocked" },
        { UC_EVENT_LOCALBLOCKED, "Local end blocked" },
        { UC_EVENT_LOCALUNBLOCKED, "Local end unblocked" },
        { UC_EVENT_ALARM, "Alarm" },
        { UC_EVENT_RESETLINEDEV, "Reset line device" },
        { UC_EVENT_L2FRAME, "L2 frame" },
        { UC_EVENT_L2BUFFERFULL, "L2 buffer full" },
        { UC_EVENT_L2NOBUFFER, "L2 no buffer" },
        { UC_EVENT_USRINFO, "L2 user info" }
    };

    return code2str(id, events, sizeof(events)/sizeof(events[0]));
}
/*- End of function --------------------------------------------------------*/

const char *uc_cause2str(int code)
{
    static msgtype_t codes[] =
    {
        { UC_CAUSE_UNASSIGNED_NUMBER, "Unallocated (unassigned) number" },
        { UC_CAUSE_NO_ROUTE_TO_TRANSIT_NETWORK, "No route to specified transmit network" },
        { UC_CAUSE_NO_ROUTE_TO_DESTINATION, "No route to destination" },
        { UC_CAUSE_CHANNEL_UNACCEPTABLE, "Channel unacceptable" },
        { UC_CAUSE_CALL_AWARDED_DELIVERED, "Call awarded and being delivered in an established channel" },
        { UC_CAUSE_NORMAL_CLEARING, "Normal Clearing" },
        { UC_CAUSE_USER_BUSY, "User busy" },
        { UC_CAUSE_NO_USER_RESPONDING, "No user responding" },
        { UC_CAUSE_NO_ANSWER_FROM_USER, "User alerting, no answer" },
        { UC_CAUSE_CALL_REJECTED, "Call Rejected" },
        { UC_CAUSE_NUMBER_CHANGED, "Number changed" },
        { UC_CAUSE_NON_SELECT_USER_CLEARED, "Non-selected user cleared" },
        { UC_CAUSE_DEST_OUT_OF_ORDER, "Destination out of order" },
        { UC_CAUSE_INVALID_NUMBER_FORMAT, "Invalid number format" },
        { UC_CAUSE_FACILITY_REJECTED, "Facility rejected" },
        { UC_CAUSE_RESP_TO_STAT_ENQ, "Response to STATus ENQuiry" },
        { UC_CAUSE_UNSPECIFIED_CAUSE, "Normal, unspecified cause" },

        { UC_CAUSE_NO_CIRCUIT_AVAILABLE, "Circuit/channel congestion" },
        { UC_CAUSE_NETWORK_OUT_OF_ORDER, "Network out of order" },
        { UC_CAUSE_TEMPORARY_FAILURE, "Temporary failure" },
        { UC_CAUSE_NETWORK_CONGESTION, "Switching equipment congestion" },
        { UC_CAUSE_ACCESS_INFO_DISCARDED, "Access information discarded" },
        { UC_CAUSE_REQ_CHANNEL_NOT_AVAILABLE, "Requested channel not available" },
        { UC_CAUSE_PRE_EMPTED, "Pre-empted" },

        { UC_CAUSE_QOS_NOT_AVAILABLE, "Quality of service not available" },
        { UC_CAUSE_FACILITY_NOT_SUBSCRIBED, "Requested facility not subscribed" },
        { UC_CAUSE_OUTGOING_CALL_BARRED, "Outgoing call barred" },
        { UC_CAUSE_INCOMING_CALL_BARRED, "Incoming call barred" },
        { UC_CAUSE_BEARER_CAP_NOT_AUTHORIZED, "Bearer capability not authorized" }, 
        { UC_CAUSE_BEARER_CAP_NOT_AVAILABLE, "Bearer capability not presently available" },
        { UC_CAUSE_SERVICE_NOT_AVAILABLE, "Service or option not available, unspecified" },

        { UC_CAUSE_BEARER_CAP_NOT_IMPLEMENTED, "Bearer capability not implemented" },
        { UC_CAUSE_CHAN_TYPE_NOT_IMPLEMENTED, "Channel type not implemented" },
        { UC_CAUSE_FACILITY_NOT_IMPLEMENTED, "Facility not implemented" },
        { UC_CAUSE_RESTRICTED_DIGITAL_BEARER, "Only restricted digital bearer capability available" },
        { UC_CAUSE_SERVICE_NOT_IMPLEMENTED, "Service or option not implemented, unspecified" },

        { UC_CAUSE_INVALID_CALL_REF, "Invalid call reference value" },
        { UC_CAUSE_CHAN_DOES_NOT_EXIST, "Identified channel does not exist" },
        { UC_CAUSE_SUSPENDED_CALL_BAD_CALL_REF, "A suspended call exists, but this call identity does not" },
        { UC_CAUSE_CALL_REF_IN_USE, "Call identity in use" },
        { UC_CAUSE_NO_CALL_SUSPENDED, "No call suspended" },
        { UC_CAUSE_CALL_REF_CLEARED, "Call having the requested call identity has been cleared" },
        { UC_CAUSE_INCOMPATIBLE_DEST, "Incompatible destination" },
        { UC_CAUSE_INVALID_TRANSIT_NETWORK, "Invalid transit network selection" },
        { UC_CAUSE_INVALID_MSG_UNSPECIFIED, "Invalid message, unspecified" },

        { UC_CAUSE_MANDATORY_IE_MISSING, "Mandatory IE is missing" },
        { UC_CAUSE_NONEXISTANT_MSG, "Message type non-existant or not implemented" },
        { UC_CAUSE_WRONG_MESSAGE, "Message not compatible with call state, or message type non-existant or not implemented" },
        { UC_CAUSE_BAD_INFO_ELEM, "IE non-existant or not implemented" },
        { UC_CAUSE_INVALID_ELEM_CONTENTS, "Invalid IE contents" },
        { UC_CAUSE_WRONG_MSG_FOR_STATE, "Message not compatible with call state" },
        { UC_CAUSE_TIMER_EXPIRY, "Recover on timer expiry" },
        { UC_CAUSE_MANDATORY_IE_LEN_ERROR, "Mandatory IE length error" },
        { UC_CAUSE_PROTOCOL_ERROR, "Protocol error, unspecified" },

        { UC_CAUSE_INTERWORKING_UNSPECIFIED, "Interworking, unspecified" },
    };

    return code2str(code, codes, sizeof(codes)/sizeof(codes[0]));
}
/*- End of function --------------------------------------------------------*/

const char *uc_npi2str(int code)
{
    static msgtype_t codes[] =
    {
        { UC_NPI_UNKNOWN, "Unknown Number Plan" },
        { UC_NPI_E163_E164, "ISDN/Telephony Numbering Plan (E.164/E.163)" },
        { UC_NPI_X121, "Data Numbering Plan (X.121)" },
        { UC_NPI_F69, "Telex Numbering Plan (F.69)" },
        { UC_NPI_NATIONAL, "National Standard Numbering Plan" },
        { UC_NPI_PRIVATE, "Private Numbering Plan" },
        { UC_NPI_RESERVED, "Reserved Number Plan" },
    };
    return code2str(code, codes, sizeof(codes)/sizeof(codes[0]));
}
/*- End of function --------------------------------------------------------*/

const char *uc_ton2str(int code)
{
    static msgtype_t codes[] =
    {
        { UC_TON_UNKNOWN, "Unknown Number Type" },
        { UC_TON_INTERNATIONAL, "International Number" },
        { UC_TON_NATIONAL, "National Number" },
        { UC_TON_NET_SPECIFIC, "Network Specific Number" },
        { UC_TON_SUBSCRIBER, "Subscriber Number" },
        { UC_TON_ABBREVIATED, "Abbreviated number" },
        { UC_TON_RESERVED, "Reserved Number" },
    };

    return code2str(code, codes, sizeof(codes)/sizeof(codes[0]));
}
/*- End of function --------------------------------------------------------*/

const char *uc_pres2str(int code)
{
    static msgtype_t codes[] =
    {
        { UC_PRES_ALLOWED_USER_NUMBER_NOT_SCREENED, "Presentation permitted, user number not screened" },
        { UC_PRES_ALLOWED_USER_NUMBER_PASSED_SCREEN, "Presentation permitted, user number passed network screening" },
        { UC_PRES_ALLOWED_USER_NUMBER_FAILED_SCREEN, "Presentation permitted, user number failed network screening" },
        { UC_PRES_ALLOWED_NETWORK_NUMBER, "Presentation allowed of network provided number" },
        { UC_PRES_PROHIB_USER_NUMBER_NOT_SCREENED, "Presentation prohibited, user number not screened" },
        { UC_PRES_PROHIB_USER_NUMBER_PASSED_SCREEN, "Presentation prohibited, user number passed network screening" },
        { UC_PRES_PROHIB_USER_NUMBER_FAILED_SCREEN, "Presentation prohibited, user number failed network screening" },
        { UC_PRES_PROHIB_NETWORK_NUMBER, "Presentation prohibbited of network provided number" },
    };

    return code2str(code, codes, sizeof(codes)/sizeof(codes[0]));
}
/*- End of function --------------------------------------------------------*/

const char *uc_redirection_reason2str(int code)
{
    static msgtype_t codes[] =
    {
        { UC_REDIR_UNKNOWN, "Unknown" },
        { UC_REDIR_FORWARD_ON_BUSY, "Forwarded on busy" },
        { UC_REDIR_FORWARD_ON_NO_REPLY, "Forwarded on no reply" },
        { UC_REDIR_DEFLECTION, "Call deflected" },
        { UC_REDIR_DTE_OUT_OF_ORDER, "Called DTE out of order" },
        { UC_REDIR_FORWARDED_BY_DTE, "Forwarded by called DTE" },
        { UC_REDIR_UNCONDITIONAL, "Forwarded unconditionally" },
    };

    return code2str(code, codes, sizeof(codes)/sizeof(codes[0]));
}
/*- End of function --------------------------------------------------------*/

const char *uc_subaddrtype2str(int code)
{
    static msgtype_t codes[] =
    {
        { 0, "NSAP (X.213/ISO 8348 AD2)" },
        { 2, "User specified" },
    };

    return code2str(code, codes, sizeof(codes)/sizeof(codes[0]));
}
/*- End of function --------------------------------------------------------*/

const char *uc_calltype2str(int call_type)
{
    static msgtype_t call_types[] =
    {
        { UC_CALL_TYPE_NO_CHARGE_CALL, "Free call" },
        { UC_CALL_TYPE_CHARGED_CALL, "Charged call" },
    };

    return code2str(call_type, call_types, sizeof(call_types)/sizeof(call_types[0]));
}
/*- End of function --------------------------------------------------------*/

const char *uc_trans_cap2str(int code)
{
    static msgtype_t codes[] =
    {
        { UC_TRANS_CAP_SPEECH, "Speech" },
        { UC_TRANS_CAP_DIGITAL, "Unrestricted digital information" },
        { UC_TRANS_CAP_RESTRICTED_DIGITAL, "Restricted digital information" },
        { UC_TRANS_CAP_3_1K_AUDIO, "3.1kHz audio" },
        { UC_TRANS_CAP_7K_AUDIO, "7kHz audio" },
        { UC_TRANS_CAP_VIDEO, "Video" },
    };

    return code2str(code, codes, sizeof(codes)/sizeof(codes[0]));
}
/*- End of function --------------------------------------------------------*/

const char *uc_trans_mode2str(int code)
{
    static msgtype_t codes[] =
    {
        { UC_TRANS_MODE_64_CIRCUIT, "64kbps, circuit mode" },
        { UC_TRANS_MODE_2x64_CIRCUIT, "2x64kbps, circuit mode" },
        { UC_TRANS_MODE_384_CIRCUIT, "384kbps, circuit mode" },
        { UC_TRANS_MODE_1536_CIRCUIT, "1536kbps, circuit mode" },
        { UC_TRANS_MODE_1920_CIRCUIT, "1920kbps, circuit mode" },
        { UC_TRANS_MODE_MULTIRATE, "Multi-rate (Nx64kbps)" },
        { UC_TRANS_MODE_PACKET, "Packet mode" },
    };

    return code2str(code, codes, sizeof(codes)/sizeof(codes[0]));
}
/*- End of function --------------------------------------------------------*/

const char *uc_rate_adapt2str(int code)
{
    static msgtype_t codes[] =
    {
        { UC_RATE_ADAPT_EBITS, "as specified by E-bits" },
        { UC_RATE_ADAPT_0K6, "600bps" },
        { UC_RATE_ADAPT_1K2, "1200bps" },
        { UC_RATE_ADAPT_2K4, "2400bps" },
        { UC_RATE_ADAPT_3K6, "3600bps" },
        { UC_RATE_ADAPT_4K8, "4800bps" },
        { UC_RATE_ADAPT_7K2, "7200bps" },
        { UC_RATE_ADAPT_8K, "8000bps" },
        { UC_RATE_ADAPT_9K6, "9600bps" },
        { UC_RATE_ADAPT_14K4, "14400bps" },
        { UC_RATE_ADAPT_16K, "16Kbps" },
        { UC_RATE_ADAPT_19K2, "19200bps" },
        { UC_RATE_ADAPT_32K, "32kbps" },
        { UC_RATE_ADAPT_48K, "48kbps" },
        { UC_RATE_ADAPT_56K, "56kbps" },
        { UC_RATE_ADAPT_64K, "64kbps" },
        { UC_RATE_ADAPT_0K1345, "134.5bps" },
        { UC_RATE_ADAPT_0K1, "100bps" },
        { UC_RATE_ADAPT_0K075_1K2, "75/1200bps" },
        { UC_RATE_ADAPT_1K2_0K075, "1200/75bps" },
        { UC_RATE_ADAPT_0K05, "50bps" },
        { UC_RATE_ADAPT_0K075, "75bps" },
        { UC_RATE_ADAPT_0K110, "110bps" },
        { UC_RATE_ADAPT_0K150, "150bps" },
        { UC_RATE_ADAPT_0K200, "200bps" },
        { UC_RATE_ADAPT_0K300, "300bps" },
        { UC_RATE_ADAPT_12K, "12kbps" },
    };

    return code2str(code, codes, sizeof(codes)/sizeof(codes[0]));
}
/*- End of function --------------------------------------------------------*/

const char *uc_layer12str(int code)
{
    static msgtype_t codes[] =
    {
        { UC_LAYER_1_ITU_RATE_ADAPT, "ITU rate adaption" },
        { UC_LAYER_1_ULAW, "u-law" },
        { UC_LAYER_1_ALAW, "A-law" },
        { UC_LAYER_1_G721, "G.721 ADPCM" },
        { UC_LAYER_1_G722_G725, "G.722/G.725 7kHz audio" },
        { UC_LAYER_1_G7XX_384K, "G.7xx 384k video" },
        { UC_LAYER_1_NON_ITU_ADAPT, "Non-ITU rate adaption" },
        { UC_LAYER_1_V120_RATE_ADAPT, "V.120 rate adaption" },
        { UC_LAYER_1_X31_RATE_ADAPT, "X.31 rate adaption" },
    };

    return code2str(code, codes, sizeof(codes)/sizeof(codes[0]));
}
/*- End of function --------------------------------------------------------*/

const char *uc_layer22str(int code)
{
    static msgtype_t codes[] =
    {
        { UC_LAYER_2_Q921, "Q.921" },
        { UC_LAYER_2_LAPB, "LAPB" },
    };

    return code2str(code, codes, sizeof(codes)/sizeof(codes[0]));
}
/*- End of function --------------------------------------------------------*/

const char *uc_layer32str(int code)
{
    static msgtype_t codes[] =
    {
        { UC_LAYER_3_Q931, "Q.931" },
        { UC_LAYER_3_X25, "X.25" },
    };

    return code2str(code, codes, sizeof(codes)/sizeof(codes[0]));
}
/*- End of function --------------------------------------------------------*/

const char *uc_location2str(int code)
{
    static msgtype_t codes[] =
    {
        { UC_LOCATION_USER, "User" },
        { UC_LOCATION_PRIV_NET_LOCAL_USER, "Private network serving the local user" },
        { UC_LOCATION_PUB_NET_LOCAL_USER, "Public network serving the local user" },
        { UC_LOCATION_TRANSIT_NET, "Transit network" },
        { UC_LOCATION_PUB_NET_REMOTE_USER, "Public network serving the remote user" },
        { UC_LOCATION_PRIV_NET_REMOTE_USER, "Private network serving the remote user" },
        { UC_LOCATION_INTERNATIONAL_NETWORK, "International network" },
        { UC_LOCATION_NETWORK_BEYOND_INTERWORKING, "Network beyond the interworking point" },
    };
    return code2str(code, codes, sizeof(codes)/sizeof(codes[0]));
}
/*- End of function --------------------------------------------------------*/

const char *uc_calling_party_category2str(int code)
{
    static msgtype_t codes[] =
    {
        { UC_CALLER_CATEGORY_NATIONAL_SUBSCRIBER_CALL, "National subscriber call" },
        { UC_CALLER_CATEGORY_NATIONAL_PRIORITY_SUBSCRIBER_CALL, "National priority subscriber call" },
        { UC_CALLER_CATEGORY_NATIONAL_MAINTENANCE_CALL, "National maintenance call" },
        { UC_CALLER_CATEGORY_NATIONAL_OPERATOR_CALL, "National operator call" },
        { UC_CALLER_CATEGORY_NATIONAL_DATA_CALL, "National data call" },
        { UC_CALLER_CATEGORY_NATIONAL_PAYPHONE_CALL, "National payphone call" },
        { UC_CALLER_CATEGORY_INTERNATIONAL_SUBSCRIBER_CALL, "International subscriber call" },
        { UC_CALLER_CATEGORY_INTERNATIONAL_PRIORITY_SUBSCRIBER_CALL, "International priority subscriber call" },
        { UC_CALLER_CATEGORY_INTERNATIONAL_OPERATOR_CALL, "International operator call" },
        { UC_CALLER_CATEGORY_INTERNATIONAL_DATA_CALL, "International data call" },
    };
    return code2str(code, codes, sizeof(codes)/sizeof(codes[0]));
}
/*- End of function --------------------------------------------------------*/

void uc_dump_callparms(uc_t *uc, uc_callparms_t *callparms, int verbose)
{
    uc_message("Bearer transfer cap: %s\n", uc_trans_cap2str(callparms->bear_cap_transfer_cap));
    uc_message("Bearer transfer mode: %s\n", uc_trans_mode2str(callparms->bear_cap_transfer_mode));
    uc_message("Bearer transfer rate: %s\n", uc_rate_adapt2str(callparms->bear_cap_transfer_rate));
    uc_message("User layer 1: %s\n", uc_layer12str(callparms->userinfo_layer1_protocol));
    uc_message("User rate: %d\n", callparms->user_rate);
    if (verbose  ||  callparms->destination_number[0])
    {
        uc_message("Destination address: %s (%s, %s)\n",
                   callparms->destination_number,
                   uc_ton2str(callparms->destination_ton),
                   uc_npi2str(callparms->destination_npi));
    }
    if (verbose  ||  callparms->destination_sub_addr_number[0])
    {
        uc_message("Destination sub-address: %s (%s, %s)\n",
                   callparms->destination_sub_addr_number,
                   uc_ton2str(callparms->destination_sub_addr_ton),
                   uc_npi2str(callparms->destination_sub_addr_npi));
    }
    if (verbose  ||  callparms->originating_number[0]  ||  callparms->originating_name[0])
    {
        uc_message("Originating address: %s [%s] (%s, %s, %s) [%s]\n",
                   callparms->originating_number,
                   uc_ton2str(callparms->originating_ton),
                   uc_npi2str(callparms->originating_npi),
                   uc_pres2str(callparms->originating_pres),
                   callparms->originating_name);
    }
    if (verbose  ||  callparms->originating_sub_addr_number[0])
    {
        uc_message("Originating sub-address: %s (%s, %s, %s)\n",
                   callparms->originating_sub_addr_number,
                   uc_ton2str(callparms->originating_sub_addr_ton),
                   uc_npi2str(callparms->originating_sub_addr_npi),
                   uc_pres2str(callparms->originating_pres));
    }
    if (verbose  ||  callparms->redirecting_number[0])
    {
        uc_message("Redirecting address: %s (%s, %s, %s)\n",
                   callparms->redirecting_number,
                   uc_ton2str(callparms->redirecting_ton),
                   uc_npi2str(callparms->redirecting_npi),
                   uc_pres2str(callparms->redirecting_pres));
    }
    uc_message("Calling party category: %s\n",
               uc_calling_party_category2str(callparms->calling_party_category));
    uc_message("Call type: %s\n", uc_calltype2str(callparms->call_type));
}
/*- End of function --------------------------------------------------------*/

void uc_dump_event(uc_t *uc, uc_event_t *ev)
{
    if (uc == NULL  ||  ev == NULL)
        return;
    /*endif*/
    uc_message("Event type: %s (%d)\n", uc_event2str(ev->e), ev->e);
    switch(ev->e)
    {
    case UC_EVENT_SIGCHANSTATUS:
        break;
    case UC_EVENT_OFFERED:
        uc_dump_callparms(uc, &(ev->offered.parms), FALSE);
        break;
    case UC_EVENT_FARDISCONNECTED:
        uc_message("Far disconnected, CRN: %d, reason: %s\n", ev->fardisconnected.crn, uc_cause2str(ev->fardisconnected.cause));
        break;
    case UC_EVENT_ALERTING:
        break;
    case UC_EVENT_ANSWERED:
        break;
    case UC_EVENT_CONNECTED:
        break;
    default:
        uc_message("Don't know how to dump events of type %d\n", ev->e);
        break;
    }
    /*endswitch*/
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
