/*
 * An implementation of hash tables based on a section of TCL which is:
 *
 * Copyright (c) 1991-1993 The Regents of the University of California.
 * Copyright (c) 1994 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * $Id: hashtable.h,v 1.3 2005/07/11 13:16:30 steveu Exp $
 */

/*! \file */

#ifndef _HASH_HASH_H_
#define _HASH_HASH_H_

#ifdef __cplusplus
#define EXTERN extern "C"
#else
#define EXTERN extern
#endif

#ifdef __cplusplus
/*
 * Forward declaration of hash_HashTable.  Needed by some C++ compilers
 * to prevent errors when the forward reference to hash_HashTable is
 * encountered in the hash_HashEntry structure.
 */

struct hash_HashTable_s;
#endif

/*
 * Structure definition for an entry in a hash table. Clients should not
 * directly access any of these fields directly. They should use the macros
 * defined below.
 */

typedef struct hash_HashEntry_s
{
    /*! Pointer to next entry in this hash bucket, or NULL for end of the chain. */
    struct hash_HashEntry_s *nextPtr;
    /*! Pointer to table containing entry. */  
    struct hash_HashTable_s *tablePtr;
    /*! Pointer to bucket that points to first entry in this entry's chain. Used
        for deleting the entry. */
    struct hash_HashEntry_s **bucketPtr;
    /*! Application stores something here with hash_SetHashValue. */
    void *clientData;
    union
    {                                   /* Key has one of these forms: */
        char *oneWordValue;             /* One-word value for key. */
        int words[1];                   /* Multiple integer words for key.
                                         * The actual size will be as large
                                         * as necessary for this table's
                                         * keys. */
        char string[sizeof(int)];       /* String for key. The actual size
                                         * will be as large as needed to hold
                                         * the key. */
    } key;                              /* MUST BE LAST FIELD IN RECORD!! */
} hash_HashEntry_t;

/*
 * Structure definition for a hash table.  This must be publicly declared,
 * so clients can allocate space for these structures, and access members
 * through macros. Clients should never directly access any fields in this
 * structure.
 */

#define HASH_SMALL_HASH_TABLE 4
typedef struct hash_HashTable_s
{
    hash_HashEntry_t **buckets;         /* Pointer to bucket array.  Each
                                         * element points to first entry in
                                         * bucket's hash chain, or NULL. */
    hash_HashEntry_t *staticBuckets[HASH_SMALL_HASH_TABLE];
                                        /* Bucket array used for small tables
                                         * (to avoid mallocs and frees). */
    int numBuckets;                     /* Total number of buckets allocated
                                         * at **bucketPtr. */
    int numEntries;                     /* Total number of entries present
                                         * in table. */
    int rebuildSize;                    /* Enlarge table when numEntries gets
                                         * to be this large. */
    int downShift;                      /* Shift count used in hashing
                                         * function.  Designed to use high-
                                         * order bits of randomized keys. */
    int mask;                           /* Mask value used in hashing
                                         * function. */
    int keyType;                        /* Type of keys used in this table. 
                                         * It's either HASH_STRING_KEYS,
                                         * HASH_ONE_WORD_KEYS, or an integer
                                         * giving the number of ints that
                                         * is the size of the key.
                                         */
    hash_HashEntry_t *(*findProc)(struct hash_HashTable_s *tablePtr, const char *key);
    hash_HashEntry_t *(*createProc)(struct hash_HashTable_s *tablePtr, const char *key, int *newPtr);
} hash_HashTable_t;

/*
 * Structure definition for information used to keep track of searches
 * through hash tables:
 */

typedef struct hash_HashSearch_s
{
    hash_HashTable_t *tablePtr;         /* Table being searched. */
    int nextIndex;                      /* Index of next bucket to be
                                         * enumerated after present one. */
    hash_HashEntry_t *nextEntryPtr;     /* Next entry to be enumerated in the
                                         * the current bucket. */
} hash_HashSearch_t;

/* Acceptable key types for hash tables: */

#define HASH_STRING_KEYS            0
#define HASH_ONE_WORD_KEYS            1
/* All higher values are the length of an integer array that is the key */

/* Macros for clients to use to access fields of hash entries: */

static __inline__ void *hash_GetHashValue(hash_HashEntry_t *h)
{
    return h->clientData;
}

static __inline__ void hash_SetHashValue(hash_HashEntry_t *h, void *value)
{
    h->clientData = value;
}

static __inline__ void *hash_GetHashKey(hash_HashTable_t *tablePtr, hash_HashEntry_t *h)
{
    return (tablePtr->keyType == HASH_ONE_WORD_KEYS)  ?  h->key.oneWordValue  :  h->key.string;
}

/*
 * Macros to use for clients to use to invoke find and create procedures
 * for hash tables:
 */

static __inline__ hash_HashEntry_t *hash_FindHashEntry(hash_HashTable_t *tablePtr, const char *key)
{
    return (*tablePtr->findProc)(tablePtr, key);
}

static __inline__ hash_HashEntry_t *hash_CreateHashEntry(hash_HashTable_t *tablePtr, const char *key, int *newPtr)
{
    return (*tablePtr->createProc)(tablePtr, key, newPtr);
}

EXTERN void              hash_DeleteHashEntry(hash_HashEntry_t *entryPtr);
EXTERN void              hash_DeleteHashTable(hash_HashTable_t *tablePtr);
EXTERN hash_HashEntry_t *hash_FirstHashEntry(hash_HashTable_t *tablePtr, hash_HashSearch_t *searchPtr);
EXTERN char             *hash_HashStats(hash_HashTable_t *tablePtr);
EXTERN void              hash_InitHashTable(hash_HashTable_t *tablePtr, int keyType);
EXTERN hash_HashEntry_t *hash_NextHashEntry(hash_HashSearch_t *searchPtr);

EXTERN void panic(char *format, ...);

#endif
