/*
 * supertone.c
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2004 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: supertone.c,v 1.5 2005/09/03 15:10:18 steveu Exp $
 */

/*! \file */

//#define _ISOC9X_SOURCE  1
//#define _ISOC99_SOURCE  1

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <limits.h>
#include <stdio.h>
#include <stdarg.h>
#include <math.h>
#include <fcntl.h>
#include <stdlib.h>
#include <inttypes.h>
#include <unistd.h>
#include <string.h>
#include <strings.h>
#include <errno.h>
#include <signal.h>
#include <time.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/ioctl.h>

#include <libxml/xmlmemory.h>
#include <libxml/parser.h>
#include <libxml/xinclude.h>

#include <tiffio.h>
#include <spandsp.h>

#include "supertone.h"
#include "libsupertone.h"

#define MAX_TONE_SETS   20

super_tone_set_t sets[MAX_TONE_SETS];

static int parse_tone(super_tone_rx_descriptor_t *desc, int tone_id, super_tone_tx_step_t **tree, xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur)
{
    xmlChar *x;
    float f1;
    float f2;
    float f_tol;
    float l1;
    float l2;
    float length;
    float length_tol;
    float recognition_length;
    float recognition_length_tol;
    int cycles;
    int min_duration;
    int max_duration;
    super_tone_tx_step_t *treep;

    cur = cur->xmlChildrenNode;
    while (cur)
    {
        if (xmlStrcmp(cur->name, (const xmlChar *) "step") == 0)
        {
            printf("Step - ");
            /* Set some defaults */
            f1 = 0.0;
            f2 = 0.0;
            f_tol = 1.0;
            l1 = -11.0;
            l2 = -11.0;
            length = 0.0;
            length_tol = 10.0;
            recognition_length = 0.0;
            recognition_length_tol = 10.0;
            cycles = 1;
            if ((x = xmlGetProp(cur, (const xmlChar *) "freq")))
            {
                sscanf((const char *) x, "%f [%f%%]", &f1, &f_tol);
                sscanf((const char *) x, "%f+%f [%f%%]", &f1, &f2, &f_tol);
                printf("Frequency=%.2f+%.2f [%.2f%%]", f1, f2, f_tol);
            }
            if ((x = xmlGetProp(cur, (const xmlChar *) "level")))
            {
                if (sscanf((const char *) x, "%f+%f", &l1, &l2) < 2)
                    l2 = l1;
                printf("Level=%.2f+%.2f", l1, l2);
            }
            if ((x = xmlGetProp(cur, (const xmlChar *) "length")))
            {
                sscanf((const char *) x, "%f [%f%%]", &length, &length_tol);
                printf("Length=%.2f [%.2f%%]", length, length_tol);
            }
            if ((x = xmlGetProp(cur, (const xmlChar *) "recognition-length")))
            {
                sscanf((const char *) x, "%f [%f%%]", &recognition_length, &recognition_length_tol);
                printf(" Recognition length=%.2f [%.2f%%]", recognition_length, recognition_length_tol);
            }
            if ((x = xmlGetProp(cur, (const xmlChar *) "cycles")))
            {
                if (xmlStrcasecmp(x, (const xmlChar *) "endless") == 0)
                    cycles = 0;
                else
                    cycles = atoi((const char *) x);
                printf("Cycles=%d ", cycles);
            }
            if ((x = xmlGetProp(cur, (const xmlChar *) "recorded-announcement")))
                printf("Recorded announcement='%s'", x);
            printf("\n");
            if (f1  ||  f2  ||  length)
            {
                /* TODO: This cannot handle cycling patterns */
                if (length == 0.0)
                {
                    if (recognition_length)
                        min_duration = recognition_length*1000.0 + 0.5;
                    else
                        min_duration = 700;
                    max_duration = 0;
                }
                else
                {
                    if (recognition_length)
                        min_duration = recognition_length*1000.0 + 0.5;
                    else
                        min_duration = (length*1000.0 + 0.5)*(1.0 - length_tol/100.0) - 30;
                    max_duration = (length*1000.0 + 0.5)*(1.0 + length_tol/100.0) + 30;
                }
                printf(">>>Detector element %10d %10d %10d %10d\n", (int) (f1 + 0.5), (int) (f2 + 0.5), min_duration, max_duration);
                super_tone_rx_add_element(desc, tone_id, f1 + 0.5, f2 + 0.5, min_duration, max_duration);
            }
            treep = super_tone_tx_make_step(NULL,
                                            f1,
                                            l1,
                                            f2,
                                            l2,
                                            length*1000.0 + 0.5,
                                            cycles);
            *tree = treep;
            tree = &(treep->next);
            parse_tone(desc, tone_id, &(treep->nest), doc, ns, cur);
        }
        /*endif*/
        cur = cur->next;
    }
    /*endwhile*/
    return  0;
}
/*- End of function --------------------------------------------------------*/

static void parse_tone_set(super_tone_set_t *s, xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur)
{
    int tone_id;
    xmlChar *x;

    printf("Parsing tone set\n");
    cur = cur->xmlChildrenNode;
    while (cur)
    {
        if (xmlStrcmp(cur->name, (const xmlChar *) "dial-tone") == 0)
        {
            printf("Hit %s\n", cur->name);
            tone_id = super_tone_rx_add_tone(s->tone_detector);
            s->tone[ST_TYPE_DIALTONE] = NULL;
            parse_tone(s->tone_detector, tone_id, &s->tone[ST_TYPE_DIALTONE], doc, ns, cur);
            s->tone_names[tone_id] = (const char *) xmlStrdup(cur->name);
            if ((x = xmlGetProp(cur, (const xmlChar *) "domain")))
                s->tone_domains[tone_id] = (const char *) xmlStrdup(x);
            /*endif*/
            s->tone_types[tone_id] = ST_TYPE_DIALTONE;
        }
        else if (xmlStrcmp(cur->name, (const xmlChar *) "ringback-tone") == 0)
        {
            printf("Hit %s\n", cur->name);
            tone_id = super_tone_rx_add_tone(s->tone_detector);
            s->tone[ST_TYPE_RINGBACK] = NULL;
            parse_tone(s->tone_detector, tone_id, &s->tone[ST_TYPE_RINGBACK], doc, ns, cur);
            s->tone_names[tone_id] = (const char *) xmlStrdup(cur->name);
            if ((x = xmlGetProp(cur, (const xmlChar *) "domain")))
                s->tone_domains[tone_id] = (const char *) xmlStrdup(x);
            /*endif*/
            s->tone_types[tone_id] = ST_TYPE_RINGBACK;
        }
        else if (xmlStrcmp(cur->name, (const xmlChar *) "busy-tone") == 0)
        {
            printf("Hit %s\n", cur->name);
            tone_id = super_tone_rx_add_tone(s->tone_detector);
            s->tone[ST_TYPE_BUSY] = NULL;
            parse_tone(s->tone_detector, tone_id, &s->tone[ST_TYPE_BUSY], doc, ns, cur);
            s->tone_names[tone_id] = strdup((const char *) cur->name);
            if ((x = xmlGetProp(cur, (const xmlChar *) "domain")))
                s->tone_domains[tone_id] = (const char *) xmlStrdup(x);
            /*endif*/
            s->tone_types[tone_id] = ST_TYPE_BUSY;
            /* Usually the disconnect tone is the same as busy tone,
               so use busy tone if we have not found a specifc
               disconnect tone. */
            if (s->tone[ST_TYPE_DISCONNECTED] == NULL)
                s->tone[ST_TYPE_DISCONNECTED] = s->tone[ST_TYPE_BUSY];
            /*endif*/
        }
        else if (xmlStrcmp(cur->name, (const xmlChar *) "number-unobtainable-tone") == 0)
        {
            printf("Hit %s\n", cur->name);
            tone_id = super_tone_rx_add_tone(s->tone_detector);
            s->tone[ST_TYPE_NU] = NULL;
            parse_tone(s->tone_detector, tone_id, &s->tone[ST_TYPE_NU], doc, ns, cur);
            s->tone_names[tone_id] = strdup((const char *) cur->name);
            if ((x = xmlGetProp(cur, (const xmlChar *) "domain")))
                s->tone_domains[tone_id] = strdup((const char *) x);
            /*endif*/
            s->tone_types[tone_id] = ST_TYPE_NU;
        }
        else if (xmlStrcmp(cur->name, (const xmlChar *) "congestion-tone") == 0)
        {
            printf("Hit %s\n", cur->name);
            tone_id = super_tone_rx_add_tone(s->tone_detector);
            s->tone[ST_TYPE_CONGESTED] = NULL;
            parse_tone(s->tone_detector, tone_id, &s->tone[ST_TYPE_CONGESTED], doc, ns, cur);
            s->tone_names[tone_id] = (const char *) xmlStrdup(cur->name);
            if ((x = xmlGetProp(cur, (const xmlChar *) "domain")))
                s->tone_domains[tone_id] = strdup((const char *) x);
            /*endif*/
            s->tone_types[tone_id] = ST_TYPE_CONGESTED;
        }
        else if (xmlStrcmp(cur->name, (const xmlChar *) "disconnect-tone") == 0)
        {
            printf("Hit %s\n", cur->name);
            tone_id = super_tone_rx_add_tone(s->tone_detector);
            s->tone[ST_TYPE_DISCONNECTED] = NULL;
            parse_tone(s->tone_detector, tone_id, &s->tone[ST_TYPE_DISCONNECTED], doc, ns, cur);
            s->tone_names[tone_id] = strdup((const char *) cur->name);
            if ((x = xmlGetProp(cur, (const xmlChar *) "domain")))
                s->tone_domains[tone_id] = (const char *) xmlStrdup(x);
            /*endif*/
            s->tone_types[tone_id] = ST_TYPE_DISCONNECTED;
        }
        /*endif*/
        cur = cur->next;
    }
    /*endwhile*/
}
/*- End of function --------------------------------------------------------*/

super_tone_set_t *get_supervisory_tone_set(char *set_id)
{
    xmlDocPtr doc;
    xmlNsPtr ns;
    xmlNodePtr cur;
#if 0
    xmlValidCtxt valid;
#endif
    xmlChar *x;
    super_tone_set_t *s;
    char *tone_file;
    int slot;

    /* See if we have this set loaded already */
    for (slot = 0;  slot < MAX_TONE_SETS;  slot++)
    {
        if (sets[slot].uncode  &&  strcmp(sets[slot].uncode, set_id) == 0)
            return &sets[slot];
        /*endif*/
    }
    /*endfor*/

    /* Find a free slot to use */
    for (slot = 0;  slot < MAX_TONE_SETS;  slot++)
    {
        if (sets[slot].uncode == NULL)
            break;
        /*endif*/
    }
    /*endfor*/
    
    if (slot >= MAX_TONE_SETS)
        return NULL;
    /*endif*/
    
    s = &sets[slot];

    tone_file = DATADIR "/global-tones.xml";
    if ((s->tone_detector = super_tone_rx_make_descriptor(NULL)) == NULL)
        return NULL;
    /*endif*/

    xmlKeepBlanksDefault(0);
    xmlCleanupParser();
    doc = xmlParseFile(tone_file);
    if (doc == NULL)
    {
        fprintf(stderr, "No document\n");
        return NULL;
    }
    /*endif*/
    xmlXIncludeProcess(doc);
#if 0
    if (!xmlValidateDocument(&valid, doc))
    {
        fprintf(stderr, "Invalid document\n");
        return NULL;
    }
    /*endif*/
#endif
    /* Check the document is of the right kind */
    if ((cur = xmlDocGetRootElement(doc)) == NULL)
    {
        fprintf(stderr, "Empty document\n");
        xmlFreeDoc(doc);
        return NULL;
    }
    /*endif*/
    if (xmlStrcmp(cur->name, (const xmlChar *) "global-tones"))
    {
        fprintf(stderr, "Wrong type of document. Root node is not global-tones");
        xmlFreeDoc(doc);
        return NULL;
    }
    /*endif*/
    cur = cur->xmlChildrenNode;
    while (cur  &&  xmlIsBlankNode(cur))
        cur = cur->next;
    /*endwhile*/
    if (cur == NULL)
        return NULL;
    /*endif*/
    while (cur)
    {
        if (xmlStrcmp(cur->name, (const xmlChar *) "tone-set") == 0)
        {
            if ((x = xmlGetProp(cur, (const xmlChar *) "uncode")))
            {
                if (xmlStrcmp(x, (const xmlChar *) set_id) == 0)
                {
                    sets[slot].uncode = (const char *) xmlStrdup(x);
                    if ((x = xmlGetProp(cur, (const xmlChar *) "country")))
                        sets[slot].country = (const char *) xmlStrdup(x);
                    /*endif*/
                    parse_tone_set(s, doc, ns, cur);
                    xmlFreeDoc(doc);
                    return s;
                }
                /*endif*/
            }
            /*endif*/
        }
        /*endif*/
        cur = cur->next;
    }
    /*endwhile*/
    xmlFreeDoc(doc);
    free(s->tone_detector);
    return NULL;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
