/*
 * LibStream - a library for media streaming.
 *
 * unaligned.h - Cross platform unaligned data access
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2006 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: unaligned.h,v 1.3 2007/02/10 11:53:15 steveu Exp $
 */

/*! \file */

#if !defined(_UNALIGNED_H_)
#define _UNALIGNED_H_

#if defined(__cplusplus)
extern "C" {
#endif

#if defined(__GNUC__)

/* If we just tell GCC what's going on, we can trust it to behave optimally */
static __inline__ uint32_t get_unaligned_uint32(const void *p)
{
    const struct { uint32_t d; } __attribute__((packed)) *pp = p;

    return pp->d;
}
/*- End of function --------------------------------------------------------*/

static __inline__ uint16_t get_unaligned_uint16(const void *p)
{
    const struct { uint16_t d; } __attribute__((packed)) *pp = p;

    return pp->d;
}
/*- End of function --------------------------------------------------------*/

static __inline__ void put_unaligned_uint32(void *p, uint32_t datum)
{
    struct { uint32_t d; } __attribute__((packed)) *pp = p;

    pp->d = datum;
}
/*- End of function --------------------------------------------------------*/

static __inline__ void put_unaligned_uint16(void *p, uint16_t datum)
{
    struct { uint16_t d; } __attribute__((packed)) *pp = p;

    pp->d = datum;
}
/*- End of function --------------------------------------------------------*/

#elif defined(SOLARIS)  &&  defined(__sparc__)

static __inline__ uint32_t get_unaligned_uint32(const void *p)
{
    const uint8_t *cp = p;

    return (cp[0] << 24) | (cp[1] << 16) | (cp[2] << 8) | cp[3];
}
/*- End of function --------------------------------------------------------*/

static __inline__ uint16_t get_unaligned_uint16(const void *p)
{
    const uint8_t *cp = p;

    return (cp[0] << 8) | cp[1] ;
}
/*- End of function --------------------------------------------------------*/

static __inline__ void put_unaligned_uint32(void *p, uint32_t datum)
{
    const uint8_t *cp = p;

    cp[0] = datum >> 24;
    cp[1] = datum >> 16;
    cp[2] = datum >> 8;
    cp[3] = datum;
}
/*- End of function --------------------------------------------------------*/

static __inline__ void put_unaligned_uint16(void *p, uint16_t datum)
{
    uint8_t *cp = p;

    cp[0] = datum >> 8;
    cp[1] = datum;
}
/*- End of function --------------------------------------------------------*/

#else

/* The generic case. Assume we can handle direct load/store. */
#define get_unaligned_uint32(p) (*((uint32_t *) (p)))
#define get_unaligned_uint16(p) (*((uint16_t *) (p)))
#define put_unaligned_uint32(p,d) do { uint32_t *__P = (p); *__P = d; } while(0)
#define put_unaligned_uint16(p,d) do { uint16_t *__P = (p); *__P = d; } while(0)
#endif

#if defined(__cplusplus)
}
#endif

#endif
/*- End of file ------------------------------------------------------------*/
