/*
 * LibStream - a library for media streaming.
 *
 * rfc2198_tests.c - RTP redundancy tests
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2007 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: rfc2198_tests.c,v 1.7 2007/02/12 14:01:50 steveu Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <math.h>
#include <memory.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <sys/select.h>
#include <sys/time.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <netdb.h>

#include "libstream.h"

#define FALSE 0
#define TRUE (!FALSE)

#define STREAM_LEN  10000000

rfc3550_state_t rfc3550_state;
rfc2198_state_t rfc2198_state;

uint8_t buffer[STREAM_LEN];
int total_len;
int rx_len;

/* Use a local random generator, so the results are consistent across platforms */
static int my_rand(void)
{
    static int rndnum = 1234567;

    return (rndnum = 1664525U*rndnum + 1013904223U);
}
/*- End of function --------------------------------------------------------*/

static int process_rx_rfc2198_packet(void *user_data,
                                     uint16_t seq_no,
                                     uint32_t timestamp,
                                     uint32_t ssrc,
                                     int payload_type,
                                     int mark,
                                     const uint8_t buf[],
                                     int len)
{
    static int next_seq_no = 0;
    static uint32_t next_timestamp = 0;

    if (seq_no != (next_seq_no & 0xFFFF))
    {
        printf("Test failed - seq_no=%d, next_seq_no=%d\n", seq_no, next_seq_no);
        exit(2);
    }
    if (timestamp != next_timestamp)
    {
        printf("Test failed - timestamp=%" PRIu32 ", next_timestamp=%" PRIu32 "\n", timestamp, next_timestamp);
        exit(2);
    }
    //printf("Seq %d, type %d, len %d\n", seq_no, payload_type, len);
    if (memcmp(buffer + rx_len, buf, len) != 0)
    {
        printf("Test failed - len=%d, rx_len=%d, %02x %02x %02x %02x\n", len, rx_len, buf[0], buffer[rx_len], buf[len - 1], buffer[rx_len + len - 1]);
        exit(2);
    }
    rx_len += len;
    next_seq_no++;
    next_timestamp += 42;
    return 0;
}
/*- End of function --------------------------------------------------------*/

static void back_to_back_test(int loss_rate)
{
    int i;
    int msg_len;
    uint8_t buf[4096];
    int len;
    int payload_type;
    uint32_t timestamp;
    int pos;
    int payload_ok[128];
    int sent_pkts;
    int dropped_pkts;
    
    /* Send packets between two instances, with a low rate of random packet loss. See that the received packet stream has no gaps. */
    /* TODO: Even with a low rate of loss, it is possible for gaps to occur, so the outcome of this test is not deterministic,
             and really ought to be made so. */
    
    printf("Back to back lossy testing, loss rate 1/%d\n", loss_rate);
    if ((rfc3550_init(&rfc3550_state, 1)) == NULL)
    {
        printf("Test failed\n");
        exit(2);
    }
    if ((rfc2198_init(&rfc2198_state, &rfc3550_state, 3, 42)) == NULL)
    {
        printf("Test failed\n");
        exit(2);
    }
    for (i = 0;  i <= 0x7F;  i++)
        payload_ok[i] = rfc3550_set_payload_handler(&rfc3550_state, i, process_rx_rfc2198_packet, NULL);

    /* Generate some test data */
    for (i = 0;  i < STREAM_LEN;  i++)
        buffer[i] = my_rand() & 0xFF;

    /* Now pass it, in random sized pieces, through the protocol software */
    rx_len = 0;
    sent_pkts = 0;
    dropped_pkts = 0;
    for (i = 0, pos = 0, payload_type = 0, timestamp = 0;  pos < STREAM_LEN - 256;  i++, pos += msg_len, payload_type++, timestamp += 42)
    {
        /* Skip invalid payload types */
        while (payload_ok[payload_type & 0x7F])
            payload_type++;
        //printf("Pkt %d\n", i);
        msg_len = my_rand() & 0xFF;
        len = rfc2198_build_packet(&rfc2198_state, buf, timestamp, payload_type & 0x7F, 0, buffer + pos, msg_len);
#if 0
        {
            int i;
        
            printf("Msg");
            for (i = 0;  i < len;  i++)
                printf(" %02X", buf[i]);
            printf("\n");
        }
#endif
        /* Drop some packets. There is no protection here against dropping a burst
           long enough to cause the test to fail. */
        if (my_rand()%loss_rate)
        {
            //printf("Sending %d - %d\n", i, len);
            rfc2198_process_rx_packet((void *) &rfc2198_state,
                                      i,
                                      timestamp,
                                      1234,
                                      42,
                                      0,
                                      buf,
                                      len);
            sent_pkts++;
        }
        else
        {
            //printf("Dropping %d - %d\n", i, len);
            dropped_pkts++;
        }
    }
    printf("%d packets - %d sent, %d dropped\n", i, sent_pkts, dropped_pkts);
    if (pos != rx_len)
    {
        printf("Test failed\n");
        exit(2);
    }
}
/*- End of function --------------------------------------------------------*/

int main(int argc, char *argv[])
{
    int i;
    int loss_rate;
    
    loss_rate = 25;
    for (i = 1;  i < argc;  i++)
    {
        if (strcmp(argv[i], "-l") == 0)
        {
            loss_rate = atoi(argv[++i]);
            continue;
        }
    }

    back_to_back_test(loss_rate);

    printf("Tests passed\n");
    return  0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
