/*
 * LibStream - a library for media streaming.
 *
 * t38_tests.c - Use T.38 to exercise most forms of streaming
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2007 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: t38_tests.c,v 1.2 2007/02/13 13:38:52 steveu Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <unistd.h>
#include <sys/uio.h>
#include <math.h>
#include <memory.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <sys/select.h>
#include <sys/time.h>
#include <netdb.h>
#include <fcntl.h>

#include <tiffio.h>
#include <spandsp.h>

#include "libstream.h"

#define INPUT_FILE_NAME         "../../spandsp-0.0.3/itutests/fax/itutests.tif"
#define OUTPUT_FILE_NAME        "t38.tif"

typedef struct
{
    t38_terminal_state_t t38_state;

    int transport;

    tcp_state_t *tcp_state;
    udp_state_t *udp_state;
    tpkt_state_t tpkt_state;
    udptl_state_t udptl_state;
    rfc2198_state_t rfc2198_state;
    rfc3550_state_t rfc3550_state;
} tx_packet_state_t;

tx_packet_state_t tx_packet_state;

int done[2] = {FALSE, FALSE};
int succeeded[2] = {FALSE, FALSE};

static void phase_b_handler(t30_state_t *s, void *user_data, int result)
{
    int i;
    
    i = (int) (intptr_t) user_data;
    printf("%c: Phase B handler on channel %c - (0x%X) %s\n", i, i, result, t30_frametype(result));
}
/*- End of function --------------------------------------------------------*/

static void phase_d_handler(t30_state_t *s, void *user_data, int result)
{
    int i;
    t30_stats_t t;
    char ident[21];

    i = (int) (intptr_t) user_data;
    printf("%c: Phase D handler on channel %c - (0x%X) %s\n", i, i, result, t30_frametype(result));
    t30_get_transfer_statistics(s, &t);
    printf("%c: Phase D: bit rate %d\n", i, t.bit_rate);
    printf("%c: Phase D: ECM %s\n", i, (t.error_correcting_mode)  ?  "on"  :  "off");
    printf("%c: Phase D: pages transferred %d\n", i, t.pages_transferred);
    printf("%c: Phase D: image size %d x %d\n", i, t.width, t.length);
    printf("%c: Phase D: image resolution %d x %d\n", i, t.x_resolution, t.y_resolution);
    printf("%c: Phase D: bad rows %d\n", i, t.bad_rows);
    printf("%c: Phase D: longest bad row run %d\n", i, t.longest_bad_row_run);
    printf("%c: Phase D: coding method %s\n", i, t4_encoding_to_str(t.encoding));
    printf("%c: Phase D: image size %d\n", i, t.image_size);
    t30_get_local_ident(s, ident);
    printf("%c: Phase D: local ident '%s'\n", i, ident);
    t30_get_far_ident(s, ident);
    printf("%c: Phase D: remote ident '%s'\n", i, ident);
}
/*- End of function --------------------------------------------------------*/

static void phase_e_handler(t30_state_t *s, void *user_data, int result)
{
    int i;
    t30_stats_t t;
    char ident[21];
    
    i = (int) (intptr_t) user_data;
    printf("%c: Phase E handler on channel %c - (%d) %s\n", i, i, result, t30_completion_code_to_str(result));
    t30_get_transfer_statistics(s, &t);
    printf("%c: Phase E: bit rate %d\n", i, t.bit_rate);
    printf("%c: Phase E: ECM %s\n", i, (t.error_correcting_mode)  ?  "on"  :  "off");
    printf("%c: Phase E: pages transferred %d\n", i, t.pages_transferred);
    printf("%c: Phase E: image size %d x %d\n", i, t.width, t.length);
    printf("%c: Phase E: image resolution %d x %d\n", i, t.x_resolution, t.y_resolution);
    printf("%c: Phase E: bad rows %d\n", i, t.bad_rows);
    printf("%c: Phase E: longest bad row run %d\n", i, t.longest_bad_row_run);
    printf("%c: Phase E: coding method %s\n", i, t4_encoding_to_str(t.encoding));
    printf("%c: Phase E: image size %d bytes\n", i, t.image_size);
    t30_get_local_ident(s, ident);
    printf("%c: Phase E: local ident '%s'\n", i, ident);
    t30_get_far_ident(s, ident);
    printf("%c: Phase E: remote ident '%s'\n", i, ident);
    succeeded[i - 'A'] = (result == T30_ERR_OK)  &&  (t.pages_transferred == 12);
    //done[i - 'A'] = TRUE;
}
/*- End of function --------------------------------------------------------*/

static int process_rx_tpkt_packet(void *user_data, const uint8_t buf[], int len)
{
    static int seq_no = 0;

    t38_core_rx_ifp_packet(&tx_packet_state.t38_state.t38, seq_no++, buf, len);
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int process_rx_udptl_packet(void *user_data, int seq_no, const uint8_t *buf, int len)
{
    t38_core_rx_ifp_packet(&tx_packet_state.t38_state.t38, seq_no, buf, len);
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int process_rx_rtp_packet(void *user_data,
                                  uint16_t seq_no,
                                  uint32_t timestamp,
                                  uint32_t ssrc,
                                  int payload_type,
                                  int mark,
                                  const uint8_t buf[],
                                  int len)
{
    printf("Got packet - ssrc=%" PRIu32 ", ts=%" PRIu32 ", pt=%d, mark=%d\n", ssrc, timestamp, payload_type, mark);
    t38_core_rx_ifp_packet(&tx_packet_state.t38_state.t38, seq_no, buf, len);
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int tx_packet_handler(t38_core_state_t *s, void *user_data, const uint8_t *buf, int len, int count)
{
    uint8_t tpkt_buf[4096];
    int tpkt_len;
    int i;
    tx_packet_state_t *tx;
    uint8_t rfc3550_buf[4096];
    int rfc3550_len;
    uint8_t rfc2198_buf[4096];
    int rfc2198_len;
    int payload_type;
    int mark;
    uint32_t ts;
    uint8_t udptl_buf[4096];
    int udptl_len;
    static uint32_t tx_ts = 0;

    span_log(&s->logging, SPAN_LOG_FLOW, "Send seq %d, len %d, count %d\n", s->tx_seq_no, len, count);

    printf("IFP: ");
    for (i = 0;  i < len;  i++)
        printf(" %02X", buf[i]);
    printf("\n");

    tx = (tx_packet_state_t *) user_data;
    
    switch (tx_packet_state.transport)
    {
    case 0:
#if 0
        tpkt_len = tpkt_build_packet(&tx->tpkt_state, tpkt_buf, buf, len);
        tcp_socket_send(tx->tcp_state, tpkt_buf, tpkt_len);
#else
        tpkt_len = tpkt_build_header(&tx->tpkt_state, tpkt_buf, buf, len);
        tcp_socket_send_with_header(tx->tcp_state, tpkt_buf, tpkt_len, buf, len);
#endif
        break;
    case 1:
        udptl_len = udptl_build_packet(&tx->udptl_state, udptl_buf, buf, len);

        for (i = 0;  i < count;  i++)
        {
            if (udp_socket_send(tx->udp_state, udptl_buf, udptl_len, 0) < 0)
                printf("send error - %d\n", errno);
        }
        break;
    case 2:
        payload_type = 42;
        ts = tx_ts++;
        mark = 0;
        rfc2198_len = rfc2198_build_packet(&tx->rfc2198_state, rfc2198_buf, ts, payload_type, 0, buf, len);
        rfc3550_len = rfc3550_build_rtp_packet(&tx->rfc3550_state, rfc3550_buf, ts, 43, mark, rfc2198_buf, rfc2198_len);
        for (i = 0;  i < count;  i++)
        {
            if (udp_socket_send(tx->udp_state, rfc3550_buf, rfc3550_len, 0) < 0)
                printf("send error - %d\n", errno);
        }
        break;
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int t38_setup(int t38_version, int caller, int use_ecm, const char *input_file_name)
{
    if (t38_terminal_init(&tx_packet_state.t38_state, caller, tx_packet_handler, &tx_packet_state) == NULL)
    {
        fprintf(stderr, "Cannot start the T.38 channel\n");
        exit(2);
    }
    t38_set_t38_version(&tx_packet_state.t38_state.t38, t38_version);
    span_log_set_level(&tx_packet_state.t38_state.logging, SPAN_LOG_DEBUG | SPAN_LOG_SHOW_TAG | SPAN_LOG_SHOW_SAMPLE_TIME);
    span_log_set_tag(&tx_packet_state.t38_state.logging, "T.38");
    span_log_set_level(&tx_packet_state.t38_state.t38.logging, SPAN_LOG_DEBUG | SPAN_LOG_SHOW_TAG | SPAN_LOG_SHOW_SAMPLE_TIME);
    span_log_set_tag(&tx_packet_state.t38_state.t38.logging, "T.38");
    span_log_set_level(&tx_packet_state.t38_state.t30_state.logging, SPAN_LOG_DEBUG | SPAN_LOG_SHOW_TAG | SPAN_LOG_SHOW_SAMPLE_TIME);
    span_log_set_tag(&tx_packet_state.t38_state.t30_state.logging, "T.38");
    if (use_ecm)
        t30_set_supported_compressions(&tx_packet_state.t38_state.t30_state, T30_SUPPORT_T4_1D_COMPRESSION | T30_SUPPORT_T4_2D_COMPRESSION | T30_SUPPORT_T6_COMPRESSION);
    t30_set_phase_b_handler(&tx_packet_state.t38_state.t30_state, phase_b_handler, (void *) (intptr_t) 'A');
    t30_set_phase_d_handler(&tx_packet_state.t38_state.t30_state, phase_d_handler, (void *) (intptr_t) 'A');
    t30_set_phase_e_handler(&tx_packet_state.t38_state.t30_state, phase_e_handler, (void *) (intptr_t) 'A');
    t30_set_ecm_capability(&tx_packet_state.t38_state.t30_state, use_ecm);

    if (caller)
    {
        t30_set_local_ident(&tx_packet_state.t38_state.t30_state, "11111111");
        t30_set_tx_file(&tx_packet_state.t38_state.t30_state, input_file_name, -1, -1);
    }
    else
    {
        t30_set_local_ident(&tx_packet_state.t38_state.t30_state, "22222222");
        t30_set_rx_file(&tx_packet_state.t38_state.t30_state, OUTPUT_FILE_NAME, -1);
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int tpkt_handling(int caller, int local_port, const char *far_host, int far_port)
{
    struct sockaddr_in local;
    struct sockaddr_in far;
    struct hostent *hp;
    int len;
    uint8_t buf[2048];
    fd_set rfds;
    struct timeval tv;
    tcp_state_t *s;
    int res;
    int i;
    
    if (caller)
    {
        i = local_port;
        local_port = far_port;
        far_port = i;
    }
    else
    {
        far_port = -1;
    }
    tpkt_init(&tx_packet_state.tpkt_state, process_rx_tpkt_packet, NULL);
    if ((hp = gethostbyname(far_host)) == NULL) 
    {
        printf("Invalid address for far end server\n");
        exit(2);
    }

    memset(&local, '\0', sizeof (struct sockaddr_in));
    local.sin_family = AF_INET;
    local.sin_addr.s_addr = htonl(INADDR_ANY);
    local.sin_port = htons(local_port);

    s = tcp_socket_create_and_bind(&local.sin_addr, local_port, local_port + 10);

    far.sin_family = AF_INET;
    memcpy(&far.sin_addr, hp->h_addr, sizeof(far.sin_addr));
    far.sin_port = htons(far_port);
    tcp_socket_set_far(s, &far);

    printf("local_port %d, far port is %d\n", local_port, far_port);
    if (far_port < 0)
    {
        if (tcp_socket_listen(s) < 0)
        {
            printf("Listen failed - errno = %d", errno);
            exit(2);
        }
        /*endif*/
        FD_ZERO(&rfds);
        FD_SET(tcp_socket_fd(s), &rfds);
        if (select(tcp_socket_fd(s) + 1, &rfds, NULL, NULL, NULL) <= 0)
        {
            printf("Wait for accept failed - errno = %d", errno);
            exit(2);
        }
        while ((tx_packet_state.tcp_state = tcp_socket_accept(s)) == NULL)
        {
            if (errno != EAGAIN  &&  errno != EWOULDBLOCK)
            {
                printf("Accept failed - errno = %d", errno);
                exit(2);
            }
            /*endif*/
        }
        /*endwhile*/
    }
    else
    {
        if (tcp_socket_connect(s) < 0)
        {
            printf("Connect failed - errno = %d", errno);
        	exit (2);
        }
        printf("Connecting\n");
        /* Wait until writable */
        FD_ZERO(&rfds);
        FD_SET(tcp_socket_fd(s), &rfds);
        if (select(tcp_socket_fd(s) + 1, NULL, &rfds, NULL, NULL) <= 0)
        {
            printf("Wait for connect failed - errno = %d", errno);
            exit(2);
        }
        if (tcp_socket_connection_result(s) < 0)
        {
            printf("Connection result %d\n", errno);
            exit(2);
        }
        printf("Connected\n");
        tx_packet_state.tcp_state = s;
    }
    tv.tv_sec = 0;
    tv.tv_usec = 0;

    for (;;)
    {
        FD_ZERO(&rfds);
        FD_SET(tcp_socket_fd(tx_packet_state.tcp_state), &rfds);
        tv.tv_sec = 0;
        tv.tv_usec = 20000 - tv.tv_usec;
        res = select(tcp_socket_fd(tx_packet_state.tcp_state) + 1, &rfds, NULL, NULL, &tv);
        if (res > 0)
        {
            if ((len = tcp_socket_recv(tx_packet_state.tcp_state, buf, 2048, 0)) < 0)
            {
                printf("Error (%d) %s\n", errno, strerror(errno));
                break;
            }
            tpkt_rx_chunk(&tx_packet_state.tpkt_state, buf, len);
        }
        if (tv.tv_usec == 0)
        {
            done[0] = t38_terminal_send_timeout(&tx_packet_state.t38_state, 160);
        }
    }
}
/*- End of function --------------------------------------------------------*/

static int udptl_handling(int caller, int local_port, const char *far_host, int far_port, int redundancy_config)
{
    int i;
    struct sockaddr_in farx;
    struct sockaddr_in far;
    struct hostent *hp;
    socklen_t sock_len;
    int len;
    uint8_t buf[2048];
    fd_set rfds;
    struct timeval tv;
    int res;
    int action;
    
    if (!caller)
    {
        i = far_port;
        far_port = local_port;
        local_port = i;
    }

    if ((hp = gethostbyname(far_host)) == NULL) 
    {
        printf("Invalid address for far end server\n");
        exit(2);
    }
    farx.sin_family = AF_INET;
    memcpy(&farx.sin_addr, hp->h_addr, sizeof(farx.sin_addr));
    farx.sin_port = htons(far_port);

    tx_packet_state.udp_state = udp_socket_create_and_bind(1, NULL, local_port, local_port + 1);
    if (tx_packet_state.udp_state == NULL)
    {
        printf("Cannot allocate the ports\n");
        exit(2);
    }
    udp_socket_set_nat(tx_packet_state.udp_state, TRUE);
    udp_socket_set_far(tx_packet_state.udp_state, &farx);

    tv.tv_sec = 0;
    tv.tv_usec = 0;
    
    if ((udptl_init(&tx_packet_state.udptl_state, redundancy_config, 3, 3, process_rx_udptl_packet, NULL)) == NULL)
    {
        printf("Test failed\n");
        exit(2);
    }

    for (;;)
    {
        FD_ZERO(&rfds);
        FD_SET(udp_socket_fd(tx_packet_state.udp_state), &rfds);
        tv.tv_sec = 0;
        tv.tv_usec = 20000 - tv.tv_usec;
        res = select(udp_socket_fd(tx_packet_state.udp_state) + 1, &rfds, NULL, NULL, &tv);
        if (res > 0)
        {
            sock_len = sizeof(far);
            if ((len = udp_socket_recvfrom(tx_packet_state.udp_state,
                                           buf,
                                           sizeof(buf),
                                           0,
                                           (struct sockaddr *) &far,
                                           &sock_len,
                                           &action)) < 0)
            {
                printf("Error %d\n", errno);
                continue;
            }
#if 1
            printf("Packet from 0x%lx:%d\n", (unsigned long int) far.sin_addr.s_addr, ntohs(far.sin_port));
            for (i = 0;  i < len;  i++)
                printf(" %02X", buf[i]);
            printf("\n");
#endif
            if (far.sin_addr.s_addr == farx.sin_addr.s_addr
                &&
                far.sin_port == farx.sin_port)
            {
                udptl_rx_packet(&tx_packet_state.udptl_state, buf, len);
            }
        }
        if (tv.tv_usec == 0)
        {
            done[0] = t38_terminal_send_timeout(&tx_packet_state.t38_state, 160);
        }
    }
    return  0;
}
/*- End of function --------------------------------------------------------*/

static int rtp_handling(int caller, int local_port, const char *far_host, int far_port)
{
    int i;
    struct sockaddr_in farx;
    struct sockaddr_in far;
    struct hostent *hp;
    socklen_t sock_len;
    int len;
    uint8_t buf[2048];
    fd_set rfds;
    struct timeval tv;
    int res;
    int action;
    
    if (!caller)
    {
        i = far_port;
        far_port = local_port;
        local_port = i;
    }

    if ((hp = gethostbyname(far_host)) == NULL) 
    {
        printf("Invalid address for far end server\n");
        exit(2);
    }
    farx.sin_family = AF_INET;
    memcpy(&farx.sin_addr, hp->h_addr, sizeof(farx.sin_addr));
    farx.sin_port = htons(far_port);

    tx_packet_state.udp_state = udp_socket_group_create_and_bind(2, 1, NULL, local_port, local_port + 1);
    if (tx_packet_state.udp_state == NULL)
    {
        printf("Cannot allocate the ports\n");
        exit(2);
    }
    udp_socket_set_nat(tx_packet_state.udp_state, TRUE);
    udp_socket_set_far(tx_packet_state.udp_state, &farx);

    tv.tv_sec = 0;
    tv.tv_usec = 0;
    
    if ((rfc3550_init(&tx_packet_state.rfc3550_state, TRUE)) == NULL)
    {
        printf("Test failed\n");
        exit(2);
    }
    if ((rfc2198_init(&tx_packet_state.rfc2198_state, &tx_packet_state.rfc3550_state, 3, 43)) == NULL)
    {
        printf("Test failed\n");
        exit(2);
    }
    rfc3550_set_payload_handler(&tx_packet_state.rfc3550_state, 42, process_rx_rtp_packet, NULL);

    for (;;)
    {
        FD_ZERO(&rfds);
        FD_SET(udp_socket_fd(tx_packet_state.udp_state), &rfds);
        tv.tv_sec = 0;
        tv.tv_usec = 20000 - tv.tv_usec;
        res = select(udp_socket_fd(tx_packet_state.udp_state) + 1, &rfds, NULL, NULL, &tv);
        if (res > 0)
        {
            sock_len = sizeof(far);
            if ((len = udp_socket_recvfrom(tx_packet_state.udp_state,
                                           buf,
                                           sizeof(buf),
                                           0,
                                           (struct sockaddr *) &far,
                                           &sock_len,
                                           &action)) < 0)
            {
                printf("Error %d\n", errno);
                continue;
            }
#if 1
            printf("Packet from 0x%lx:%d\n", (unsigned long int) far.sin_addr.s_addr, ntohs(far.sin_port));
            for (i = 0;  i < len;  i++)
                printf(" %02X", buf[i]);
            printf("\n");
#endif
printf("Ports  %d %d\n", ntohs(far.sin_port), ntohs(farx.sin_port));
#if 0
            if (far.sin_addr.s_addr == farx.sin_addr.s_addr
                &&
                far.sin_port == farx.sin_port)
#endif
            {
                rfc3550_rx_rtp_packet(&tx_packet_state.rfc3550_state, buf, len);
            }
        }
        if (tv.tv_usec == 0)
        {
            done[0] = t38_terminal_send_timeout(&tx_packet_state.t38_state, 160);
        }
    }
    return  0;
}
/*- End of function --------------------------------------------------------*/

int main(int argc, char *argv[])
{
    int t38_version;
    int redundancy_config;
    int i;
    int use_ecm;
    int caller;
    const char *input_file_name;
    int local_port;
    int far_port;
    const char *far_host;
    int transport;

    t38_version = 1;
    redundancy_config = UDPTL_ERROR_CORRECTION_REDUNDANCY;
    input_file_name = INPUT_FILE_NAME;
    transport = 0;
    use_ecm = FALSE;
    caller = FALSE;
    local_port = 10024;
    far_port = 10026;
    far_host = "127.0.0.1";
    for (i = 1;  i < argc;  i++)
    {
        if (strcmp(argv[i], "-c") == 0)
        {
            caller = TRUE;
            continue;
        }
        if (strcmp(argv[i], "-e") == 0)
        {
            use_ecm = TRUE;
            continue;
        }
        if (strcmp(argv[i], "-h") == 0)
        {
            far_host = argv[++i];
            continue;
        }
        if (strcmp(argv[i], "-i") == 0)
        {
            input_file_name = argv[++i];
            continue;
        }
        if (strcmp(argv[i], "-r") == 0)
        {
            redundancy_config = atoi(argv[++i]);
            continue;
        }
        if (strcmp(argv[i], "-t") == 0)
        {
            transport = atoi(argv[++i]);
            continue;
        }
        if (strcmp(argv[i], "-v") == 0)
        {
            t38_version = atoi(argv[++i]);
            continue;
        }
    }

    t38_setup(t38_version, caller, use_ecm, input_file_name);

    tx_packet_state.transport = transport;
    switch (tx_packet_state.transport)
    {
    case 0:
        tpkt_handling(caller, local_port, far_host, far_port);
        break;
    case 1:
        udptl_handling(caller, local_port, far_host, far_port, redundancy_config);
        break;
    case 2:
        rtp_handling(caller, local_port, far_host, far_port);
        break;
    }
    return  0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
