/*
 * LibStream - a library for media streaming.
 *
 * udp_test.c - A test suite for UDP streaming port handling.
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2006 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: udp_tests.c,v 1.8 2007/02/12 14:01:50 steveu Exp $
 */

/*! \file */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <string.h>
#include <sys/time.h>
#include <signal.h>
#include <errno.h>
#include <unistd.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <netdb.h>

#include "libstream.h"
#include "libstream/unaligned.h"

typedef struct
{
    udp_state_t *udp;
    uint32_t tx_seq_no;
    uint32_t rx_seq_no;
} port_group_info_t;

int main(int argc, char *argv[])
{
    port_group_info_t group[1000];
    int i;
    struct sockaddr_in rfc3489_server;
    struct sockaddr_in far;
    struct sockaddr sa;
    socklen_t salen;
    char buf[1000];
    uint8_t test_packet[256];
    int action;
    int len;
    fd_set rfds_ref;
    fd_set rfds;
    struct timeval tv;
    struct hostent *hp;
    int rfc3489_state;
    const struct sockaddr_in *us;
    const char *stun_server;
    const char *far_host;
    int base_port;
    int far_base_port;
    int res;
    int max_fd;
    int groups;

    stun_server = NULL;
    far_host = "127.0.0.1";
    base_port = 10000;
    far_base_port = 12000;
    groups = 20;
    for (i = 1;  i < argc;  i++)
    {
        if (strcmp(argv[i], "-b") == 0)
        {
            base_port = atoi(argv[++i]);
            continue;
        }
        if (strcmp(argv[i], "-F") == 0)
        {
            far_base_port = atoi(argv[++i]);
            continue;
        }
        if (strcmp(argv[i], "-f") == 0)
        {
            far_host = argv[++i];
            continue;
        }
        if (strcmp(argv[i], "-g") == 0)
        {
            groups = atoi(argv[++i]);
            continue;
        }
        if (strcmp(argv[i], "-s") == 0)
        {
            stun_server = argv[++i];
            continue;
        }
    }
 
    printf("Resolving far end server\n");
    if ((hp = gethostbyname(far_host)) == NULL) 
    {
        printf("Invalid address for far end server\n");
        exit(2);
    }
    far.sin_family = AF_INET;
    memcpy(&far.sin_addr, hp->h_addr, sizeof(far.sin_addr));
    far.sin_port = htons(far_base_port);
    if (stun_server)
    {
        printf("Resolving RFC3489 (STUN) server\n");
        if ((hp = gethostbyname(stun_server)) == NULL) 
        {
            printf("Invalid address for STUN server\n");
            exit(2);
        }
        rfc3489_server.sin_family = AF_INET;
        memcpy(&rfc3489_server.sin_addr, hp->h_addr, sizeof(rfc3489_server.sin_addr));
        rfc3489_server.sin_port = htons(3478);
        printf("Initing STUN server\n");
        rfc3489_init(&rfc3489_server);
    }
    
    printf("Initing UDP for %d groups\n", groups);
    for (i = 0;  i < groups;  i++)
    {
        memset(&group[i], 0, sizeof(group[i]));
        group[i].udp = udp_socket_group_create_and_bind(2, 0, NULL, base_port, base_port + 2000);
        if (group[i].udp == NULL)
        {
            printf("Failed at %d\n", i);
            exit(2);
        }
    }
    if (stun_server)
    {
        for (i = 0;  i < groups;  i++)
            udp_socket_set_nat(group[i].udp, TRUE);
    }
    rfc3489_state = RFC3489_STATE_IDLE;

    for (i = 0;  i < groups;  i++)
    {
        /* The port randomisation on both sides should be similar, assuming none of the
           ports in range were already open. */
        us = udp_socket_get_local(group[i].udp);
        far.sin_port = htons(far_base_port + ntohs(us->sin_port) - base_port);
        udp_socket_set_far(group[i].udp, &far);
    }
    tv.tv_sec = 0;
    tv.tv_usec = 0;
    FD_ZERO(&rfds_ref);
    max_fd = -1;
    for (i = 0;  i < groups;  i++)
    {
        if (udp_socket_fd(group[i].udp) > max_fd)
            max_fd = udp_socket_fd(group[i].udp);
        FD_SET(udp_socket_fd(group[i].udp), &rfds_ref);
    }
    printf("Max FD is %d\n", max_fd);
    for (i = 0;  i < 256;  i++)
        test_packet[i] = i;
    for (;;)
    {
        tv.tv_sec = 0;
        tv.tv_usec = 20000 - tv.tv_usec;
        memcpy(&rfds, &rfds_ref, sizeof(rfds));
        res = select(max_fd + 1, &rfds, NULL, NULL, &tv);

        for (i = 0;  i < groups;  i++)
        {
            if (!FD_ISSET(udp_socket_fd(group[i].udp), &rfds_ref))
                continue;
            salen = sizeof(sa);
            len = udp_socket_recvfrom(group[i].udp,
                                      buf,
                                      sizeof(buf),
                                      0,
                                      &sa,
                                      &salen,
                                      &action);
            if (stun_server)
            {
                if (rfc3489_state != RFC3489_STATE_RESPONSE_RECEIVED)
                {
                    rfc3489_state = udp_socket_get_rfc3489_state(group[i].udp);
                    if (rfc3489_state == RFC3489_STATE_RESPONSE_RECEIVED)
                    {
                        us = udp_socket_get_local(group[i].udp);
                        printf("We are %d 0x%X %d\n", us->sin_family, (unsigned int) us->sin_addr.s_addr, ntohs(us->sin_port));
                        us = udp_socket_get_apparent_local(group[i].udp);
                        printf("We appear to be %d 0x%X %d\n", us->sin_family, (unsigned int) us->sin_addr.s_addr, ntohs(us->sin_port));
                    }
                }
            }
            if (len < 0  &&  errno != EAGAIN  &&  errno != ECONNRESET)
                exit(2);
            if (len >= 0)
            {
                //printf("%d - %5d\n", i, len);
                if (ntohl(get_unaligned_uint32(buf)) != i)
                    printf("%d: Bad channel %" PRIu32 " %d\n", i, ntohl(get_unaligned_uint32(buf)), i);
                if (ntohl(get_unaligned_uint32(buf + sizeof(uint32_t))) != group[i].rx_seq_no)
                {
                    if (group[i].rx_seq_no == 0)
                        group[i].rx_seq_no = ntohl(get_unaligned_uint32(buf + sizeof(uint32_t)));
                    else
                        printf("%d: Bad sequence no %" PRIu32 " %" PRIu32 "\n", i, ntohl(get_unaligned_uint32(buf + sizeof(uint32_t))), group[i].rx_seq_no);
                }
                group[i].rx_seq_no++;
            }
        }
        if (res == 0)
        {
            for (i = 0;  i < groups;  i++)
            {
                put_unaligned_uint32(test_packet, htonl(i));
                put_unaligned_uint32(test_packet + sizeof(uint32_t), htonl(group[i].tx_seq_no++));
                if (udp_socket_send(group[i].udp, test_packet, 172, 0) < 0)
                    printf("send error - %d\n", errno);
            }
        }
    }
    printf("Done\n");
    getchar();
    return 0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
