/*
 * LibStream - a library for media streaming.
 *
 * rfc4733.h - RTP payload for DTMF digits, telephony tones and telephony signals
 *             (An update of RFC2833) (includes event definitions from RFC4734)
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2006 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: rfc4733.h,v 1.1 2007/02/13 13:38:51 steveu Exp $
 */

/*! \file */

#if !defined(_RFC4733_H_)
#define _RFC4733_H_

typedef int (rfc4733_event_handler_t)(void *user_data, int volume, int duration, int event_end, int freqs, int frequencies[]);

enum
{
    RFC4733_EVENT_DTMF_0 = 0,
    RFC4733_EVENT_DTMF_1 = 1,
    RFC4733_EVENT_DTMF_2 = 2,
    RFC4733_EVENT_DTMF_3 = 3,
    RFC4733_EVENT_DTMF_4 = 4,
    RFC4733_EVENT_DTMF_5 = 5,
    RFC4733_EVENT_DTMF_6 = 6,
    RFC4733_EVENT_DTMF_7 = 7,
    RFC4733_EVENT_DTMF_8 = 8,
    RFC4733_EVENT_DTMF_9 = 9,
    RFC4733_EVENT_DTMF_STAR = 10,
    RFC4733_EVENT_DTMF_HASH = 11,
    RFC4733_EVENT_DTMF_A = 12,
    RFC4733_EVENT_DTMF_B = 13,
    RFC4733_EVENT_DTMF_C = 14,
    RFC4733_EVENT_DTMF_D = 15,
    RFC4733_EVENT_DTMF_HOOK_FLASH = 16,
    RFC4733_EVENT_CED = 32,
    RFC4733_EVENT_PH_CED = 33,
    RFC4733_EVENT_CNG = 36,
    RFC4733_EVENT_V21_CH1_0 = 37,
    RFC4733_EVENT_V21_CH1_1 = 38,
    RFC4733_EVENT_V21_CH2_0 = 39,
    RFC4733_EVENT_V21_CH2_1 = 40,
    RFC4733_EVENT_V21_PREAMBLE = 54
};

typedef struct
{
    int event_payload_type;
    int tone_payload_type;
    int cisco_dtmf_payload_type;
    rfc3550_state_t *rfc3550_state;
    rfc4733_event_handler_t *handler;
    void *user_data;

    int resp;
} rfc4733_state_t;

#if defined(__cplusplus)
extern "C" {
#endif

const char *rfc4733_event_to_str(int event);

/* Send a signaling event message */
int rfc4733_event_send(rfc4733_state_t *s, uint8_t buf[], int event, int volume, int duration, int end);

/* Send a tone start/end message */
int rfc4733_tone_send(rfc4733_state_t *s, uint8_t buf[], int modulation, int volume, int duration, int freqs, int freq[]);

/* Add support for Cisco's pre-RFC2833/RFC4733 DTMF messages */
int rfc4733_set_cisco_dtmf_type(rfc4733_state_t *s, int cisco_dtmf_pt);

/* Send one of Cisco's pre-RFC2833/RFC4733 DTMF messages */
int rfc4733_cisco_dtmf_send(rfc4733_state_t *s, uint8_t buf[], int event);

/* Initialise an RFC4733 context, and associate it with an existing RFC3550 context. */
rfc4733_state_t *rfc4733_init(rfc4733_state_t *s,
                              rfc3550_state_t *t,
                              int event_pt,
                              int tone_pt,
                              rfc4733_event_handler_t *handler,
                              void *user_data);

/* Release an RFC4733 context */
int rfc4733_release(rfc4733_state_t *s);

#if defined(__cplusplus)
}
#endif

#endif
/*- End of file ------------------------------------------------------------*/
