/*
 * LibStream - a library for media streaming.
 *
 * rfc3389.c - Real-time Transport Protocol (RTP) payload for comfort noise (CN)
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2006 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: rfc3389.c,v 1.4 2007/02/10 14:01:44 steveu Exp $
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <math.h>
#include <memory.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <sys/select.h>
#include <sys/time.h>
#ifdef ENABLE_SRTP
#include <srtp.h>
#endif

#include "libstream/unaligned.h"
#include "libstream/udp.h"
#include "libstream/rfc3550.h"
#include "libstream/rfc3389.h"

#define FALSE 0
#define TRUE (!FALSE)

/*
    Level is in dBov, 0 to -127dBov. The sign is not transmitted, so the
    level is a byte, as follows:

         0 1 2 3 4 5 6 7
        +-+-+-+-+-+-+-+-+
        |0|   level     |
        +-+-+-+-+-+-+-+-+

    Spectral information is sent as reflection coefficients, uniformly quantised
    to 8 bits, from their natural range of -1 to +1. Up to 0 to 254 coefficients
    may be used.
    
    k_i(N_i) = 258*(N_i-127)     for N_i = 0...254; -1 < k_i < 1
               -------------
                  32768

        Byte        1      2    3    ...   M+1
        +-----+-----+-----+-----+-----+
        |level|  N1 |  N2 | ... |  NM |
        +-----+-----+-----+-----+-----+
    
    Comfort noise, using the RFC3389 spec., uses a static payload type of 13, for
    a sampling rate of 8000/second. For other sampling rates, a dynamic payload type
    must be used.
*/

int rfc3389_process_rx_packet(void *user_data,
                              uint16_t seq_no,
                              uint32_t timestamp,
                              uint32_t ssrc,
                              int payload_type,
                              int mark,
                              const uint8_t buf[],
                              int len)
{
    rfc3389_state_t *s;

    s = (rfc3389_state_t *) user_data;
    if (len <= 0  ||  len > 255)
        return -1;
    return 0;
}
/*- End of function --------------------------------------------------------*/

int rfc3389_send(rfc3389_state_t *s, uint8_t buf[], int level, int coeffs, int rc[])
{
    int i;
    int len;
    uint8_t *msg;

    if (s->payload_type < 0)
        return -1;
    level = 127 - (level & 0x7F);
    msg = buf + RFC3550_MAX_HEADER;
    msg[0] = level;
    for (i = 0;  i < coeffs;  i++)
        msg[i + 1] = rc[i];
    len = rfc3550_build_rtp_packet_in_place(s->rfc3550_state,
                                            buf,
                                            RFC3550_MAX_HEADER,
                                            0,
                                            s->payload_type,
                                            0,
                                            coeffs + 1);
    return len;
}
/*- End of function --------------------------------------------------------*/

rfc3389_state_t *rfc3389_init(rfc3389_state_t *s, rfc3550_state_t *t, int payload_type)
{
    if (s == NULL)
    {
        if ((s = (rfc3389_state_t *) malloc(sizeof(*s))) == NULL)
            return NULL;
    }
    memset(s, 0, sizeof(*s));
    s->rfc3550_state = t;
    s->payload_type = payload_type;
    if (payload_type >= 0)
        rfc3550_set_payload_handler(t, payload_type, rfc3389_process_rx_packet, (void *) s);
    return s;
}
/*- End of function --------------------------------------------------------*/

int rfc3389_release(rfc3389_state_t *s)
{
    if (s->payload_type >= 0)
        rfc3550_set_payload_handler(s->rfc3550_state, s->payload_type, NULL, NULL);
    free(s);
    return 0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
