/*
 * LibStream - a library for media streaming.
 *
 * rfc3550_tests.c - RTP core
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2006 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: rfc3550_tests.c,v 1.7 2007/02/19 15:46:14 steveu Exp $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <math.h>
#include <memory.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <sys/select.h>
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#include "libstream.h"

rfc3550_state_t rfc3550_state;

static int process_rx_packet(void *user_data,
                             uint16_t seq_no,
                             uint32_t timestamp,
                             uint32_t ssrc,
                             int payload_type,
                             int mark,
                             const uint8_t buf[],
                             int len)
{
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int rtcp_tests(void)
{
    rfc3550_state_t state;
    rfc3550_state_t *s;
    uint8_t buf[1000];
    int len;
    int i;
    rtcp_sdes_item_t sdes_items[10];
    uint32_t srcs[16];

    printf("RTCP function tests\n");
    s = &state;
    if ((rfc3550_init(s, TRUE)) == NULL)
    {
        printf("Test failed\n");
        exit(2);
    }
    rfc3550_add_csrc(s, 1);
    rfc3550_add_csrc(s, 2);
    rfc3550_add_csrc(s, 3);
    rfc3550_add_csrc(s, 4);
    rfc3550_set_payload_handler(s, 42, process_rx_packet, NULL);
    
    for (i = 0;  i < 16;  i++)
        srcs[i] = 0x57C00001 + i;

    len = rfc3550_build_rtcp_packet_start(s, buf, 0);
    len = rfc3550_build_rtcp_add_sr_rr_start(s, buf, len, 1);
    len = rfc3550_build_rtcp_add_sr_rr_chunk(s, buf, len, &s->source);
    len = rfc3550_build_rtcp_add_sr_rr_end(s, buf, len);
    //len = rfc3550_build_rtcp_add_sr(s, buf, 0);
    len = rfc3550_build_rtcp_add_bye(s, buf, len, 9, srcs, "Parting is such sweet sorrow");
    len = rfc3550_build_rtcp_add_app(s, buf, len, 0x1234, "FRED", "\x01\x02\x03", 3);
    len = rfc3550_build_rtcp_packet_end(s, buf, len);
    printf("RTCP: [%d] ", len);
    for (i = 0;  i < len;  i++)
        printf(" %02X", buf[i]);
    printf("\n");

    if (rfc3550_rx_rtcp_packet(s, buf, len) < 0)
    {
        printf("Test failed - Bad RTCP\n");
        exit(2);
    }

    sdes_items[0].type = RTCP_SDES_CNAME;
    sdes_items[0].length = strlen("cname");
    sdes_items[0].value = "cname";
    sdes_items[1].type = RTCP_SDES_EMAIL;
    sdes_items[1].length = strlen("address@x.com");
    sdes_items[1].value = "address@x.com";
    sdes_items[2].type = RTCP_SDES_NAME;
    sdes_items[2].length = strlen("fred");
    sdes_items[2].value = "fred";
    for (i = 0;  i < 16;  i++)
        srcs[i] = 0x57C00001 + i;
    len = rfc3550_build_rtcp_packet_start(s, buf, 0);
    len = rfc3550_build_rtcp_add_sr_rr_start(s, buf, 0, 0);
    len = rfc3550_build_rtcp_add_sr_rr_chunk(s, buf, len, &s->source);
    len = rfc3550_build_rtcp_add_sr_rr_end(s, buf, len);
    len = rfc3550_build_rtcp_add_bye(s, buf, len, 5, srcs, "Parting is such sweet sorrow");
    len = rfc3550_build_rtcp_add_app(s, buf, len, 0x4321, "fred", "\x01\x02\x03\x04", 4);
    len = rfc3550_build_rtcp_add_sdes_start(s, buf, len);
    len = rfc3550_build_rtcp_add_sdes_chunk(s, buf, len, 0x4321, 3, sdes_items);
    len = rfc3550_build_rtcp_add_sdes_chunk(s, buf, len, 0x1234, 2, sdes_items);
    len = rfc3550_build_rtcp_add_sdes_end(s, buf, len);
    len = rfc3550_build_rtcp_packet_end(s, buf, len);
    printf("RTCP: [%d] ", len);
    for (i = 0;  i < len;  i++)
        printf(" %02X", buf[i]);
    printf("\n");

    if (rfc3550_rx_rtcp_packet(s, buf, len) < 0)
    {
        printf("Test failed - Bad RTCP\n");
        exit(2);
    }
}
/*- End of function --------------------------------------------------------*/

static void ntp_tests(void)
{
    uint64_t when;
    uint32_t reduced_when;
    struct timeval t;
    
    printf("NTP style time function tests\n");
    /* Check that nothing nasty happens just before the fraction rolls over. */
    when = 0x12345FFFFFFFFLLU;
    t = ntp_to_timeval(when);
    printf("12345.999999 NTP -> %" PRId32 ".%06" PRId32 " as a timeval\n", t.tv_sec, t.tv_usec);
    
    when = timeval_to_ntp(&t);
    printf("12345.999999 NTP -> 0x%" PRIx64 " as restored NTP\n", when);
    if (when != 0x12345FFFFEF3D)
    {
        printf("Test failed\n");
        exit(2);
    }

    /* Check that a reasonable 0.999999 value works properly. */
    when = 0x12345FFFFEF30LLU;
    t = ntp_to_timeval(when);
    printf("12345.999998 NTP -> %" PRId32 ".%06" PRId32 " as a timeval\n", t.tv_sec, t.tv_usec);
    
    when = timeval_to_ntp(&t);
    printf("12345.999998 NTP -> 0x%" PRIx64 " as restored NTP\n", when);
    if (when != 0x12345FFFFDE76)
    {
        printf("Test failed\n");
        exit(2);
    }

    when = 0x12345FFFFFFFFLLU;
    reduced_when = ntp_to_reduced_ntp(when);
    printf("0x%" PRIx64 " NTP -> 0x%" PRIx32 " as reduced NTP\n", when, reduced_when);
    if (reduced_when != 0x2345FFFF)
    {
        printf("Test failed\n");
        exit(2);
    }
}
/*- End of function --------------------------------------------------------*/

int main(int argc, char *argv[])
{
    rtcp_tests();
    ntp_tests();

    printf("Tests passed\n");    
    return  0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
