/*
 * LibStream - a library for media streaming.
 *
 * udp_test.c - A test suite for UDP streaming port handling.
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2006 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: udp_tests.c,v 1.12 2007/02/20 13:57:16 steveu Exp $
 */

/*! \file */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <string.h>
#include <sys/time.h>
#include <signal.h>
#include <errno.h>
#include <unistd.h>
#include <netinet/in.h>
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#include <sys/socket.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <netdb.h>

#include <event.h>

#include "libstream.h"
#include "libstream/unaligned.h"

struct event timer_evt;
struct event monitor_evt;

typedef struct
{
    udp_state_t *udp;
    struct event evt;
    int rfc3489_state;
    uint32_t tx_seq_no;
    uint32_t rx_seq_no;
    int tag;
} port_group_info_t;

port_group_info_t group[1000];
int groups = 0;
uint8_t test_packet[256];
const char *stun_server = NULL;
struct timeval target_time;

int total_sent;
int total_received;
int total_seq_misses;

static void monitor_cb(int fd, short int event, void *arg)
{
    struct timeval tv;

    timerclear(&tv);
    tv.tv_sec = 2;
    event_add(&monitor_evt, &tv);
    printf("Monitor - %d %d %d\n", total_sent, total_received, total_seq_misses);
    total_sent = 0;
    total_received = 0;
    total_seq_misses = 0;
}
/*- End of function --------------------------------------------------------*/

static void timeout_cb(int fd, short int event, void *arg)
{
    struct timeval tv;
    int i;

    timerclear(&tv);
    tv.tv_usec = 20000;
    timeradd(&target_time, &tv, &target_time);

    gettimeofday(&tv, NULL);
    timersub(&target_time, &tv, &tv);

    event_add(&timer_evt, &tv);
    for (i = 0;  i < groups;  i++)
    {
        put_unaligned_uint32(test_packet, htonl(i));
        put_unaligned_uint32(test_packet + sizeof(uint32_t), htonl(group[i].tx_seq_no++));
        if (udp_socket_send(group[i].udp, test_packet, 172, 0) < 0)
            printf("send error - %d\n", errno);
        total_sent++;
    }
}
/*- End of function --------------------------------------------------------*/

static void read_cb(int fd, short int event, void *arg)
{
    char buf[255];
    int len;
    port_group_info_t *group;
    struct sockaddr sa;
    const struct sockaddr_in *us;
    socklen_t salen;
    int action;

    group = (port_group_info_t *) arg;
    /* Reschedule this event */
    event_add(&group->evt, NULL);

    salen = sizeof(sa);
    len = udp_socket_recvfrom(group->udp,
                              buf,
                              sizeof(buf),
                              0,
                              &sa,
                              &salen,
                              &action);
    if (stun_server)
    {
        if (group->rfc3489_state != RFC3489_STATE_RESPONSE_RECEIVED)
        {
            group->rfc3489_state = udp_socket_get_rfc3489_state(group->udp);
            if (group->rfc3489_state == RFC3489_STATE_RESPONSE_RECEIVED)
            {
                us = udp_socket_get_local(group->udp);
                printf("We are %d 0x%X %d\n", us->sin_family, (unsigned int) us->sin_addr.s_addr, ntohs(us->sin_port));
                us = udp_socket_get_apparent_local(group->udp);
                printf("We appear to be %d 0x%X %d\n", us->sin_family, (unsigned int) us->sin_addr.s_addr, ntohs(us->sin_port));
            }
        }
    }
    if (len < 0  &&  errno != EAGAIN  &&  errno != ECONNRESET)
        exit(2);
    if (len >= 0)
    {
        total_received++;
        //printf("%d - %5d\n", i, len);
        if (ntohl(get_unaligned_uint32(buf)) != group->tag)
            printf("%d: Bad channel %" PRIu32 " %d\n", group->tag, ntohl(get_unaligned_uint32(buf)), group->tag);
        if (ntohl(get_unaligned_uint32(buf + sizeof(uint32_t))) != group->rx_seq_no)
        {
            if (group->rx_seq_no == 0)
            {
                group->rx_seq_no = ntohl(get_unaligned_uint32(buf + sizeof(uint32_t)));
            }
            else
            {
                total_seq_misses++;
                //printf("%d: Bad sequence no %" PRIu32 " %" PRIu32 "\n", group->tag, ntohl(get_unaligned_uint32(buf + sizeof(uint32_t))), group->rx_seq_no);
            }
            group->rx_seq_no = ntohl(get_unaligned_uint32(buf + sizeof(uint32_t)));
        }
        group->rx_seq_no++;
    }
}
/*- End of function --------------------------------------------------------*/

int main(int argc, char *argv[])
{
    int i;
    struct sockaddr_in rfc3489_server;
    struct sockaddr_in far;
    const struct sockaddr_in *us;
    struct timeval tv;
    struct hostent *hp;
    const char *far_host;
    int base_port;
    int far_base_port;

    stun_server = NULL;
    far_host = "127.0.0.1";
    base_port = 10000;
    far_base_port = 12000;
    groups = 20;
    for (i = 1;  i < argc;  i++)
    {
        if (strcmp(argv[i], "-b") == 0)
        {
            base_port = atoi(argv[++i]);
            continue;
        }
        if (strcmp(argv[i], "-F") == 0)
        {
            far_base_port = atoi(argv[++i]);
            continue;
        }
        if (strcmp(argv[i], "-f") == 0)
        {
            far_host = argv[++i];
            continue;
        }
        if (strcmp(argv[i], "-g") == 0)
        {
            groups = atoi(argv[++i]);
            continue;
        }
        if (strcmp(argv[i], "-s") == 0)
        {
            stun_server = argv[++i];
            continue;
        }
    }
 
    event_init();

    printf("Resolving far end server\n");
    if ((hp = gethostbyname(far_host)) == NULL) 
    {
        printf("Invalid address for far end server\n");
        exit(2);
    }
    far.sin_family = AF_INET;
    memcpy(&far.sin_addr, hp->h_addr, sizeof(far.sin_addr));
    far.sin_port = htons(far_base_port);
    if (stun_server)
    {
        printf("Resolving RFC3489 (STUN) server\n");
        if ((hp = gethostbyname(stun_server)) == NULL) 
        {
            printf("Invalid address for STUN server\n");
            exit(2);
        }
        rfc3489_server.sin_family = AF_INET;
        memcpy(&rfc3489_server.sin_addr, hp->h_addr, sizeof(rfc3489_server.sin_addr));
        rfc3489_server.sin_port = htons(3478);
        printf("Initing STUN server\n");
        rfc3489_init(&rfc3489_server);
    }

    printf("Initing UDP for %d groups\n", groups);
    evtimer_set(&monitor_evt, monitor_cb, &monitor_evt);
    evtimer_set(&timer_evt, timeout_cb, &timer_evt);
    for (i = 0;  i < groups;  i++)
    {
        memset(&group[i], 0, sizeof(group[i]));
        group[i].udp = udp_socket_group_create_and_bind(2, 0, NULL, base_port, base_port + 10000);
        if (group[i].udp == NULL)
        {
            printf("Failed at %d\n", i);
            exit(2);
        }
#ifdef WIN32
        event_set(&group[i].evt, udp_socket_fd(group[i].udp), EV_READ, read_cb, &group[i]);
#else
        event_set(&group[i].evt, udp_socket_fd(group[i].udp), EV_READ, read_cb, &group[i]);
#endif
        /* Add it to the active events, without a timeout */
        event_add(&group[i].evt, NULL);
    }
    if (stun_server)
    {
        for (i = 0;  i < groups;  i++)
            udp_socket_set_nat(group[i].udp, TRUE);
    }

    for (i = 0;  i < groups;  i++)
    {
        /* The port randomisation on both sides should be similar, assuming none of the
           ports in range were already open. */
        us = udp_socket_get_local(group[i].udp);
        far.sin_port = htons(far_base_port + ntohs(us->sin_port) - base_port);
        //printf("Near %5d, far %5d\n", htons(us->sin_port), htons(far.sin_port));
        udp_socket_set_far(group[i].udp, &far);
        group[i].rfc3489_state = RFC3489_STATE_IDLE;
        group[i].tag = i;
    }
    for (i = 0;  i < 256;  i++)
        test_packet[i] = i;
    timerclear(&tv);
    tv.tv_usec = 20000;
    gettimeofday(&target_time, NULL);
    timeradd(&target_time, &tv, &target_time);
    event_add(&timer_evt, &tv);

    timerclear(&tv);
    tv.tv_sec = 2;
    event_add(&monitor_evt, &tv);

    event_dispatch();

    printf("Done\n");
    getchar();
    return 0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
