/*
 * LibStream - a library for media streaming.
 *
 * udptl_tests.c - A test suite for the UDPTL protocol from T.38.
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2006 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: udptl_tests.c,v 1.4 2007/02/19 15:46:14 steveu Exp $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <math.h>
#include <memory.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <sys/select.h>
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <netdb.h>

#include "libstream.h"

udptl_state_t udptl_state;
udp_state_t *s;

#define STREAM_LEN  10000000

uint8_t buffer[STREAM_LEN];
int msg_lens[128];
int total_len;
int rx_len;
int step;
int bad_messages;
int msg_len;
int next_rx_seq_no;

/* Use a local random generator, so the results are consistent across platforms */
static int my_rand(void)
{
    static int rndnum = 1234567;

    return (rndnum = 1664525U*rndnum + 1013904223U);
}
/*- End of function --------------------------------------------------------*/

static int process_rx_lossy_packet(void *user_data, int seq_no, const uint8_t *buf, int len)
{
    int i;

    if (seq_no != (next_rx_seq_no & 0xFFFF))
    {
        printf("Test failed - seq_no=%d, next_seq_no=%d\n", seq_no, next_rx_seq_no);
        exit(2);
    }
    //printf("Seq_no %d, len %d, rx_len %d\n", seq_no, len, rx_len);
    /* If the packet was reconstructed from parity information, it may be padded at the end with some zeros. */
    /* Test the received packet is long enough to contain what was sent... */
    if (len < msg_lens[next_rx_seq_no & 0x7F])
    {
        printf("Test failed - seq_no=%d, len=%d, msg_len=%d, rx_len=%d\n", seq_no, len, msg_lens[next_rx_seq_no & 0x7F], rx_len);
        exit(2);
    }
    /* ...that the expected amount of the received packet matches what was sent... */
    if (memcmp(buffer + rx_len, buf, msg_lens[next_rx_seq_no & 0x7F]) != 0)
    {
        for (i = 0;  i < msg_lens[next_rx_seq_no & 0x7F];  i++)
        {
            if (buf[i] != buffer[rx_len + i])
            {
                printf("Test failed - seq_no=%d, len=%d, msg_len=%d, rx_len=%d, at octet %d - %02x %02x\n", seq_no, len, msg_lens[next_rx_seq_no & 0x7F], rx_len, i, buf[i], buffer[rx_len + i]);
                exit(2);
            }
        }
    }
    /* ...and any extra padding is all zeros... */
    if (len > msg_lens[next_rx_seq_no & 0x7F])
    {
        for (i = msg_lens[next_rx_seq_no & 0x7F];  i < len;  i++)
        {
            if (buf[i])
            {
                printf("Test failed - seq_no=%d, padding not all zeros\n", seq_no);
                exit(2);
            }
        }
    }
    rx_len += msg_lens[next_rx_seq_no & 0x7F];
    next_rx_seq_no++;
    return 0;
}
/*- End of function --------------------------------------------------------*/

static void lossy_test(int ec_scheme, int ec_span, int ec_entries, int loss_rate)
{
    int i;
    int len;
    uint8_t buf[4096];
    int pos;
    int sent_pkts;
    int dropped_pkts;

    /* Send packets between two instances, with a low rate of random packet loss. See that the received packet stream has no gaps. */
    
    printf("Back to back lossy testing EC mode %d, span %d, entries %d, loss rate 1/%d\n", ec_scheme, ec_span, ec_entries, loss_rate);

    /* TODO: Even with a low rate of loss, it is possible for gaps to occur, so the outcome of this test is not deterministic,
             and really ought to be made so. */
    if ((udptl_init(&udptl_state, ec_scheme, ec_span, ec_entries, process_rx_lossy_packet, NULL)) == NULL)
    {
        printf("Test failed\n");
        exit(2);
    }

    /* Generate some test data */
    for (i = 0;  i < STREAM_LEN;  i++)
        buffer[i] = my_rand() & 0xFF;

    /* Now pass it, in random sized pieces, through the protocol software */
    rx_len = 0;
    next_rx_seq_no = 0;
    sent_pkts = 0;
    dropped_pkts = 0;
    for (i = 0, pos = 0;  pos < STREAM_LEN - 256;  i++, pos += msg_len)
    {
        msg_len = (my_rand() & 0xFF) + 1;
        len = udptl_build_packet(&udptl_state, buf, buffer + pos, msg_len);
        /* Drop some packets. There is no protection here against dropping a burst
           long enough to cause the test to fail. */
        msg_lens[i & 0x7F] = msg_len;
        if (my_rand()%loss_rate)
        {
            //printf("Sending %d - %d\n", i, msg_len);
            if (udptl_rx_packet(&udptl_state, buf, len))
            {
                /* The packet was rejected */
                printf("Test failed at udptl_rx_packet\n");
                exit(2);
            }
            sent_pkts++;
        }
        else
        {
            //printf("Dropping %d - %d\n", i, msg_len);
            if (ec_scheme == UDPTL_ERROR_CORRECTION_NONE)
            {
                rx_len += msg_len;
                next_rx_seq_no++;
            }
            dropped_pkts++;
        }
    }
    printf("%d packets - %d sent, %d dropped\n", i, sent_pkts, dropped_pkts);
    if (pos != rx_len)
    {
        printf("Test failed\n");
        exit(2);
    }
}
/*- End of function --------------------------------------------------------*/

static int process_rx_junk_packet(void *user_data, int seq_no, const uint8_t *buf, int len)
{
    printf("Got junk packet seq_no=%d, len=%d\n", seq_no, len);
    return 0;
}
/*- End of function --------------------------------------------------------*/

static void junk_test(int ec_scheme, int ec_span, int ec_entries)
{
    int i;
    int len;
    int pos;

    /* Send packets between two instances, with a low rate of random packet loss. See that the received packet stream has no gaps. */
    
    printf("Junk testing EC mode %d, span %d, entries %d\n", ec_scheme, ec_span, ec_entries);

    /* TODO: Even with a low rate of loss, it is possible for gaps to occur, so the outcome of this test is not deterministic,
             and really ought to be made so. */
    if ((udptl_init(&udptl_state, ec_scheme, ec_span, ec_entries, process_rx_junk_packet, NULL)) == NULL)
    {
        printf("Test failed\n");
        exit(2);
    }

    /* Generate some test data */
    for (i = 0;  i < STREAM_LEN;  i++)
        buffer[i] = my_rand() & 0xFF;

    /* Now pass it, in random sized pieces, through the protocol decoder software */
    /* While a random packet could mimic a good one, and be accepted, we don't expect any of our
       random packets to be accepted. */
    next_rx_seq_no = 0;
    for (i = 0, pos = 0;  pos < STREAM_LEN - 256;  i++, pos += len)
    {
        len = (my_rand() & 0x1FF) + 1;
        if (udptl_rx_packet(&udptl_state, buffer + pos, len) == 0)
        {
            /* The packet was accepted */
            printf("Test failed at udptl_rx_packet\n");
            exit(2);
        }
    }
    printf("%d packets\n", i);
}
/*- End of function --------------------------------------------------------*/

int main(int argc, char *argv[])
{
    int i;
    int ec_scheme;
    int ec_span;
    int ec_entries;
    int loss_rate;
    
    ec_scheme = UDPTL_ERROR_CORRECTION_NONE;
    ec_span = 3;
    ec_entries = 3;
    loss_rate = 25;
    for (i = 1;  i < argc;  i++)
    {
        if (strcmp(argv[i], "-e") == 0)
        {
            ec_entries = atoi(argv[++i]);
            continue;
        }
        if (strcmp(argv[i], "-l") == 0)
        {
            loss_rate = atoi(argv[++i]);
            continue;
        }
        if (strcmp(argv[i], "-r") == 0)
        {
            ec_scheme = atoi(argv[++i]);
            continue;
        }
        if (strcmp(argv[i], "-s") == 0)
        {
            ec_span = atoi(argv[++i]);
            continue;
        }
    }

    lossy_test(UDPTL_ERROR_CORRECTION_NONE, 0, 0, loss_rate);
    lossy_test(UDPTL_ERROR_CORRECTION_REDUNDANCY, 0, 3, loss_rate);
    lossy_test(UDPTL_ERROR_CORRECTION_FEC, 3, 3, loss_rate);

    junk_test(UDPTL_ERROR_CORRECTION_NONE, 0, 0);
    junk_test(UDPTL_ERROR_CORRECTION_REDUNDANCY, 0, 3);
    junk_test(UDPTL_ERROR_CORRECTION_FEC, 3, 3);

    /* TODO: Throw some RTP and RTCP at the decoder, to see it does not misbehave */
    printf("Tests passed\n");
    return  0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
