/*
 * LibVale - a library for media streaming.
 *
 * queue.c - Packet queuing and buffer management.
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2007 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: queue.c,v 1.3 2007/04/07 03:37:25 steveu Exp $
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <math.h>
#include <memory.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <sys/select.h>
#include <sys/time.h>
#include <netinet/in.h>

#include "vale/udp.h"
#include "vale/rfc3550.h"
#include "vale/queue.h"

frame_queue_frame_t *frame_queue_get_free_frame(frame_queue_state_t *s)
{
    frame_queue_frame_t *frame;

    /* Acquire a frame */
    if ((frame = s->free_frames))
    {
        s->free_frames = frame->later;
    }
    else
    {
        if ((frame = (frame_queue_frame_t *) malloc(sizeof(frame_queue_frame_t) + s->data_size)) == NULL)
            return NULL;
    }
    frame->earlier = NULL;
    frame->later = NULL;
    frame->payload_type = 0;
    frame->sender_stamp = 0;
    frame->sender_len = 0;
    frame->receiver_stamp = 0;
    frame->len = 0;

    return frame;
}
/*- End of function --------------------------------------------------------*/

void frame_queue_put_free_frame(frame_queue_state_t *s, frame_queue_frame_t *frame)
{
    /* Put it on the free list */
    frame->later = s->free_frames;
    s->free_frames = frame;
}
/*- End of function --------------------------------------------------------*/

int frame_queue_free_frames(frame_queue_state_t *s)
{
    frame_queue_frame_t *p;
    int i;
    
    p = s->free_frames;
    i = 0;
    while (p)
    {
        p = p->later;
        i++;
    }
    return i;
}
/*- End of function --------------------------------------------------------*/

int frame_queue_queued_frames(frame_queue_state_t *s)
{
    frame_queue_frame_t *p;
    int i;
    
    p = s->first_frame;
    i = 0;
    while (p)
    {
        p = p->later;
        i++;
    }
    return i;
}
/*- End of function --------------------------------------------------------*/

int frame_queue_put_frame(frame_queue_state_t *s, frame_queue_frame_t *frame)
{
    frame_queue_frame_t *p;

    /* When a frame arrives we just queue it in order. We leave all the tricky stuff until frames
       are read from the queue. */
    s->stats.frames_in++;

    /* Frames are kept in a list, sorted by the timestamp assigned by the sender. */
    if (s->last_frame == NULL)
    {
        /* The queue is empty. */
        frame->later = NULL;
        frame->earlier = NULL;
        s->first_frame = frame;
        s->last_frame = frame;
    }
    else if (frame->sender_stamp > s->last_frame->sender_stamp)
    {
        /* Frame goes at the end of the queue. */
        frame->later = NULL;
        frame->earlier = s->last_frame;
        s->last_frame->later = frame;
        s->last_frame = frame;
    }
    else
    {
        /* Frame is out of sequence. */
        s->stats.frames_oos++;

        /* Find where it should go in the queue */
        p = s->last_frame;
        while (frame->sender_stamp < p->sender_stamp  &&  p->earlier)
            p = p->earlier;

        if (p->earlier)
        {
printf("Check duplicate frame - %d %u\n", frame->seq_no, frame->sender_stamp);
            /* It needs to go somewhere in the queue */
            if (frame->sender_stamp == p->sender_stamp)
            {
                if (frame->seq_no == p->seq_no)
                {
printf("Duplicate frame - %d\n", frame->seq_no);
                    /* Duplicate frame - drop it. */
                    s->stats.frames_repeats++;
                    return -1;
                }
                if (frame->seq_no > s->last_frame->seq_no)
                {
                    /* Frame goes at the end of the queue. */
                    frame->later = NULL;
                    frame->earlier = s->last_frame;
                    s->last_frame->later = frame;
                    s->last_frame = frame;
                }
                else
                {
                    /* Frame is out of sequence. */
                    s->stats.frames_oos++;
        
                    /* Find where it should go in the queue */
                    p = s->last_frame;
                    while (frame->seq_no < p->seq_no  &&  p->earlier)
                        p = p->earlier;
                    if (p->earlier)
                    {
                        /* It needs to go somewhere in the queue */
                        frame->later = p->later;
                        frame->earlier = p;
                        p->later->earlier = frame;
                        p->later = frame;
                    }
                    else
                    {
                        /* It needs to go at the very beginning of the queue */
                        frame->later = p;
                        frame->earlier = NULL;
                        p->earlier = frame;
                        s->first_frame = frame;
                    }
                }
            }
            else
            {
                frame->later = p->later;
                frame->earlier = p;
                p->later->earlier = frame;
                p->later = frame;
            }
        }
        else
        {
            /* It needs to go at the very beginning of the queue */
            frame->later = p;
            frame->earlier = NULL;
            p->earlier = frame;
            s->first_frame = frame;
        }
    }

    return 0;
}
/*- End of function --------------------------------------------------------*/

int frame_queue_put_rx_packet(void *user_data,
                              const uint8_t buf[],
                              int len,
                              uint16_t seq_no,
                              uint32_t timestamp,
                              uint32_t ssrc,
                              int payload_type,
                              int mark)
{
    frame_queue_state_t *s;
    frame_queue_frame_t *frame;

    s = (frame_queue_state_t *) user_data;

    if (len > s->data_size)
        return -1;
    if ((frame = frame_queue_get_free_frame(s)) == NULL)
        return -1;
    /* Fill out the frame */
    frame->payload_type = payload_type;
    frame->len = len;
    frame->seq_no = seq_no;
    frame->sender_stamp = timestamp;
    //frame->sender_len = sender_len;
    //frame->receiver_stamp = receiver_stamp;
    memcpy(frame->data, buf, len);
    if (frame_queue_put_frame(s, frame) < 0)
        frame_queue_put_free_frame(s, frame);
    return 0;
}
/*- End of function --------------------------------------------------------*/

frame_queue_frame_t *frame_queue_get_frame(frame_queue_state_t *s, rfc3550_timestamp_t sender_stamp)
{
    frame_queue_frame_t *frame;

    if ((frame = s->first_frame) == NULL)
        return NULL;

    if (sender_stamp <= frame->sender_stamp)
        return NULL;
   
    /* Remove this frame from the queue */
    if (frame->later)
    {
        frame->later->earlier = NULL;
        s->first_frame = frame->later;
    }
    else
    {
        /* The queue is now empty */
        s->first_frame = NULL;
        s->last_frame = NULL;
    }
    s->stats.frames_out++;
    return frame;
}
/*- End of function --------------------------------------------------------*/

void frame_queue_fill_frame(frame_queue_frame_t *frame,
                            uint8_t data[],
                            int len,
                            int payload_type,
                            int seq_no,
                            rfc3550_timestamp_t sender_len,
                            rfc3550_timestamp_t sender_stamp,
                            rfc3550_timestamp_t receiver_stamp)
{
    /* Fill out the frame */
    frame->payload_type = payload_type;
    frame->len = len;
    frame->seq_no = seq_no;
    frame->sender_stamp = sender_stamp;
    frame->sender_len = sender_len;
    frame->receiver_stamp = receiver_stamp;
    memcpy(frame->data, data, len);
}
/*- End of function --------------------------------------------------------*/

void frame_queue_restart(frame_queue_state_t *s, int data_size)
{
    frame_queue_frame_t *frame;
    frame_queue_frame_t *next;

    /* Free all the frames on the free list */
    for (frame = s->free_frames;  frame;  frame = next)
    {
        next = frame->later;
        free(frame);
    }

    memset(s, 0, sizeof(*s));
    s->data_size = data_size;
}
/*- End of function --------------------------------------------------------*/

frame_queue_state_t *frame_queue_new(int data_size)
{
    frame_queue_state_t *s;

    if ((s = (frame_queue_state_t *) malloc(sizeof(frame_queue_state_t))) == NULL)
        return NULL;
    memset(s, 0, sizeof(*s));
    frame_queue_restart(s, data_size);
    return s;
}
/*- End of function --------------------------------------------------------*/

void frame_queue_free(frame_queue_state_t *s)
{
    frame_queue_frame_t *frame;
    frame_queue_frame_t *next;
    
    /* Free all the frames in the queue. In most cases these should have been
       removed already, so their associated data could be freed. */
    for (frame = s->first_frame;  frame;  frame = next)
    {
        next = frame->later;
        free(frame);
    }
    /* Free all the frames on the free list */
    for (frame = s->free_frames;  frame;  frame = next)
    {
        next = frame->later;
        free(frame);
    }
    /* Finally, free ourselves! */ 
    free(s);
}
/*- End of function --------------------------------------------------------*/

void frame_queue_dump(frame_queue_state_t *s)
{
    frame_queue_frame_t *p;

    p = s->last_frame;
    while (p)
    {
        printf("Seq no %d, timestamp %u\n", p->seq_no, p->sender_stamp);
        p = p->earlier;
    }
}
/*- End of function --------------------------------------------------------*/

void frame_queue_stats(frame_queue_state_t *s, frame_queue_stats_t *stats)
{
    memcpy(stats, &s->stats, sizeof(*stats));
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
