/*
 * Vale - a library for media streaming.
 *
 * rfc2198.c - RTP redundancy
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2006 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: rfc2198.c,v 1.2 2007/04/06 11:14:56 steveu Exp $
 */

/*! \file */

#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <math.h>
#include <memory.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <sys/select.h>
#include <sys/time.h>
#include <netinet/in.h>

#include "vale/udp.h"
#include "vale/rfc3550.h"
#include "vale/rfc2198.h"

#define FALSE 0
#define TRUE (!FALSE)


/*
    An example of an RFC2198 type RTP packet, taken from RFC2198

    0                   1                    2                   3
    0 1 2 3 4 5 6 7 8 9 0 1 2 3  4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |V=2|P|X| CC=0  |M|      PT     |   sequence number of primary  |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |              timestamp  of primary encoding                   |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |           synchronization source (SSRC) identifier            |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |1| block PT=7  |  timestamp offset         |   block length    |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |0| block PT=5  |                                               |
   +-+-+-+-+-+-+-+-+                                               +
   |                                                               |
   +                LPC encoded redundant data (PT=7)              +
   |                (14 bytes)                                     |
   +                                               +---------------+
   |                                               |               |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+               +
   |                                                               |
   +                                                               +
   |                                                               |
   +                                                               +
   |                                                               |
   +                                                               +
   |                DVI4 encoded primary data (PT=5)               |
   +                (84 bytes, not to scale)                       +
   /                                                               /
   +                                                               +
   |                                                               |
   +                                                               +
   |                                                               |
   +                                               +---------------+
   |                                               |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
*/

int rfc2198_process_rx_packet(void *user_data,
                              const uint8_t buf[],
                              int len,
                              uint16_t seq_no,
                              uint32_t timestamp,
                              uint32_t ssrc,
                              int payload_type,
                              int mark)
{
    int payload_types[24];
    uint32_t timestamp_offsets[24];
    int lengths[24];
    int positions[24];
    int n;
    int pos;
    int ptr;
    int x;
    int i;
    int length;
    uint32_t header;
    rfc2198_state_t *s;
    rtp_rx_packet_handler_t *rx_packet_handler;
    void *user_datax;

    s = (rfc2198_state_t *) user_data;
    ptr = 0;
    n = 0;
    pos = 0;
    x = 0;
    while ((n < len  &&  (buf[n] & 0x80)))
    {
        header = ntohl(((uint32_t *) (buf + n))[0]);
        length = header & 0x3FF;
        if (x < 23)
        {
            payload_types[x] = (header >> 24) & 0x7F;
            timestamp_offsets[x] = (header >> 10) & 0x3FFF;
            positions[x] = pos;
            lengths[x] = length;
            x++;
        }
        pos += length;
        n += 4;
    }
    if (n >= len)
        return -1;
    payload_types[x] = buf[n++];
    timestamp_offsets[x] = 0;
    positions[x] = pos;
    lengths[x] = len - pos - n;
    /* Update all the positions, to allow for the header length, now that is known. */
    for (i = 0;  i <= x;  i++)
        positions[i] += n;

    if (seq_no > s->rx_seq_no)
    {
        /* Something is missing. See if we can fill it in from the
           redundant material. */
        if (seq_no - x < s->rx_seq_no)
            i = s->rx_seq_no - (seq_no - x);
        else
            i = 0;
        for (  ;  i < x;  i++)
        {
            if (rfc3550_get_payload_handler(s->rfc3550_state,
                                            payload_types[i],
                                            &rx_packet_handler,
                                            &user_datax) == 0)
            {
                rx_packet_handler(user_datax,
                                  &buf[positions[i]],
                                  lengths[i],
                                  (seq_no - x + i) & 0xFFFF,
                                  timestamp - timestamp_offsets[i],
                                  ssrc,
                                  payload_types[i],
                                  mark);
            }
        }
    }
    if (rfc3550_get_payload_handler(s->rfc3550_state,
                                    payload_types[x],
                                    &rx_packet_handler,
                                    &user_datax) == 0)
    {
        rx_packet_handler(user_datax,
                          &buf[positions[x]],
                          lengths[x],
                          seq_no,
                          timestamp,
                          ssrc,
                          payload_types[x],
                          mark);
    }
    s->rx_seq_no = (seq_no + 1) & 0xFFFF;
    return 0;
}
/*- End of function --------------------------------------------------------*/

int rfc2198_build_packet(rfc2198_state_t *s,
                         const uint8_t msg[],
                         int msg_len,
                         uint8_t buf[],
                         uint32_t timestamp,
                         int payload_type,
                         int mark)
{
    int i;
    int seq;
    int entry;
    int entryx;
    int entries;
    int n;
    int m;
    uint32_t timestamp_offset;

    seq = s->tx_seq_no & 0xFFFF;

    /* Map the sequence number to an entry in the circular buffer */
    entry = seq & RFC2198_BUF_MASK;

    /* We save the message in a circular buffer, for generating
       redundancy sets later on. */
    s->tx[entry].payload_type = payload_type;
    s->tx[entry].timestamp = timestamp;
    s->tx[entry].buf_len = msg_len;
    memcpy(s->tx[entry].buf, msg, msg_len);

    if (s->tx_seq_no < s->redundant_entries)
        entries = s->tx_seq_no;
    else
        entries = s->redundant_entries;
    entryx = (seq - entries) & RFC2198_BUF_MASK;
    n = 0;
    m = sizeof(uint32_t)*entries + 1;
    for (i = 0;  i < entries;  i++)
    {
        timestamp_offset = timestamp - s->tx[entryx].timestamp;
        buf[n] = 0x80 + s->tx[entryx].payload_type;
        buf[n + 1] = (timestamp_offset >> 6) & 0xFF;
        buf[n + 2] = ((s->tx[entryx].buf_len >> 8) & 0x03) | ((timestamp_offset & 0x3F) << 2);
        buf[n + 3] = s->tx[entryx].buf_len & 0xFF;
        n += 4;
        entryx = (entryx + 1) & RFC2198_BUF_MASK;
    }
    buf[n++] = payload_type;
    entryx = (seq - entries) & RFC2198_BUF_MASK;
    for (i = 0;  i <= entries;  i++)
    {
        memcpy(buf + m, s->tx[entryx].buf, s->tx[entryx].buf_len);
        m += s->tx[entryx].buf_len;
        entryx = (entryx + 1) & RFC2198_BUF_MASK;
    }

    s->tx_seq_no++;
    return m;
}
/*- End of function --------------------------------------------------------*/

rfc2198_state_t *rfc2198_init(rfc2198_state_t *s,
                              rfc3550_state_t *t,
                              int redundant_entries,
                              int payload_type)
{
    if (t == NULL)
        return NULL;
    if (redundant_entries < 0  ||  redundant_entries > 15)
        return NULL;

    if (s == NULL)
    {
        if ((s = (rfc2198_state_t *) malloc(sizeof(*s))) == NULL)
            return NULL;
    }
    memset(s, 0, sizeof(*s));    
    
    s->redundant_entries = redundant_entries;
    
    s->far_max_datagram_size = LOCAL_MAX_RFC2198_DATAGRAM;
    s->local_max_datagram_size = LOCAL_MAX_RFC2198_DATAGRAM;

    s->rfc3550_state = t;
    s->payload_type = payload_type;
    if (payload_type >= 0)
        rfc3550_set_payload_handler(t, payload_type, rfc2198_process_rx_packet, (void *) s);

    return s;
}
/*- End of function --------------------------------------------------------*/

int rfc2198_release(rfc2198_state_t *s)
{
    if (s->payload_type >= 0)
        rfc3550_set_payload_handler(s->rfc3550_state, s->payload_type, NULL, NULL);
    free(s);
    return 0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
