/*
 * Vale - a library for media streaming.
 *
 * rfc3550.c - RTP core
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2006 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: rfc3550.c,v 1.2 2007/04/07 03:37:25 steveu Exp $
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <math.h>
#include <memory.h>
#include <errno.h>
#include <time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <sys/select.h>
#include <sys/time.h>
#if defined(ENABLE_SRTP)
#include "srtp.h"
#endif

#include "vale/unaligned.h"
#include "vale/udp.h"
#include "vale/rfc3550.h"

#define FALSE 0
#define TRUE (!FALSE)

#define MAX_TIMESTAMP_SKEW          640
#define RTP_MTU                     1200

#define RTP_SEQ_MOD                 (1<<16)

#define FLAG_NAT_ACTIVE             (3 << 1)
#define FLAG_NAT_INACTIVE           (0 << 1)
#define FLAG_NAT_INACTIVE_NOWARN    (1 << 1)

/*
        RTCP packet formats
        -------------------
        
        
         0                   1                   2                   3
         0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
header  |V=2|P|    RC   |   PT=SR=200   |             length            |
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |                         SSRC of sender                        |
        +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+
sender  |              NTP timestamp, most significant word             |
info    +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |             NTP timestamp, least significant word             |
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |                         RTP timestamp                         |
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |                     sender's packet count                     |
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |                      sender's octet count                     |
        +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+
report  |                 SSRC_1 (SSRC of first source)                 |
block   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
  1     | fraction lost |       cumulative number of packets lost       |
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |           extended highest sequence number received           |
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |                      interarrival jitter                      |
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |                         last SR (LSR)                         |
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |                   delay since last SR (DLSR)                  |
        +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+
report  |                 SSRC_2 (SSRC of second source)                |
block   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
  2     :                               ...                             :
        +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+
        |                  profile-specific extensions                  |
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
       
        Sender report RTCP packet
       
       
         0                   1                   2                   3
         0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
header  |V=2|P|    RC   |   PT=RR=201   |             length            |
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |                     SSRC of packet sender                     |
        +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+
report  |                 SSRC_1 (SSRC of first source)                 |
block   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
  1     | fraction lost |       cumulative number of packets lost       |
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |           extended highest sequence number received           |
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |                      interarrival jitter                      |
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |                         last SR (LSR)                         |
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |                   delay since last SR (DLSR)                  |
        +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+
report  |                 SSRC_2 (SSRC of second source)                |
block   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
  2     :                               ...                             :
        +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+
        |                  profile-specific extensions                  |
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

        Receiver report RTCP packet



         0                   1                   2                   3
         0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
header  |V=2|P|    SC   |  PT=SDES=202  |             length            |
        +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+
chunk   |                          SSRC/CSRC_1                          |
  1     +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |                           SDES items                          |
        |                              ...                              |
        +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+
chunk   |                          SSRC/CSRC_2                          |
  2     +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |                           SDES items                          |
        |                              ...                              |
        +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+

        SDES: Source Description RTCP Packet



         0                   1                   2                   3
         0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |    CNAME=1    |     length    | user and domain name        ...
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

        CNAME: Canonical End-Point Identifier SDES Item


         0                   1                   2                   3
         0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |     NAME=2    |     length    | common name of source       ...
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

        NAME: User Name SDES Item


         0                   1                   2                   3
         0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |    EMAIL=3    |     length    | email address of source     ...
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

        EMAIL: Electronic Mail Address SDES Item



         0                   1                   2                   3
         0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |    PHONE=4    |     length    | phone number of source      ...
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

        PHONE: Phone Number SDES Item



         0                   1                   2                   3
         0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |     LOC=5     |     length    | geographic location of site ...
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

        LOC: Geographic User Location SDES Item



         0                   1                   2                   3
         0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |     TOOL=6    |     length    |name/version of source appl. ...
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

        TOOL: Application or Tool Name SDES Item


         0                   1                   2                   3
         0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |     NOTE=7    |     length    | note about the source       ...
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

        NOTE: Notice/Status SDES Item


         0                   1                   2                   3
         0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |     PRIV=8    |     length    | prefix length |prefix string...
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        ...             |                  value string               ...
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

        PRIV: Private Extensions SDES Item


         0                   1                   2                   3
         0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |V=2|P|    SC   |   PT=BYE=203  |             length            |
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |                           SSRC/CSRC                           |
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        :                              ...                              :
        +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+
(opt)   |     length    |               reason for leaving            ...
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

        BYE: Goodbye RTCP Packet


         0                   1                   2                   3
         0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |V=2|P| subtype |   PT=APP=204  |             length            |
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |                           SSRC/CSRC                           |
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |                          name (ASCII)                         |
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
        |                   application-dependent data                ...
        +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

        APP: Application-Defined RTCP Packet
*/

#ifdef ENABLE_SRTP
struct rfc3550_policy_s
{
    srtp_policy_t sp;
};
#endif

#if 0
{
    if (SSRC or CSRC identifier is not found in the source identifier table)
    {
        create a new entry storing the data or control source
              transport address, the SSRC or CSRC and other state;
    }
    else if (table entry was created on receipt of a control packet
               and this is the first data packet or vice versa)
    {
        store the source transport address from this packet;
    }
    else if (source transport address from the packet does not match
             the one saved in the table entry for this identifier)
    {
        /* An identifier collision or a loop is indicated */
        if (source identifier is not the participant's own)
        {
            /* OPTIONAL error counter step */
            if (source identifier is from an RTCP SDES chunk
                containing a CNAME item that differs from the CNAME
                in the table entry)
            {
                count a third-party collision;
            }
            else
            {
                count a third-party loop;
            }
            abort processing of data packet or control element;
            /* MAY choose a different policy to keep new source */
        }
        else if (source transport address is found in the list of
                  conflicting data or control source transport
                  addresses)
        {
            /* OPTIONAL error counter step */
            if (source identifier is not from an RTCP SDES chunk containing a CNAME item
                ||
                CNAME is the participant's own)
            {
                count occurrence of own traffic looped;
            }
            mark current time in conflicting address list entry;
            abort processing of data packet or control element;
        }
        else
        {
            /* New collision, change SSRC identifier */
            printf("SSRC collision. Changing SSRC\n");
            create a new entry in the conflicting data or control
                  source transport address list and mark current time;
            send an RTCP BYE packet with the old SSRC identifier;
            choose a new SSRC identifier;
            create a new entry in the source identifier table with
                  the old SSRC plus the source transport address from
                  the data or control packet being processed;
        }
    }
}
/*- End of function --------------------------------------------------------*/

void OnExpire(rfc3550_state_t *s,
               event e,
              int members,
              int senders,
              double rtcp_bw,
              int we_sent,
              int *initial,
              time_tp tc,
              time_tp *tp,
              int *pmembers)
{
    /* This function is responsible for deciding whether to send an
     * RTCP report or BYE packet now, or to reschedule transmission.
     * It is also responsible for updating the pmembers, initial, tp,
     * and avg_rtcp_size state variables.  This function should be
     * called upon expiration of the event timer used by Schedule().
     */
    double t;     /* Interval */
    double tn;    /* Next transmit time */

    /* In the case of a BYE, we use "timer reconsideration" to
     * reschedule the transmission of the BYE if necessary */

    if (TypeOfEvent(e) == EVENT_BYE)
    {
        t = rtcp_interval(s,
                          members,
                          senders,
                          rtcp_bw,
                          we_sent,
                          *initial);
        tn = *tp + t;
        if (tn <= tc)
        {
            SendBYEPacket(e);
            exit(1);
        }
        else
        {
            Schedule(tn, e);
        }
    }
    else if (TypeOfEvent(e) == EVENT_REPORT)
    {
        t = rtcp_interval(members,
                          senders,
                          rtcp_bw,
                          we_sent,
                          s->avg_rtcp_size,
                          *initial);
        tn = *tp + t;
        if (tn <= tc)
        {
            SendRTCPReport(e);
            s->avg_rtcp_size += (SentPacketSize(e) - (s->avg_rtcp_size >> 4));
            *tp = tc;

            /* We must redraw the interval.  Don't reuse the
               one computed above, since its not actually
               distributed the same, as we are conditioned
               on it being small enough to cause a packet to
               be sent */
            t = rtcp_interval(members,
                              senders,
                              rtcp_bw,
                              we_sent,
                              avg_rtcp_size,
                              *initial);
            Schedule(t + tc, e);
            *initial = 0;
        }
        else
        {
            Schedule(tn, e);
        }
        *pmembers = members;
    }
}
/*- End of function --------------------------------------------------------*/

void OnReceive(rfc3550_state_t *s,
               packet p,
               event e,
               int *pmembers,
               double *tp,
               double tc,
               double tn)
{
    /* What we do depends on whether we have left the group, and are
     * waiting to send a BYE (TypeOfEvent(e) == EVENT_BYE) or an RTCP
     * report.  p represents the packet that was just received.  */

    if (PacketType(p) == PACKET_RTCP_REPORT)
    {
        if (NewMember(p)  &&  (TypeOfEvent(e) == EVENT_REPORT))
        {
            AddMember(p);
            s->members++;
        }
        s->avg_rtcp_size += (ReceivedPacketSize(p) - (s->avg_rtcp_size >> 4));
    }
    else if (PacketType(p) == PACKET_RTP)
    {
        if (NewMember(p)  &&  (TypeOfEvent(e) == EVENT_REPORT))
        {
            AddMember(p);
            s->members++;
        }
        if (NewSender(p)  &&  (TypeOfEvent(e) == EVENT_REPORT))
        {
            AddSender(p);
            s->senders++;
        }
    }
    else if (PacketType(p) == PACKET_BYE)
    {
        s->avg_rtcp_size += (ReceivedPacketSize(p) - (s->avg_rtcp_size >> 4));
        if (TypeOfEvent(e) == EVENT_REPORT)
        {
            if (NewSender(p) == FALSE)
            {
                RemoveSender(p);
                s->senders--;
            }
            if (NewMember(p) == FALSE)
            {
                RemoveMember(p);
                s->members--;
            }
            if (s->members < *pmembers)
            {
                tn = tc + (((double) s->members)/(*pmembers))*(tn - tc);
                *tp = tc - (((double) s->members)/(*pmembers))*(tc - *tp);

                /* Reschedule the next report for time tn */
                Reschedule(tn, e);
                *pmembers = s->members;
            }
        }
        else if (TypeOfEvent(e) == EVENT_BYE)
        {
            s->members++;
        }
    }
}
/*- End of function --------------------------------------------------------*/
#endif

double rtcp_interval(rfc3550_state_t *s,
                     int members,
                     int senders,
                     double rtcp_bw,
                     int we_sent,
                     int initial)
{
    /*
     * Minimum average time between RTCP packets from this site (in
     * seconds).  This time prevents the reports from `clumping' when
     * sessions are small and the law of large numbers isn't helping
     * to smooth out the traffic.  It also keeps the report interval
     * from becoming ridiculously small during transient outages like
     * a network partition.
     */
    double const RTCP_MIN_TIME = 5.0;
    /*
     * Fraction of the RTCP bandwidth to be shared among active
     * senders.  (This fraction was chosen so that in a typical
     * session with one or two active senders, the computed report
     * time would be roughly equal to the minimum report time so that
     * we don't unnecessarily slow down receiver reports.)  The
     * receiver fraction must be 1 - the sender fraction.
     */
    double const RTCP_SENDER_BW_FRACTION = 0.25;
    double const RTCP_RCVR_BW_FRACTION = 1 - RTCP_SENDER_BW_FRACTION;
    /* To compensate for "timer reconsideration" converging to a
     * value below the intended average.
     */
    double const COMPENSATION = 2.71828 - 1.5;

    double t;                   /* interval */
    double rtcp_min_time;
    int n;                      /* no. of members for computation */

    /*
     * Very first call at application start-up uses half the min
     * delay for quicker notification while still allowing some time
     * before reporting for randomization and to learn about other
     * sources so the report interval will converge to the correct
     * interval more quickly.
     */
    rtcp_min_time = RTCP_MIN_TIME;
    if (initial)
        rtcp_min_time /= 2;
    /*
     * Dedicate a fraction of the RTCP bandwidth to senders unless
     * the number of senders is large enough that their share is
     * more than that fraction.
     */
    n = members;
    if (senders <= members*RTCP_SENDER_BW_FRACTION)
    {
        if (we_sent)
        {
            rtcp_bw *= RTCP_SENDER_BW_FRACTION;
            n = senders;
        }
        else
        {
            rtcp_bw *= RTCP_RCVR_BW_FRACTION;
            n -= senders;
        }
    }

    /*
     * The effective number of sites times the average packet size is
     * the total number of octets sent when each site sends a report.
     * Dividing this by the effective bandwidth gives the time
     * interval over which those packets must be sent in order to
     * meet the bandwidth target, with a minimum enforced.  In that
     * time interval we send one report so this time is also our
     * average time between reports.
     */
    t = (s->avg_rtcp_size >> 4)*n/rtcp_bw;
    if (t < rtcp_min_time)
        t = rtcp_min_time;

    /*
     * To avoid traffic bursts from unintended synchronization with
     * other sites, we then pick our actual next report interval as a
     * random number uniformly distributed between 0.5*t and 1.5*t.
     */
    t *= (drand48() + 0.5);
    t /= COMPENSATION;
    return t;
}
/*- End of function --------------------------------------------------------*/

static __inline__ int realign_padding32(int len)
{
    return (sizeof(uint32_t) - (len & (sizeof(uint32_t) - 1))) & (sizeof(uint32_t) - 1);
}
/*- End of function --------------------------------------------------------*/

/* Number of elapsed seconds from 1900 to 1970 */
const uint32_t NTP_EPOCH_OFFSET = 2208992400UL;

uint64_t ntptime(void)
{
    struct timeval now;
    uint64_t when;

    gettimeofday(&now, NULL);
    when = (uint64_t) now.tv_usec*281474977LLU;
    when >>= 16;
    return ((uint64_t) (now.tv_sec + NTP_EPOCH_OFFSET) << 32) | when;
}
/*- End of function --------------------------------------------------------*/

struct timeval ntp_to_timeval(uint64_t when)
{
    struct timeval t;

    t.tv_sec = (when >> 32) - NTP_EPOCH_OFFSET;
    when &= 0xFFFFFFFF;
    when *= 1000000;
    t.tv_usec = when >> 32;
    return t;
}
/*- End of function --------------------------------------------------------*/

uint64_t timeval_to_ntp(struct timeval *t)
{
    uint64_t when;
    
    when = (uint64_t) t->tv_usec*281474977LLU;
    when >>= 16;
    return ((uint64_t) (t->tv_sec + NTP_EPOCH_OFFSET) << 32) | when;
}
/*- End of function --------------------------------------------------------*/

#ifdef ENABLE_SRTP
static const char *srtp_err_to_str(int err)
{
    switch (err)
    {
    case err_status_ok:
        return "nothing to report";
    case err_status_fail:
        return "unspecified failure";
    case err_status_bad_param:
        return "unsupported parameter";
    case err_status_alloc_fail:
        return "couldn't allocate memory";
    case err_status_dealloc_fail:
        return "couldn't deallocate properly";
    case err_status_init_fail:
        return "couldn't initialize";
    case err_status_terminus:
        return "can't process as much data as requested";
    case err_status_auth_fail:
        return "authentication failure";
    case err_status_cipher_fail:
        return "cipher failure";
    case err_status_replay_fail:
        return "replay check failed (bad index)";
    case err_status_replay_old:
        return "replay check failed (index too old)";
    case err_status_algo_fail:
        return "algorithm failed test routine";
    case err_status_no_such_op:
        return "unsupported operation";
    case err_status_no_ctx:
        return "no appropriate context found";
    case err_status_cant_check:
        return "unable to perform desired validation";
    case err_status_key_expired:
        return "can't use key any more";
    default:
        return "unknown";
    }
}
/*- End of function --------------------------------------------------------*/

static void srtp_event_cb(srtp_event_data_t *data)
{
    switch (data->event)
    {
    case event_ssrc_collision:
        if (option_debug)
        {
            libvale_log(LOG_DEBUG, "SSRC collision ssrc:%u dir:%d\n",
                          ntohl(data->stream->ssrc),
                          data->stream->direction);
        }
        break;
    case event_key_soft_limit:
        if (option_debug)
            libvale_log(LOG_DEBUG, "event_key_soft_limit\n");
        break;
    case event_key_hard_limit:
        if (option_debug)
            libvale_log(LOG_DEBUG, "event_key_hard_limit\n");
        break;
    case event_packet_index_limit:
        if (option_debug)
            libvale_log(LOG_DEBUG, "event_packet_index_limit\n");
        break;
    }
}
/*- End of function --------------------------------------------------------*/

void rfc3550_set_generate_key_cb(rfc3550_state_t *s,
                                 rtp_generate_key_cb cb)
{
    s->key_cb = cb;
}
/*- End of function --------------------------------------------------------*/

void rfc3550_policy_set_ssrc(rfc3550_policy_t *policy,
                            rfc3550_state_t *s, 
                            uint32_t ssrc,
                            int inbound)
{
    if (ssrc == 0  &&  !inbound  &&  s)
        ssrc = s->ssrc;
    if (ssrc)
    {
        policy->sp.ssrc.type = ssrc_specific;
        policy->sp.ssrc.value = ssrc;
    }
    else
    {
        policy->sp.ssrc.type = (inbound)  ?  ssrc_any_inbound  :  ssrc_any_outbound;
    }
}
/*- End of function --------------------------------------------------------*/

int rfc3550_add_policy(rfc3550_state_t *s, rfc3550_policy_t *policy)
{
    int res = 0;

    printf("Adding SRTP policy: %d %d %d %d %d %d\n",
           policy->sp.rtp.cipher_type,
           policy->sp.rtp.cipher_key_len,
           policy->sp.rtp.auth_type,
           policy->sp.rtp.auth_key_len,
           policy->sp.rtp.auth_tag_len,
           policy->sp.rtp.sec_serv);

    if (!s->srtp)
        res = srtp_create(&s->srtp, &policy->sp);
    else
        res = srtp_add_stream(s->srtp, &policy->sp);
    if (res != err_status_ok)
    {
        libvale_log(LOG_WARNING, "SRTP policy: %s\n", srtp_err_to_str(res));
        return -1;
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

rfc3550_policy_t *rfc3550_policy_alloc(void)
{
    rfc3550_policy_t *tmp;

    if ((tmp = malloc(sizeof(rfc3550_policy_t))) == NULL)
        return NULL;
    memset(tmp, 0, sizeof(rfc3550_policy_t));
    return tmp;
}
/*- End of function --------------------------------------------------------*/

void rfc3550_policy_destroy(rfc3550_policy_t *policy)
{
    if (policy->sp.key)
    {
        free(policy->sp.key);
        policy->sp.key = NULL;
    }
    free(policy);
}
/*- End of function --------------------------------------------------------*/

static int policy_set_suite(crypto_policy_t *p, int suite)
{
    switch (suite)
    {
    case RFC3550_AES_CM_128_HMAC_SHA1_80:
        p->cipher_type = AES_128_ICM;
        p->cipher_key_len = 30;
        p->auth_type = HMAC_SHA1;
        p->auth_key_len = 20;
        p->auth_tag_len = 10;
        p->sec_serv = sec_serv_conf_and_auth;
        return 0;
    case RFC3550_AES_CM_128_HMAC_SHA1_32:
        p->cipher_type = AES_128_ICM;
        p->cipher_key_len = 30;
        p->auth_type = HMAC_SHA1;
        p->auth_key_len = 20;
        p->auth_tag_len = 4;
        p->sec_serv = sec_serv_conf_and_auth;
        return 0;
    default:
        libvale_log(LOG_ERROR, "Invalid crypto suite: %d\n", suite);
        return -1;
    }
}
/*- End of function --------------------------------------------------------*/

int rfc3550_policy_set_suite(rfc3550_policy_t *policy, int suite)
{
    int res;

    res = policy_set_suite(&policy->sp.rtp, suite)
        | policy_set_suite(&policy->sp.rtcp, suite);
    return res;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_policy_set_master_key(rfc3550_policy_t *policy,
                               const uint8_t *key,
                               size_t key_len,
                               const uint8_t *salt,
                               size_t salt_len)
{
    size_t size = key_len + salt_len;
    uint8_t *master_key;

    if (policy->sp.key)
    {
        free(policy->sp.key);
        policy->sp.key = NULL;
    }
    if ((master_key = malloc(size)) == NULL)
        return -1;
    memcpy(master_key, key, key_len);
    memcpy(master_key + key_len, salt, salt_len);
    policy->sp.key = master_key;
    return 0;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_policy_set_encr_alg(rfc3550_policy_t *policy, int ealg)
{
    int type = -1;

    switch (ealg)
    {
    case MIKEY_SRTP_EALG_NULL:
        type = NULL_CIPHER;
        break;
    case MIKEY_SRTP_EALG_AESCM:
        type = AES_128_ICM;
        break;
    default:
        return -1;
    }
    policy->sp.rtp.cipher_type = type;
    policy->sp.rtcp.cipher_type = type;
    return 0;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_policy_set_auth_alg(rfc3550_policy_t *policy, int aalg)
{
    int type = -1;

    switch (aalg)
    {
    case MIKEY_SRTP_AALG_NULL:
        type = NULL_AUTH;
        break;
    case MIKEY_SRTP_AALG_SHA1HMAC:
        type = HMAC_SHA1;
        break;
    default:
        return -1;
    }
    policy->sp.rtp.auth_type = type;
    policy->sp.rtcp.auth_type = type;
    return 0;
}
/*- End of function --------------------------------------------------------*/

void rfc3550_policy_set_encr_keylen(rfc3550_policy_t *policy, int ekeyl)
{
    policy->sp.rtp.cipher_key_len = ekeyl;
    policy->sp.rtcp.cipher_key_len = ekeyl;
}
/*- End of function --------------------------------------------------------*/

void rfc3550_policy_set_auth_keylen(rfc3550_policy_t *policy, int akeyl)
{
    policy->sp.rtp.auth_key_len = akeyl;
    policy->sp.rtcp.auth_key_len = akeyl;
}
/*- End of function --------------------------------------------------------*/

void rfc3550_policy_set_srtp_auth_taglen(rfc3550_policy_t *policy, int autht)
{
    policy->sp.rtp.auth_tag_len = autht;
    policy->sp.rtcp.auth_tag_len = autht;
}
/*- End of function --------------------------------------------------------*/

void rfc3550_policy_set_srtp_encr_enable(rfc3550_policy_t *policy, int enable)
{
    int serv;

    serv = (enable)  ?  sec_serv_conf  :  sec_serv_none;
    policy->sp.rtp.sec_serv = (policy->sp.rtp.sec_serv & ~sec_serv_conf) | serv;
}
/*- End of function --------------------------------------------------------*/

void rfc3550_policy_set_srtcp_encr_enable(rfc3550_policy_t *policy, int enable)
{
    int serv;

    serv = (enable)  ?  sec_serv_conf  :  sec_serv_none;
    policy->sp.rtcp.sec_serv = (policy->sp.rtcp.sec_serv & ~sec_serv_conf) | serv;
}
/*- End of function --------------------------------------------------------*/

void rfc3550_policy_set_srtp_auth_enable(rfc3550_policy_t *policy, int enable)
{
    int serv;

    serv = (enable)  ?  sec_serv_auth  :  sec_serv_none;
    policy->sp.rtp.sec_serv = (policy->sp.rtp.sec_serv & ~sec_serv_auth) | serv;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_get_random(uint8_t *key, size_t len)
{
    int res = crypto_get_random(key, len);

    return (res != err_status_ok)  ?  -1  :  0;
}
/*- End of function --------------------------------------------------------*/
#endif    /* ENABLE_SRTP */

int rfc3550_add_csrc(rfc3550_state_t *s, uint32_t csrc)
{
    int i;

    if (s->csrc_count >= 15)
        return -1;
    csrc = htonl(csrc);
    /* If it is already there. Ignore the request. */
    for (i = 0;  i < s->csrc_count;  i++)
    {
        if (s->csrc[i] == csrc)
            return 0;
    }
    s->csrc[s->csrc_count++] = csrc;
    return 0;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_del_csrc(rfc3550_state_t *s, uint32_t csrc)
{
    int i;

    csrc = htonl(csrc);
    for (i = 0;  i < s->csrc_count;  i++)
    {
        if (s->csrc[i] == csrc)
        {
            /* Found it. Delete it. */
            s->csrc_count--;
            if (i < s->csrc_count)
                memcpy(&s->csrc[i], &s->csrc[i + 1], (s->csrc_count - i)*sizeof(s->csrc[0]));
            return 0;
        }
    }
    return -1;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_find_csrc(rfc3550_state_t *s, uint32_t csrc)
{
    int i;

    csrc = htonl(csrc);
    for (i = 0;  i < s->csrc_count;  i++)
    {
        if (s->csrc[i] == csrc)
        {
            /* Found it */
            return 0;
        }
    }
    return -1;
}
/*- End of function --------------------------------------------------------*/

uint32_t rfc3550_set_ssrc(rfc3550_state_t *s, uint32_t ssrc)
{
    uint32_t old_ssrc;
    
    old_ssrc = ntohl(s->ssrc);
    s->ssrc = htonl(ssrc);
    return old_ssrc;
}
/*- End of function --------------------------------------------------------*/

uint32_t rfc3550_get_ssrc(rfc3550_state_t *s)
{
    return ntohl(s->ssrc);
}
/*- End of function --------------------------------------------------------*/

static void init_seq(rfc3550_source_info_t *s, uint16_t seq)
{
    s->base_seq = seq;
    s->max_seq = seq;
    s->bad_seq = RTP_SEQ_MOD + 1;   /* so seq == bad_seq is false */
    s->cycles = 0;
    s->received = 0;
    s->received_prior = 0;
    s->expected_prior = 0;
    /* other initialization */
}
/*- End of function --------------------------------------------------------*/

static int update_seq(rfc3550_source_info_t *s, uint16_t seq)
{
    const int MAX_DROPOUT = 3000;
    const int MAX_MISORDER = 100;
    const int MIN_SEQUENTIAL = 2;
    uint16_t udelta;

    udelta = seq - s->max_seq;
    /*
     * A source is not valid until MIN_SEQUENTIAL packets with
     * sequential sequence numbers have been received.
     */
    if (s->probation)
    {
        if (seq == s->max_seq + 1)
        {
            /* The packet is in sequence */
            s->max_seq = seq;
            if (--s->probation == 0)
            {
                init_seq(s, seq);
                s->received++;
                return 0;
            }
        }
        else
        {
            s->probation = MIN_SEQUENTIAL - 1;
            s->max_seq = seq;
        }
        return -1;
    }
    else if (udelta < MAX_DROPOUT)
    {
        /* In order, with permissible gap */
        if (seq < s->max_seq)
        {
            /* Sequence number wrapped - count another 64K cycle. */
            s->cycles += RTP_SEQ_MOD;
        }
        s->max_seq = seq;
    }
    else if (udelta <= RTP_SEQ_MOD - MAX_MISORDER)
    {
        /* The sequence number made a very large jump */
        if (seq == s->bad_seq)
        {
            /*
             * Two sequential packets -- assume that the other side
             * restarted without telling us so just re-sync
             * (i.e., pretend this was the first packet).
             */
            init_seq(s, seq);
        }
        else
        {
            s->bad_seq = (seq + 1) & (RTP_SEQ_MOD-1);
            return -1;
        }
    }
    else
    {
        /* Duplicate or reordered packet */
    }
    s->received++;
    return 0;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_rx_rtp_packet(rfc3550_state_t *s, const uint8_t buf[], int len)
{
    int hdr_len;
    int res;
    uint32_t header_1;
    uint32_t seq_no;
    uint32_t payload_type;
    uint32_t mark;
    rfc3550_timestamp_t timestamp;
    uint32_t ssrc;
    uint32_t csrc_count;
    uint32_t *rtp_header;
    uint32_t transit;
    uint32_t arrival;

#ifdef ENABLE_SRTP
    if (s->srtp)
    {
        int res = 0;
        int i;

        for (i = 0;  i < 5;  i++)
        {
            srtp_hdr_t *header = buf;

            res = srtp_unprotect(s->srtp, buf, &len);
            if (res != err_status_no_ctx)
                break;
            if (s->key_cb == NULL)
                break;
            if (s->key_cb(rtp, ntohl(header->ssrc), s->data) < 0)
                break;
        }
        if (res != err_status_ok)
        {
            if (option_debug)
                libvale_log(LOG_DEBUG, "SRTP unprotect: %s\n", srtp_err_to_str(res));
            return -1;
        }
    }
#endif
    /* Start with the minimum possible header, that we might get if this packet is RTCP */
    hdr_len = 2*sizeof(uint32_t);
    if (len < hdr_len)
    {
        /* Too short for an RTP or RTCP packet. */
        return -1;
    }
    rtp_header = (uint32_t *) buf;
    header_1 = ntohl(get_unaligned_uint32(rtp_header));
    if ((header_1 & 0xC0000000) != 0x80000000)
    {
        /* RTP/RTCP version error - this might be UDPTL or something else muxed on the
           same port. */
        return -1;
    }
    if ((header_1 & (1 << 29)))
    {
        /* There must be some padding, probably added to assist with encryption quality.
           Since we are beyond decryption, we can strip it. */
        if (len < hdr_len + 1)
        {
            /* There is no length byte for the padding. */
            return -1;
        }
        len -= buf[len - 1];
    }
    payload_type = (header_1 >> 16) & 0xFF;
    if (payload_type >= RTCP_FIRST_PAYLOAD_TYPE  &&  payload_type <= RTCP_LAST_PAYLOAD_TYPE)
    {
        /* This appears to be a RTCP packet - it must be muxed with the RTP */
        return -1;
    }

    /* Presumably we have a genuine RTP packet */
    hdr_len = 3*sizeof(uint32_t);
    if (len < hdr_len)
    {
        /* Too short for an RTP packet. */
        return -1;
    }
    mark = (payload_type >> 7) & 1;
    payload_type &= 0x7F;
    /* Allow for the contributing sources, but don't process them at this time. */
    csrc_count = (header_1 >> 24) & 0x0F;
    hdr_len += (csrc_count*sizeof(uint32_t));
    if ((header_1 & (1 << 28)))
    {
        /* An RTP extension is present */
        /* TODO: Process the extension - maybe just report its presence -
                 in some way. */
        if (len < hdr_len + sizeof(uint32_t))
        {
            /* There is no length for the extension. */
            return -1;
        }
        hdr_len += (((ntohl(get_unaligned_uint32(&rtp_header[hdr_len >> 2])) & 0xFFFF) + 1)*sizeof(uint32_t));
    }
    if (len < hdr_len)
    {
        /* There must be something wrong with the extension. */
        return -1;
    }
    seq_no = header_1 & 0xFFFF;
    timestamp = ntohl(get_unaligned_uint32(&rtp_header[1]));
    ssrc = ntohl(get_unaligned_uint32(&rtp_header[2]));

    arrival = (uint32_t) (ntptime()/(uint64_t) 8000LL);
    transit = arrival - timestamp;
    s->source.transit = transit;
    s->source.jitter += (abs(transit - s->source.transit) - ((s->source.jitter + 8) >> 4));

    res = 0;
    if (s->payload_handler[payload_type].rx_packet_handler)
    {
        if (s->payload_handler[payload_type].rx_packet_handler(s->payload_handler[payload_type].user_data,
                                                               buf + hdr_len,
                                                               len - hdr_len,
                                                               seq_no,
                                                               timestamp,
                                                               ssrc,
                                                               payload_type,
                                                               mark) < 0)
        {
            printf("Bad RTP\n");
            res = -1;
        }
    }
    else if (s->default_payload_handler.rx_packet_handler)
    {
        if (s->default_payload_handler.rx_packet_handler(s->default_payload_handler.user_data,
                                                         buf + hdr_len,
                                                         len - hdr_len,
                                                         seq_no,
                                                         timestamp,
                                                         ssrc,
                                                         payload_type,
                                                         mark) < 0)
        {
            printf("Bad RTP\n");
            res = -1;
        }
    }
    else
    {
        printf("Unrecognised payload type %" PRIu32 "\n", payload_type);
        res = -1;
    }
    s->rx_seq_no = seq_no;
    s->last_rx_timestamp = timestamp;
    return res;
}
/*- End of function --------------------------------------------------------*/

static void rx_rtcp_sr(rfc3550_state_t *s, const uint8_t buf[], int len)
{
    int count;
    int i;
    uint32_t header_1;
    uint32_t ssrc;
    rfc3550_timestamp_t last_tx_timestamp;
    uint32_t tx_packets;
    uint32_t tx_octets;
    uint32_t lost;
    uint32_t fraction;
    uint32_t max_seq;
    uint32_t jitter;
    const uint8_t *rtcp_ptr;
    uint64_t now;

    header_1 = ntohl(get_unaligned_uint32(buf));
    count = (header_1 >> 24) & 0x1F;

    printf("RTCP type SR\n");

    rtcp_ptr = buf;

    ssrc = ntohl(get_unaligned_uint32(rtcp_ptr + 1*sizeof(uint32_t)));
    /* Unpack the sender info */
    now = ntohl(get_unaligned_uint32(rtcp_ptr + 2*sizeof(uint32_t)));
    now = (now << 32) | ntohl(get_unaligned_uint32(rtcp_ptr + 3*sizeof(uint32_t)));
    last_tx_timestamp = ntohl(get_unaligned_uint32(rtcp_ptr + 4*sizeof(uint32_t)));
    tx_packets = ntohl(get_unaligned_uint32(rtcp_ptr + 5*sizeof(uint32_t)));
    tx_octets = ntohl(get_unaligned_uint32(rtcp_ptr + 6*sizeof(uint32_t)));
    rtcp_ptr += 7*sizeof(uint32_t);
    
    printf("time %f, timestamp %" PRId32 ", packets %" PRId32 ", octets %" PRId32 "\n", (double) now/((double) UINT32_MAX + 1.0), last_tx_timestamp, tx_packets, tx_octets);
{
    struct timeval when;
    struct tm tm;
        
    when = ntp_to_timeval(now);
    printf("When %" PRId32 ":%" PRId32 "\n", (int32_t) when.tv_sec, (int32_t) when.tv_usec);
    tm = *localtime(&(when.tv_sec));
    printf("Date %02d/%02d/%02d %02d:%02d:%02d\n", tm.tm_year - 100, tm.tm_mon + 1, tm.tm_mday, tm.tm_hour, tm.tm_min, tm.tm_sec);    
}
    /* Unpack the block reports */
    for (i = 0;  i < count;  i++)
    {
        ssrc = ntohl(get_unaligned_uint32(rtcp_ptr + 0*sizeof(uint32_t)));
        lost = ntohl(get_unaligned_uint32(rtcp_ptr + 1*sizeof(uint32_t)));
        fraction = lost >> 24;
        lost &= 0xFFFFFF;
        max_seq = ntohl(get_unaligned_uint32(rtcp_ptr + 2*sizeof(uint32_t)));
        jitter = ntohl(get_unaligned_uint32(rtcp_ptr + 3*sizeof(uint32_t)));
        ntohl(get_unaligned_uint32(rtcp_ptr + 4*sizeof(uint32_t)));
        ntohl(get_unaligned_uint32(rtcp_ptr + 5*sizeof(uint32_t)));
        rtcp_ptr += 6*sizeof(uint32_t);
        printf("SSRC 0x%" PRIX32 ", lost %" PRId32 ", fraction %" PRId32 ", max_seq %" PRId32 ", jitter %" PRId32 "\n", ssrc, lost, fraction, max_seq, jitter);
    }
}
/*- End of function --------------------------------------------------------*/

static void rx_rtcp_rr(rfc3550_state_t *s, const uint8_t buf[], int len)
{
    int count;
    int i;
    uint32_t header_1;
    uint32_t ssrc;
    uint32_t lost;
    uint32_t fraction;
    uint32_t max_seq;
    uint32_t jitter;
    const uint8_t *rtcp_ptr;

    header_1 = ntohl(get_unaligned_uint32(buf));
    count = (header_1 >> 24) & 0x1F;

    printf("RTCP type RR\n");

    rtcp_ptr = buf;

    ssrc = ntohl(get_unaligned_uint32(rtcp_ptr + 1*sizeof(uint32_t)));
    rtcp_ptr += 2*sizeof(uint32_t);
    
    /* Unpack the block reports */
    for (i = 0;  i < count;  i++)
    {
        ssrc = ntohl(get_unaligned_uint32(rtcp_ptr + 0*sizeof(uint32_t)));
        lost = ntohl(get_unaligned_uint32(rtcp_ptr + 1*sizeof(uint32_t)));
        fraction = lost >> 24;
        lost &= 0xFFFFFF;
        max_seq = ntohl(get_unaligned_uint32(rtcp_ptr + 2*sizeof(uint32_t)));
        jitter = ntohl(get_unaligned_uint32(rtcp_ptr + 3*sizeof(uint32_t)));
        ntohl(get_unaligned_uint32(rtcp_ptr + 4*sizeof(uint32_t)));
        ntohl(get_unaligned_uint32(rtcp_ptr + 5*sizeof(uint32_t)));
        rtcp_ptr += 6*sizeof(uint32_t);
        printf("SSRC 0x%" PRIX32 ", lost %" PRId32 ", fraction %" PRId32 ", max_seq %" PRId32 ", jitter %" PRId32 "\n", ssrc, lost, fraction, max_seq, jitter);
    }
}
/*- End of function --------------------------------------------------------*/

static void rx_rtcp_sdes(rfc3550_state_t *s, const uint8_t buf[], int len)
{
    int count;
    int i;
    uint32_t header_1;
    uint32_t src;
    const uint8_t *ptr;
    const uint8_t *end;
    int element_type;
    int element_len;
    const uint8_t *element_value;
    char bufit[1000];

    /* An SDES packet has a chunk for each SRC, and each chunk contains multiple
       elements. */
    header_1 = ntohl(get_unaligned_uint32(buf));
    count = (header_1 >> 24) & 0x1F;
    printf("RTCP type SDES - %d chunks\n", count);
    ptr = buf + sizeof(uint32_t);
    end = buf + len*sizeof(uint32_t);
    for (i = 0;  i < count;  i++)
    {
        src = ntohl(get_unaligned_uint32(ptr));
        printf("RTCP type SDES - SRC %d is 0x%" PRIx32 "\n", i, src);
        ptr += sizeof(uint32_t);
        //s = find_member(src);
        while ((element_type = *ptr++) != RTCP_SDES_END)
        {
            element_len = *ptr++;
            element_value = ptr;
            ptr += element_len;
            if (ptr >= end)
            {
                printf("RTCP packet invalid\n");
                return;
            }
            memcpy(bufit, element_value, element_len);
            bufit[element_len] = '\0';
            printf("SDES element %d, len %d, value '%s'\n", element_type, element_len, bufit);
            //member_sdes(s, element_type, element_value, element_len);
        }
        /* Step past any padding */
        ptr += realign_padding32((int) (intptr_t) ptr);
    }
}
/*- End of function --------------------------------------------------------*/

static void rx_rtcp_bye(rfc3550_state_t *s, const uint8_t buf[], int len)
{
    int count;
    int i;
    uint32_t header_1;
    const uint8_t *ptr;
    char bufit[256 + 1];

    header_1 = ntohl(get_unaligned_uint32(buf));
    count = (header_1 >> 24) & 0x1F;

    printf("RTCP type BYE - %d entries\n", count);

    ptr = buf + sizeof(uint32_t);
    for (i = 0;  i < count;  i++)
        printf("SRC 0x%" PRIX32 "\n", ntohl(get_unaligned_uint32(ptr + i*sizeof(uint32_t))));
    ptr += count*sizeof(uint32_t);
    if (ptr < buf + len*sizeof(uint32_t))
    {
        if (ptr + ptr[0] + 1 <= buf + len*sizeof(uint32_t))
        {
            memcpy(bufit, ptr + 1, ptr[0]);
            bufit[ptr[0]] = '\0';
            printf("Message '%s'\n", bufit);
        }
        else
        {
            printf("Bad length\n");
        }
    }
}
/*- End of function --------------------------------------------------------*/

static void rx_rtcp_app(rfc3550_state_t *s, const uint8_t buf[], int len)
{
    int subtype;
    uint32_t header_1;
    const uint8_t *ptr;
    char name[4 + 1];

    header_1 = ntohl(get_unaligned_uint32(buf));
    subtype = (header_1 >> 24) & 0x1F;

    printf("RTCP type APP - subtype %d\n", subtype);
    ptr = buf + sizeof(uint32_t);
    printf("SRC 0x%" PRIX32 "\n", ntohl(get_unaligned_uint32(ptr)));
    ptr += 1*sizeof(uint32_t);
    memcpy(name, ptr, 4);
    name[4] = '\0';
    printf("APP name '%s' - %d bytes of data\n", name, (len - 3)*sizeof(uint32_t));
}
/*- End of function --------------------------------------------------------*/

static void rx_rtcp_rtpfb(rfc3550_state_t *s, const uint8_t buf[], int len)
{
}
/*- End of function --------------------------------------------------------*/

static void rx_rtcp_psfb(rfc3550_state_t *s, const uint8_t buf[], int len)
{
}
/*- End of function --------------------------------------------------------*/

static void rx_rtcp_xr(rfc3550_state_t *s, const uint8_t buf[], int len)
{
}
/*- End of function --------------------------------------------------------*/

int rfc3550_rx_rtcp_packet(rfc3550_state_t *s, const uint8_t buf[], int len)
{
    int hdr_len;
    uint32_t header_1;
    uint32_t payload_type;
    uint32_t rc_count;
    uint32_t length;
    uint32_t *rtcp_header;

#ifdef ENABLE_SRTP
    if (s->srtp)
    {
        int res = 0;
        int i;

        for (i = 0;  i < 5;  i++)
        {
            srtp_hdr_t *header = buf;

            res = srtcp_unprotect(s->srtp, buf, &len);
            if (res != err_status_no_ctx)
                break;
            if (s->key_cb == NULL)
                break;
            if (s->key_cb(rtp, ntohl(header->ssrc), s->data) < 0)
                break;
        }
        if (res != err_status_ok)
        {
            if (option_debug)
                libvale_log(LOG_DEBUG, "SRTP unprotect: %s\n", srtp_err_to_str(res));
            return -1;
        }
    }
#endif
    rtcp_header = (uint32_t *) buf;
    for (;;)
    {
        if (len < 2*sizeof(uint32_t))
        {
            /* Too short for an RTCP packet. */
            return -1;
        }
        hdr_len = 2*sizeof(uint32_t);
        header_1 = ntohl(rtcp_header[0]);
        if ((header_1 & 0xC0000000) != 0x80000000)
        {
            /* RTCP version error - this might be UDPTL or something else muxed on the
               same port. */
            return -1;
        }
        if ((header_1 & (1 << 29)))
        {
            /* There must be some padding, probably added to assist with encryption quality.
               Since we are beyond decryption, we can strip it. */
            if (len < hdr_len + 1)
            {
                /* There is no length byte for the padding. */
                return -1;
            }
            len -= buf[len - 1];
        }
        payload_type = (header_1 >> 16) & 0xFF;
        if (payload_type < RTCP_FIRST_PAYLOAD_TYPE  ||  payload_type > RTCP_LAST_PAYLOAD_TYPE)
        {
            /* This appears not to be a RTCP packet */
            return -1;
        }
        rc_count = (header_1 >> 24) & 0x1F;
        length = (header_1 & 0xFFFF) + 1;
        switch (payload_type)
        {
        case RTCP_SR:
            rx_rtcp_sr(s, (uint8_t *) rtcp_header, length);
            break;
        case RTCP_RR:
            rx_rtcp_rr(s, (uint8_t *) rtcp_header, length);
            break;
        case RTCP_SDES:
            rx_rtcp_sdes(s, (uint8_t *) rtcp_header, length);
            break;
        case RTCP_BYE:
            rx_rtcp_bye(s, (uint8_t *) rtcp_header, length);
            break;
        case RTCP_APP:
            rx_rtcp_app(s, (uint8_t *) rtcp_header, length);
            break;
        case RTCP_RTPFB:
            rx_rtcp_rtpfb(s, (uint8_t *) rtcp_header, length);
            break;
        case RTCP_PSFB:
            rx_rtcp_psfb(s, (uint8_t *) rtcp_header, length);
            break;
        case RTCP_XR:
            rx_rtcp_xr(s, (uint8_t *) rtcp_header, length);
            break;
        }
        printf("RTCP type %" PRIu32 ", length %" PRIu32 "\n", payload_type, length);
        rtcp_header += length;
        len -= length*sizeof(uint32_t);
        printf("0x%" PRIx32 " %d\n", rtcp_header[0], len);
        if (len < sizeof(uint32_t))
        {
            /* We should be exactly at the end of the data now */
            if (len)
                return -1;
            break;
        }
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

void rfc3550_set_peer(rfc3550_state_t *s, struct sockaddr_in *far)
{
    struct sockaddr_in far_rtcp;
    
    udp_socket_set_far(s->rtp_port, far);
    /* We need to cook up the RTCP address */
    memcpy(&far_rtcp, far, sizeof(far_rtcp));
    far_rtcp.sin_port = htons(ntohs(far->sin_port) + 1);
    udp_socket_set_far(s->rtcp_port, &far_rtcp);
    s->rx_seq_no = 0;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_build_rtp_packet(rfc3550_state_t *s,
                             const uint8_t msg[],
                             int msg_len,
                             uint8_t buf[],
                             rfc3550_timestamp_t timestamp,
                             int payload_type,
                             int mark)
{
    uint8_t *rtp_header;
    int hdr_len;
    int len;

    /* Take the timestamp from the caller, if a timestamp is provided. */
    if (timestamp)
        s->last_tx_timestamp = timestamp;
    else
        s->last_tx_timestamp += s->tx_timestamp_step;

    hdr_len = (3 + s->csrc_count)*sizeof(uint32_t);
    rtp_header = buf;
    put_unaligned_uint32(rtp_header, htonl((2 << 30) | (s->csrc_count << 24) | (mark << 23) | (payload_type << 16) | (s->tx_seq_no)));
    put_unaligned_uint32(rtp_header + sizeof(uint32_t), htonl(s->last_tx_timestamp));
    put_unaligned_uint32(rtp_header + 2*sizeof(uint32_t), s->ssrc);
    memcpy(rtp_header + 3*sizeof(uint32_t), s->csrc, s->csrc_count*sizeof(s->csrc[0]));
    memcpy(buf + hdr_len, msg, msg_len);
    len = msg_len + hdr_len;
#ifdef ENABLE_SRTP
    if (s->srtp)
    {
        if ((res = srtp_protect(s->srtp, buf, &len)) != err_status_ok)
            len = -1;
    }
#endif
    s->tx_seq_no++;
    s->tx_packets++;
    s->tx_octets += len;
    return len;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_build_rtp_packet_in_place(rfc3550_state_t *s,
                                      uint8_t msg[],
                                      int offset,
                                      rfc3550_timestamp_t timestamp,
                                      int payload_type,
                                      int mark,
                                      int msg_len)
{
    uint8_t *rtp_header;
    int hdr_len;
    int len;

    /* Take the timestamp from the caller, if a timestamp is provided. */
    if (timestamp)
        s->last_tx_timestamp = timestamp;
    else
        s->last_tx_timestamp += s->tx_timestamp_step;

    hdr_len = (3 + s->csrc_count)*sizeof(uint32_t);
    if (offset < hdr_len)
        return -1;
    rtp_header = msg - hdr_len;
    put_unaligned_uint32(rtp_header, htonl((2 << 30) | (s->csrc_count << 24) | (mark << 23) | (payload_type << 16) | (s->tx_seq_no)));
    put_unaligned_uint32(rtp_header + sizeof(uint32_t), htonl(s->last_tx_timestamp));
    put_unaligned_uint32(rtp_header + 2*sizeof(uint32_t), s->ssrc);
    memcpy(rtp_header + 3*sizeof(uint32_t), s->csrc, s->csrc_count*sizeof(s->csrc[0]));
    len = msg_len + hdr_len;
#ifdef ENABLE_SRTP
    if (s->srtp)
    {
        if ((res = srtp_protect(s->srtp, buf, &len)) != err_status_ok)
            len = -1;
    }
#endif
    s->tx_seq_no++;
    s->tx_packets++;
    s->tx_octets += (len + msg_len);
    return len;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_build_rtcp_packet_start(rfc3550_state_t *s,
                                    uint8_t buf[],
                                    int offset)
{
    /* There is actually nothing to do right now, but there may be in the future. */
    return offset;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_build_rtcp_add_sr_rr_start(rfc3550_state_t *s,
                                       uint8_t buf[],
                                       int offset,
                                       int add_sender)
{
    uint8_t *rtcp_ptr;
    uint64_t now;

    rtcp_ptr = buf + offset;

    /* We will have to fill in the header at the end, when we know the real length */
    if (add_sender)
        put_unaligned_uint32(rtcp_ptr, ((2 + 5)*sizeof(uint32_t) << 16) | 0x8000);
    else
        put_unaligned_uint32(rtcp_ptr, 2*sizeof(uint32_t) << 16);

    /* Pack the header */
    put_unaligned_uint32(rtcp_ptr + 1*sizeof(uint32_t), s->ssrc);

    /* Pack the sender info */
    now = ntptime();
    put_unaligned_uint32(rtcp_ptr + 2*sizeof(uint32_t), htonl((uint32_t) (now >> 32)));
    put_unaligned_uint32(rtcp_ptr + 3*sizeof(uint32_t), htonl((uint32_t) (now & 0xFFFFFFFF)));
    put_unaligned_uint32(rtcp_ptr + 4*sizeof(uint32_t), htonl(s->last_tx_timestamp));
    put_unaligned_uint32(rtcp_ptr + 5*sizeof(uint32_t), htonl(s->tx_packets));
    put_unaligned_uint32(rtcp_ptr + 6*sizeof(uint32_t), htonl(s->tx_octets));
    return offset;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_build_rtcp_add_sr_rr_chunk(rfc3550_state_t *s,
                                       uint8_t buf[],
                                       int offset,
                                       rfc3550_source_info_t *source)
{
    uint8_t *rtcp_ptr;
    uint32_t count;
    int length;
    int extended_max;
    int expected;
    int lost;
    int expected_interval;
    int lost_interval;
    int received_interval;
    int fraction;
    int sender_report;

    count = get_unaligned_uint32(buf + offset);
    length = (count >> 16) & 0xFFFF;
    sender_report = count & 0x8000;
    count &= 0x00FF;
    rtcp_ptr = buf + offset + length;

    /* Pack the block report */
    
    /* The following calculations come straight from RFC3550 */
    extended_max = source->cycles + source->max_seq;
    expected = extended_max - source->base_seq + 1;
    lost = expected - source->received;
    if (lost > 0xFFFFFF)
        lost = 0xFFFFFF;
    expected_interval = expected - source->expected_prior;
    source->expected_prior = expected;
    received_interval = source->received - source->received_prior;
    source->received_prior = source->received;
    lost_interval = expected_interval - received_interval;
    if (expected_interval == 0  ||  lost_interval <= 0)
        fraction = 0;
    else
        fraction = (lost_interval << 8)/expected_interval;

    put_unaligned_uint32(rtcp_ptr + 0*sizeof(uint32_t), htonl(source->ssrc));
    put_unaligned_uint32(rtcp_ptr + 1*sizeof(uint32_t), htonl((fraction << 24) | lost));
    put_unaligned_uint32(rtcp_ptr + 2*sizeof(uint32_t), htonl(source->max_seq));
    put_unaligned_uint32(rtcp_ptr + 3*sizeof(uint32_t), htonl(source->jitter >> 4));
    if (source->last_sender_report_rx_time)
    {
        put_unaligned_uint32(rtcp_ptr + 4*sizeof(uint32_t), htonl(ntp_to_reduced_ntp(source->last_sender_report_time)));
        put_unaligned_uint32(rtcp_ptr + 5*sizeof(uint32_t), htonl(ntp_to_reduced_ntp(ntptime() - source->last_sender_report_rx_time)));
    }
    else
    {
        put_unaligned_uint32(rtcp_ptr + 4*sizeof(uint32_t), 0);
        put_unaligned_uint32(rtcp_ptr + 5*sizeof(uint32_t), 0);
    }

    count++;
    length += 6*sizeof(uint32_t);
    put_unaligned_uint32(buf + offset, count | sender_report | (length << 16));
    return offset;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_build_rtcp_add_sr_rr_end(rfc3550_state_t *s,
                                     uint8_t buf[],
                                     int offset)
{
    int length;
    int final_offset;
    int report_type;
    uint32_t count;

    count = get_unaligned_uint32(buf + offset);
    length = (count >> 16) & 0xFFFF;
    final_offset = offset + length;
    report_type = (count & 0x8000)  ?  RTCP_SR  :  RTCP_RR;
    count &= 0x7FFF;
    length = (length >> 2) - 1;
    put_unaligned_uint32(buf + offset, htonl((2 << 30) | (count << 24) | (report_type << 16) | (length)));
    return final_offset;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_build_rtcp_add_sdes_start(rfc3550_state_t *s,
                                       uint8_t buf[],
                                       int offset)
{
    /* We will have to fill in the header at the end, when we know the real length */
    put_unaligned_uint32(buf + offset, sizeof(uint32_t) << 16);
    return offset;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_build_rtcp_add_sdes_chunk(rfc3550_state_t *s,
                                      uint8_t buf[],
                                      int offset,
                                      uint32_t ssrc,
                                      int sdes_count,
                                      rtcp_sdes_item_t items[])
{
    uint8_t *rtcp_ptr;
    int i;
    int pad_len;
    int length;
    uint32_t count;

    count = get_unaligned_uint32(buf + offset);
    length = (count >> 16) & 0xFFFF;
    count &= 0xFFFF;

    rtcp_ptr = buf + offset + length;

    put_unaligned_uint32(rtcp_ptr, htonl(ssrc));
    rtcp_ptr += 1*sizeof(uint32_t);
    for (i = 0;  i < sdes_count;  i++)
    {
        /* We have no need to check the range of the length, as it cannot
           actually exceed RTCP_MAX_SDES, being only a byte value */
        *rtcp_ptr++ = items[i].type;
        *rtcp_ptr++ = items[i].length;
        memcpy(rtcp_ptr, items[i].value, items[i].length);
        rtcp_ptr += items[i].length;
    }
    /*endfor*/
    /* Add the terminator */
    *rtcp_ptr++ = 0;

    /* Pad (if necessary) to next 4-octet boundary */
    if ((pad_len = realign_padding32((int) (intptr_t) rtcp_ptr)))
    {
        memset(rtcp_ptr, 0, pad_len);
        rtcp_ptr += pad_len;
    }
    count++;
    length = rtcp_ptr - buf - offset;
    put_unaligned_uint32(buf + offset, count | (length << 16));
    return offset;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_build_rtcp_add_sdes_end(rfc3550_state_t *s,
                                    uint8_t buf[],
                                    int offset)
{
    int length;
    int final_offset;
    uint32_t count;

    count = get_unaligned_uint32(buf + offset);
    length = (count >> 16) & 0xFFFF;
    final_offset = offset + length;
    count &= 0xFFFF;
    length = (length >> 2) - 1;
    put_unaligned_uint32(buf + offset, htonl((2 << 30) | (count << 24) | (RTCP_SDES << 16) | (length)));
    return final_offset;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_build_rtcp_add_bye(rfc3550_state_t *s,
                               uint8_t buf[],
                               int offset,
                               int sc_count,
                               uint32_t ssrc[],
                               const char *message)
{
    uint8_t *rtcp_ptr;
    int pad_len;
    int data_len;
    int i;
    int length;

    rtcp_ptr = buf + offset;

    data_len = (message  &&  message[0])  ?  (strlen(message) + 1)  :  0;
    pad_len = realign_padding32(data_len);
    length = sc_count + ((data_len + pad_len) >> 2);

    put_unaligned_uint32(rtcp_ptr, htonl((2 << 30) | (sc_count << 24) | (RTCP_BYE << 16) | (length)));
    for (i = 0;  i < sc_count;  i++)
        put_unaligned_uint32(rtcp_ptr + (i + 1)*sizeof(uint32_t), htonl(ssrc[i]));
    rtcp_ptr += (1 + sc_count)*sizeof(uint32_t);
    if (data_len)
    {
        rtcp_ptr[0] = data_len - 1;
        memcpy(rtcp_ptr + 1, message, data_len - 1);
        rtcp_ptr += data_len;
        memset(rtcp_ptr, 0, pad_len);
        rtcp_ptr += pad_len;
    }
    return (rtcp_ptr - buf);
}
/*- End of function --------------------------------------------------------*/

int rfc3550_build_rtcp_add_app(rfc3550_state_t *s,
                               uint8_t buf[],
                               int offset,
                               uint32_t ssrc,
                               const char name[4],
                               const void *data,
                               unsigned int data_len)
{
    uint8_t *rtcp_ptr;
    int pad_len;
    int sub_type;
    int length;

    rtcp_ptr = buf + offset;
    sub_type = 0;
    
    pad_len = realign_padding32(data_len);
    length = 2 + ((data_len + pad_len) >> 2);

    put_unaligned_uint32(rtcp_ptr, htonl((2 << 30) | (sub_type << 24) | (RTCP_APP << 16) | (length)));
    put_unaligned_uint32(rtcp_ptr + 1*sizeof(uint32_t), htonl(ssrc));
    rtcp_ptr += (2*sizeof(uint32_t));

    memcpy(rtcp_ptr, name, 1*sizeof(uint32_t));
    rtcp_ptr += (1*sizeof(uint32_t));
    memmove(rtcp_ptr, data, data_len);
    rtcp_ptr += data_len;
    memset(rtcp_ptr, 0, pad_len);
    rtcp_ptr += pad_len;
    
    return (rtcp_ptr - buf);
}
/*- End of function --------------------------------------------------------*/

int rfc3550_build_rtcp_add_rtpfb(rfc3550_state_t *s,
                                 uint8_t buf[],
                                 int offset,
                                 uint32_t ssrc)
{    
    return 0;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_build_rtcp_add_psfb(rfc3550_state_t *s,
                                uint8_t buf[],
                                int offset,
                                uint32_t ssrc)
{    
    return 0;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_build_rtcp_add_xr(rfc3550_state_t *s,
                              uint8_t buf[],
                              int offset,
                              uint32_t ssrc)
{    
    return 0;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_build_rtcp_packet_end(rfc3550_state_t *s,
                                  uint8_t buf[],
                                  int offset)
{
#ifdef ENABLE_SRTP
    if (s->srtp)
    {
        if ((res = srtp_protect(s->srtp, buf, &offset)) != err_status_ok)
            len = -1;
    }
#endif
    return offset;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_set_payload_handler(rfc3550_state_t *s,
                                int payload_type,
                                rtp_rx_packet_handler_t rx_packet_handler,
                                void *user_data)
{
    if (payload_type < -1  ||  payload_type > 127)
        return -1;
    /* Don't allowed the reserved types, which could be confused with RTCP types, if
       RTCP is muxed with RTP. */
    if (payload_type >= (RTCP_FIRST_PAYLOAD_TYPE & 0x7F)  &&  payload_type <= (RTCP_LAST_PAYLOAD_TYPE & 0x7F))
        return -1;
    if (payload_type == -1)
    {
        s->default_payload_handler.rx_packet_handler = rx_packet_handler;
        s->default_payload_handler.user_data = user_data;
    }
    else
    {
        s->payload_handler[payload_type].rx_packet_handler = rx_packet_handler;
        s->payload_handler[payload_type].user_data = user_data;
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_get_payload_handler(rfc3550_state_t *s,
                                int payload_type,
                                rtp_rx_packet_handler_t **rx_packet_handler,
                                void **user_data)
{
    if (payload_type < 0  ||  payload_type > 127)
        return -1;
    if (s->payload_handler[payload_type].rx_packet_handler == NULL)
        return -1;
    *rx_packet_handler = s->payload_handler[payload_type].rx_packet_handler;
    *user_data = s->payload_handler[payload_type].user_data;
    return 0;
}
/*- End of function --------------------------------------------------------*/

void rfc3550_reset(rfc3550_state_t *s)
{
    s->last_tx_timestamp = 0;
    s->last_rx_timestamp = 0;
    s->tx_seq_no = 0;
    s->rx_seq_no = 0;
    s->csrc_count = 0;
}
/*- End of function --------------------------------------------------------*/

rfc3550_state_t *rfc3550_init(rfc3550_state_t *s, int with_rtcp)
{
    if (s == NULL)
    {
        if ((s = (rfc3550_state_t *) malloc(sizeof(*s))) == NULL)
            return NULL;
    }
    memset(s, 0, sizeof(*s));
    s->ssrc = rand();
    s->tx_seq_no = rand() & 0xFFFF;
    return s;
}
/*- End of function --------------------------------------------------------*/

int rfc3550_release(rfc3550_state_t *s)
{
#ifdef ENABLE_SRTP
    if (s->srtp)
        srtp_dealloc(s->srtp);
#endif
    free(s);
    return 0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
