/*
 * Vale - a library for media streaming.
 *
 * udptl.h - An implementation of the UDPTL protocol defined in T.38,
 *           less the packet exchange part
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2005 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: udptl.h,v 1.1.1.1 2007/04/03 13:13:37 steveu Exp $
 */

/*! \file */

#if !defined(_VALE_UDPTL_H_)
#define _VALE_UDPTL_H_

#define LOCAL_FAX_MAX_DATAGRAM      400
#define LOCAL_FAX_MAX_FEC_PACKETS   5

#define UDPTL_BUF_MASK              15

enum
{
    UDPTL_ERROR_CORRECTION_NONE,
    UDPTL_ERROR_CORRECTION_FEC,
    UDPTL_ERROR_CORRECTION_REDUNDANCY
};

typedef struct
{
    int buf_len;
    uint8_t buf[LOCAL_FAX_MAX_DATAGRAM];
} udptl_fec_tx_buffer_t;

typedef struct
{
    int buf_len;
    uint8_t buf[LOCAL_FAX_MAX_DATAGRAM];
    int fec_len[LOCAL_FAX_MAX_FEC_PACKETS];
    uint8_t fec[LOCAL_FAX_MAX_FEC_PACKETS][LOCAL_FAX_MAX_DATAGRAM];
    int fec_span;
    int fec_entries;
} udptl_fec_rx_buffer_t;

typedef int (udptl_rx_packet_handler_t)(void *user_data, const uint8_t msg[], int len, int seq_no);

typedef struct
{
    udptl_rx_packet_handler_t *rx_packet_handler;
    void *user_data;

    /*! This option indicates the error correction scheme used in transmitted UDPTL
        packets. */
    int error_correction_scheme;

    /*! This option indicates the number of error correction entries transmitted in
        UDPTL packets. */
    int error_correction_entries;

    /*! This option indicates the span of the error correction entries in transmitted
        UDPTL packets (FEC only). */
    int error_correction_span;

    /*! This option indicates the maximum size of a datagram that can be accepted by
        the remote device. */
    int far_max_datagram_size;

    /*! This option indicates the maximum size of a datagram that we are prepared to
        accept. */
    int local_max_datagram_size;

    int verbose;

    int tx_seq_no;
    int rx_seq_no;
    int rx_expected_seq_no;

    udptl_fec_tx_buffer_t tx[UDPTL_BUF_MASK + 1];
    udptl_fec_rx_buffer_t rx[UDPTL_BUF_MASK + 1];
} udptl_state_t;

#if defined(__cplusplus)
extern "C"
{
#endif

int udptl_rx_packet(udptl_state_t *s, const uint8_t buf[], int len);

int udptl_build_packet(udptl_state_t *s, uint8_t buf[], const uint8_t msg[], int msg_len);

udptl_state_t *udptl_init(udptl_state_t *s,
                          int ec_scheme,
                          int span,
                          int entries,
                          udptl_rx_packet_handler_t rx_packet_handler,
                          void *user_data);

int udptl_release(udptl_state_t *s);

#if defined(__cplusplus)
}
#endif

#endif
/*- End of file ------------------------------------------------------------*/
