/*
 * Vale - a library for media streaming.
 *
 * t38_tests.c - Use T.38 to exercise most forms of streaming
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2007 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: t38_tests.c,v 1.1.1.1 2007/04/03 13:13:37 steveu Exp $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <unistd.h>
#include <sys/uio.h>
#include <math.h>
#include <memory.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#include <netdb.h>
#if !defined(HAVE_GETADDRINFO) || !defined(HAVE_GETNAMEINFO)
#include <getaddrinfo.h>
#endif
#include <fcntl.h>

#include <event.h>

#include <tiffio.h>
#include <spandsp.h>

#include "vale.h"

#define INPUT_FILE_NAME         "/home/steveu/spandsp/itutests/fax/itutests.tif"
#define OUTPUT_FILE_NAME        "t38.tif"

#define RFC3550_PAYLOAD_TYPE_RFC2198    42
#define RFC3550_PAYLOAD_TYPE_T38        43

typedef int (io_callback_handler_t)(void *user_data, int fd, int type);

typedef struct
{
    t38_terminal_state_t t38_state;

    int transport;

    tcp_state_t *tcp_state;
    tcp_state_t *tcp_state2;
    udp_state_t *udp_state;
    udp_state_t *udp_state2;
    tpkt_state_t tpkt_state;
    udptl_state_t udptl_state;
    rfc3550_state_t rfc3550_state;
    rfc2198_state_t rfc2198_state;
    struct sockaddr_storage farx;
    struct event evts[10];
} tx_packet_state_t;

tx_packet_state_t tx_packet_state;

int done[2] = {FALSE, FALSE};
int succeeded[2] = {FALSE, FALSE};

struct timeval target_time;
struct event timer_evt;

static void phase_b_handler(t30_state_t *s, void *user_data, int result)
{
    int i;
    
    i = (int) (intptr_t) user_data;
    printf("%c: Phase B handler on channel %c - (0x%X) %s\n", i, i, result, t30_frametype(result));
}
/*- End of function --------------------------------------------------------*/

static void phase_d_handler(t30_state_t *s, void *user_data, int result)
{
    int i;
    t30_stats_t t;
    char ident[21];

    i = (int) (intptr_t) user_data;
    printf("%c: Phase D handler on channel %c - (0x%X) %s\n", i, i, result, t30_frametype(result));
    t30_get_transfer_statistics(s, &t);
    printf("%c: Phase D: bit rate %d\n", i, t.bit_rate);
    printf("%c: Phase D: ECM %s\n", i, (t.error_correcting_mode)  ?  "on"  :  "off");
    printf("%c: Phase D: pages transferred %d\n", i, t.pages_transferred);
    printf("%c: Phase D: image size %d x %d\n", i, t.width, t.length);
    printf("%c: Phase D: image resolution %d x %d\n", i, t.x_resolution, t.y_resolution);
    printf("%c: Phase D: bad rows %d\n", i, t.bad_rows);
    printf("%c: Phase D: longest bad row run %d\n", i, t.longest_bad_row_run);
    printf("%c: Phase D: coding method %s\n", i, t4_encoding_to_str(t.encoding));
    printf("%c: Phase D: image size %d\n", i, t.image_size);
    t30_get_local_ident(s, ident);
    printf("%c: Phase D: local ident '%s'\n", i, ident);
    t30_get_far_ident(s, ident);
    printf("%c: Phase D: remote ident '%s'\n", i, ident);
}
/*- End of function --------------------------------------------------------*/

static void phase_e_handler(t30_state_t *s, void *user_data, int result)
{
    int i;
    t30_stats_t t;
    char ident[21];
    
    i = (int) (intptr_t) user_data;
    printf("%c: Phase E handler on channel %c - (%d) %s\n", i, i, result, t30_completion_code_to_str(result));
    t30_get_transfer_statistics(s, &t);
    printf("%c: Phase E: bit rate %d\n", i, t.bit_rate);
    printf("%c: Phase E: ECM %s\n", i, (t.error_correcting_mode)  ?  "on"  :  "off");
    printf("%c: Phase E: pages transferred %d\n", i, t.pages_transferred);
    printf("%c: Phase E: image size %d x %d\n", i, t.width, t.length);
    printf("%c: Phase E: image resolution %d x %d\n", i, t.x_resolution, t.y_resolution);
    printf("%c: Phase E: bad rows %d\n", i, t.bad_rows);
    printf("%c: Phase E: longest bad row run %d\n", i, t.longest_bad_row_run);
    printf("%c: Phase E: coding method %s\n", i, t4_encoding_to_str(t.encoding));
    printf("%c: Phase E: image size %d bytes\n", i, t.image_size);
    t30_get_local_ident(s, ident);
    printf("%c: Phase E: local ident '%s'\n", i, ident);
    t30_get_far_ident(s, ident);
    printf("%c: Phase E: remote ident '%s'\n", i, ident);
    succeeded[i - 'A'] = (result == T30_ERR_OK)  &&  (t.pages_transferred == 12);
    //done[i - 'A'] = TRUE;
}
/*- End of function --------------------------------------------------------*/

static void timeout_cb(int fd, short int event, void *user_data)
{
    t38_terminal_state_t *s;
    struct timeval tv;

    s = (t38_terminal_state_t *) user_data;
    timerclear(&tv);
    tv.tv_usec = 20000;
    timeradd(&target_time, &tv, &target_time);

    gettimeofday(&tv, NULL);
    timersub(&target_time, &tv, &tv);

    event_add(&timer_evt, &tv);
    
    done[0] = t38_terminal_send_timeout(s, 160);
}
/*- End of function --------------------------------------------------------*/

static int process_rx_tpkt_packet(void *user_data, const uint8_t buf[], int len)
{
    t38_terminal_state_t *s;
    static int seq_no = 0;

    s = (t38_terminal_state_t *) user_data;
    t38_core_rx_ifp_packet(&s->t38, buf, len, seq_no++);
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int process_rx_rtp_packet(void *user_data,
                                 const uint8_t buf[],
                                 int len,
                                 uint16_t seq_no,
                                 uint32_t timestamp,
                                 uint32_t ssrc,
                                 int payload_type,
                                 int mark)
{
    t38_terminal_state_t *s;

    s = (t38_terminal_state_t *) user_data;
    t38_core_rx_ifp_packet(&s->t38, buf, len, seq_no);
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int process_rx_udptl_packet(void *user_data,
                                   const uint8_t buf[],
                                   int len,
                                   int seq_no)
{
    t38_terminal_state_t *s;

    s = (t38_terminal_state_t *) user_data;
    t38_core_rx_ifp_packet(&s->t38, buf, len, seq_no);
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int tx_packet_handler(t38_core_state_t *s, void *user_data, const uint8_t *buf, int len, int count)
{
    tx_packet_state_t *tx;
    uint8_t pkt_buf[4096];
    int pkt_len;
    int i;
    uint8_t rfc2198_buf[4096];
    int rfc2198_len;
    int mark;
    uint32_t ts;
    static uint32_t tx_ts = 0;

    span_log(&s->logging, SPAN_LOG_FLOW, "Send seq %d, len %d, count %d\n", s->tx_seq_no, len, count);

    printf("IFP: ");
    for (i = 0;  i < len;  i++)
        printf(" %02X", buf[i]);
    printf("\n");

    tx = (tx_packet_state_t *) user_data;
    
    switch (tx_packet_state.transport)
    {
    case 0:
#if 0
        pkt_len = tpkt_build_packet(&tx->tpkt_state, pkt_buf, buf, len);
        tcp_socket_send(tx->tcp_state2, pkt_buf, pkt_len);
#else
        pkt_len = tpkt_build_header(&tx->tpkt_state, pkt_buf, buf, len);
        tcp_socket_send_with_header(tx->tcp_state2, pkt_buf, pkt_len, buf, len);
#endif
        break;
    case 1:
        pkt_len = udptl_build_packet(&tx->udptl_state, pkt_buf, buf, len);
        for (i = 0;  i < count;  i++)
        {
            if (udp_socket_send(tx->udp_state, pkt_buf, pkt_len, 0) < 0)
                printf("send error - %d\n", errno);
        }
        break;
    case 2:
        ts = tx_ts++;
        mark = 0;
        rfc2198_len = rfc2198_build_packet(&tx->rfc2198_state, buf, len, rfc2198_buf, ts, RFC3550_PAYLOAD_TYPE_T38, 0);
        pkt_len = rfc3550_build_rtp_packet(&tx->rfc3550_state, rfc2198_buf, rfc2198_len, pkt_buf, ts, RFC3550_PAYLOAD_TYPE_RFC2198, mark);
        for (i = 0;  i < count;  i++)
        {
            if (udp_socket_send(tx->udp_state, pkt_buf, pkt_len, 0) < 0)
                printf("send error - %d\n", errno);
        }
        break;
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int t38_setup(int t38_version, int caller, int use_ecm, const char *input_file_name)
{
    if (t38_terminal_init(&tx_packet_state.t38_state, caller, tx_packet_handler, &tx_packet_state) == NULL)
    {
        fprintf(stderr, "Cannot start the T.38 channel\n");
        exit(2);
    }
    t38_set_t38_version(&tx_packet_state.t38_state.t38, t38_version);
    span_log_set_level(&tx_packet_state.t38_state.logging, SPAN_LOG_DEBUG | SPAN_LOG_SHOW_TAG | SPAN_LOG_SHOW_SAMPLE_TIME);
    span_log_set_tag(&tx_packet_state.t38_state.logging, "T.38");
    span_log_set_level(&tx_packet_state.t38_state.t38.logging, SPAN_LOG_DEBUG | SPAN_LOG_SHOW_TAG | SPAN_LOG_SHOW_SAMPLE_TIME);
    span_log_set_tag(&tx_packet_state.t38_state.t38.logging, "T.38");
    span_log_set_level(&tx_packet_state.t38_state.t30_state.logging, SPAN_LOG_DEBUG | SPAN_LOG_SHOW_TAG | SPAN_LOG_SHOW_SAMPLE_TIME);
    span_log_set_tag(&tx_packet_state.t38_state.t30_state.logging, "T.38");
    if (use_ecm)
        t30_set_supported_compressions(&tx_packet_state.t38_state.t30_state, T30_SUPPORT_T4_1D_COMPRESSION | T30_SUPPORT_T4_2D_COMPRESSION | T30_SUPPORT_T6_COMPRESSION);
    t30_set_phase_b_handler(&tx_packet_state.t38_state.t30_state, phase_b_handler, (void *) (intptr_t) 'A');
    t30_set_phase_d_handler(&tx_packet_state.t38_state.t30_state, phase_d_handler, (void *) (intptr_t) 'A');
    t30_set_phase_e_handler(&tx_packet_state.t38_state.t30_state, phase_e_handler, (void *) (intptr_t) 'A');
    t30_set_ecm_capability(&tx_packet_state.t38_state.t30_state, use_ecm);

    if (caller)
    {
        t30_set_local_ident(&tx_packet_state.t38_state.t30_state, "11111111");
        t30_set_tx_file(&tx_packet_state.t38_state.t30_state, input_file_name, -1, -1);
    }
    else
    {
        t30_set_local_ident(&tx_packet_state.t38_state.t30_state, "22222222");
        t30_set_rx_file(&tx_packet_state.t38_state.t30_state, OUTPUT_FILE_NAME, -1);
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

static void rtp_udptl_handler(int fd, short int event, void *user_data)
{
    int len;
    int action;
    tx_packet_state_t *s;
    uint8_t buf[2048];
    
    s = (tx_packet_state_t *) user_data;
    /* Reschedule this event */
    event_add(&s->evts[0], NULL);
    if ((len = udp_socket_recv(s->udp_state,
                               buf,
                               sizeof(buf),
                               0,
                               &action)) < 0)
    {
        printf("Error in RTP/UDPTL receive (%d) %s\n", errno, strerror(errno));
        return;
    }
    switch (tx_packet_state.transport)
    {
    case 1:
        udptl_rx_packet(&s->udptl_state, buf, len);
        break;
    case 2:
        rfc3550_rx_rtp_packet(&s->rfc3550_state, buf, len);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void rtcp_handler(int fd, short int event, void *user_data)
{
    int len;
    int action;
    tx_packet_state_t *s;
    uint8_t buf[2048];
    
    s = (tx_packet_state_t *) user_data;
    /* Reschedule this event */
    event_add(&s->evts[1], NULL);
    if ((len = udp_socket_recv(s->udp_state2,
                               buf,
                               sizeof(buf),
                               0,
                               &action)) < 0)
    {
        printf("Error in RTCP receive (%d) %s\n", errno, strerror(errno));
        return;
    }
    rfc3550_rx_rtcp_packet(&tx_packet_state.rfc3550_state, buf, len);
}
/*- End of function --------------------------------------------------------*/

static void tcp_handler(int fd, short int event, void *user_data)
{
    int len;
    tx_packet_state_t *s;
    uint8_t buf[2048];
    
    s = (tx_packet_state_t *) user_data;
    /* Reschedule this event */
    event_add(&s->evts[3], NULL);
    if ((len = tcp_socket_recv(tx_packet_state.tcp_state2, buf, 2048, 0)) < 0)
    {
        printf("Error (%d) %s\n", errno, strerror(errno));
        return;
    }
    if (tx_packet_state.transport == 0)
        tpkt_rx_chunk(&tx_packet_state.tpkt_state, buf, len);
}
/*- End of function --------------------------------------------------------*/

static void tcp_listen_handler(int fd, short int event, void *user_data)
{
    tx_packet_state_t *s;
    
    s = (tx_packet_state_t *) user_data;
    /* Reschedule this event */
    event_add(&s->evts[2], NULL);
    if ((s->tcp_state2 = tcp_socket_accept(s->tcp_state)) == NULL)
    {
        printf("Accept result %d\n", errno);
        return;
    }
    /*endif*/
    event_set(&s->evts[3], tcp_socket_fd(s->tcp_state2), EV_READ, tcp_handler, s);
    /* Add it to the active events, without a timeout */
    event_add(&s->evts[3], NULL);
    printf("Answered\n");
}
/*- End of function --------------------------------------------------------*/

static void tcp_connect_handler(int fd, short int event, void *user_data)
{
    tx_packet_state_t *s;
    
    s = (tx_packet_state_t *) user_data;
    if (tcp_socket_connection_result(s->tcp_state) < 0)
        return;
    /*endif*/
    s->tcp_state2 = s->tcp_state;
    event_set(&s->evts[3], tcp_socket_fd(s->tcp_state2), EV_READ, tcp_handler, s);
    /* Add it to the active events, without a timeout */
    event_add(&s->evts[3], NULL);
    if (tcp_socket_connection_result(s->tcp_state) < 0)
    {
        printf("Connection result %d\n", errno);
        exit(2);
    }
    /*endif*/
    s->tcp_state2 = s->tcp_state;
    printf("Connected\n");
}
/*- End of function --------------------------------------------------------*/

static int exchange_handling(int caller, int local_port, int far_port)
{
    struct timeval tv;
    int i;
    udp_state_t *udp;
    tx_packet_state_t *s;

    s = &tx_packet_state;
    if (s->transport == 0)
    {
        printf("local_port %d, far port is %d\n", local_port, far_port);
        if (far_port < 0)
        {
            //printf("Listening on %d\n", ntohs(((struct sockaddr_in) s->farx).sin_port));
            if (tcp_socket_listen(s->tcp_state) < 0)
            {
                printf("Listen failed - errno = %d", errno);
                exit(2);
            }
            /*endif*/
        }
        else
        {
            printf("Calling\n");
            if (tcp_socket_connect(s->tcp_state) < 0)
            {
                printf("Connect failed - errno = %d", errno);
                exit (2);
            }
            printf("Connecting\n");
        }
    }

    for (i = 0;  i < 10;  i++)
    {
        udp = udp_socket_find_group_element(tx_packet_state.udp_state, i);
        if (udp == NULL)
            break;
        printf("Add element %d\n", i);
        if (i == 0)
        {
            event_set(&s->evts[i], udp_socket_fd(udp), EV_READ, rtp_udptl_handler, &tx_packet_state);
        }
        if (i == 1)
        {
            event_set(&s->evts[i], udp_socket_fd(udp), EV_READ, rtcp_handler, &tx_packet_state);
        }
        /* Add it to the active events, without a timeout */
        event_add(&s->evts[i], NULL);
    }
    if (s->transport == 0)
    {
        if (far_port < 0)
        {
            printf("Add listener\n");
            event_set(&s->evts[2], tcp_socket_fd(tx_packet_state.tcp_state), EV_READ, tcp_listen_handler, &tx_packet_state);
            /* Add it to the active events, without a timeout */
            event_add(&s->evts[2], NULL);
        }
        else
        {
            printf("Add caller\n");
            event_set(&s->evts[2], tcp_socket_fd(tx_packet_state.tcp_state), EV_WRITE, tcp_connect_handler, &tx_packet_state);
            /* Add it to the active events, without a timeout */
            event_add(&s->evts[2], NULL);
        }
    }
    tx_packet_state.udp_state2 = udp_socket_find_group_element(tx_packet_state.udp_state, 1);

    timerclear(&tv);
    tv.tv_usec = 20000;
    gettimeofday(&target_time, NULL);
    timeradd(&target_time, &tv, &target_time);
    evtimer_set(&timer_evt, timeout_cb, &s->t38_state);
    event_add(&timer_evt, &tv);

    event_dispatch();

    return  0;
}
/*- End of function --------------------------------------------------------*/

static void fred(const char *host)
{
    struct addrinfo hints;
    struct addrinfo *res;
    int error;
    char hbuf[NI_MAXHOST];
    char sbuf[NI_MAXSERV];

    memset(&hints, 0, sizeof(hints));
    /* set-up hints structure */
    hints.ai_family = PF_UNSPEC;
    hints.ai_flags = AI_PASSIVE;
    hints.ai_socktype = SOCK_STREAM;

    if ((error = getaddrinfo(host, "telnet", &hints, &res)))
    {
        perror(gai_strerror(error));
        return;
    }
    /*
     * "res" has a chain of addrinfo structure filled with
     * 0.0.0.0 (for IPv4), 0:0:0:0:0:0:0:0 (for IPv6) and alike,
     * with port filled for "myservice".
     */
    for (  ;  res;  res = res->ai_next)
    {
        error = getnameinfo(res->ai_addr,
                            res->ai_addrlen,
                            hbuf,
                            sizeof(hbuf),
                            sbuf,
                            sizeof(sbuf),
                            NI_NUMERICHOST | NI_NUMERICSERV);
        if (error)
        {
            fprintf(stderr, "%s\n", gai_strerror(error));
            continue;
        }
        fprintf(stderr, "trying (%d) %s port %s\n", res->ai_family, hbuf, sbuf);
    }
}
/*- End of function --------------------------------------------------------*/

int main(int argc, char *argv[])
{
    int t38_version;
    int redundancy_config;
    int i;
    int use_ecm;
    int caller;
    const char *input_file_name;
    int local_port;
    int far_port;
    const char *far_host;
    int transport;
    int error;
    struct addrinfo hints;
    struct addrinfo *res;
    char portx[10];

    t38_version = 1;
    redundancy_config = UDPTL_ERROR_CORRECTION_REDUNDANCY;
    input_file_name = INPUT_FILE_NAME;
    transport = 0;
    use_ecm = FALSE;
    caller = FALSE;
    local_port = 10024;
    far_port = 10026;
    far_host = "127.0.0.1";
    for (i = 1;  i < argc;  i++)
    {
        if (strcmp(argv[i], "-b") == 0)
        {
            local_port = atoi(argv[++i]);
            continue;
        }
        if (strcmp(argv[i], "-c") == 0)
        {
            caller = TRUE;
            continue;
        }
        if (strcmp(argv[i], "-e") == 0)
        {
            use_ecm = TRUE;
            continue;
        }
        if (strcmp(argv[i], "-h") == 0)
        {
            far_host = argv[++i];
            continue;
        }
        if (strcmp(argv[i], "-i") == 0)
        {
            input_file_name = argv[++i];
            continue;
        }
        if (strcmp(argv[i], "-r") == 0)
        {
            redundancy_config = atoi(argv[++i]);
            continue;
        }
        if (strcmp(argv[i], "-t") == 0)
        {
            transport = atoi(argv[++i]);
            continue;
        }
        if (strcmp(argv[i], "-v") == 0)
        {
            t38_version = atoi(argv[++i]);
            continue;
        }
    }

    event_init();

    t38_setup(t38_version, caller, use_ecm, input_file_name);

    tx_packet_state.transport = transport;

    if (tx_packet_state.transport == 0)
    {
        if (caller)
        {
            i = far_port;
            far_port = local_port;
            local_port = i;
        }
        else
        {
            far_port = -1;
        }
    }
    else
    {
        if (!caller)
        {
            i = far_port;
            far_port = local_port;
            local_port = i;
        }
    }

    snprintf(portx, sizeof(portx), "%d", far_port);
    memset(&hints, 0, sizeof(hints));
    /* Set-up hints structure */
    hints.ai_family = PF_UNSPEC;
    hints.ai_flags = AI_PASSIVE;
    hints.ai_socktype = SOCK_STREAM;

    if ((error = getaddrinfo(far_host, portx, &hints, &res)))
    {
        perror(gai_strerror(error));
        exit(2);
    }
    if (res == NULL)
    {
        perror(gai_strerror(error));
        exit(2);
    }
    memcpy(&tx_packet_state.farx, res->ai_addr, res->ai_addrlen);

    tx_packet_state.tcp_state = tcp_socket_create_and_bind(NULL, local_port, local_port);
    if (tx_packet_state.tcp_state == NULL)
    {
        printf("Cannot allocate the TCP ports\n");
        exit(2);
    }

    tcp_socket_set_far(tx_packet_state.tcp_state, (struct sockaddr_in *) &tx_packet_state.farx);

    tx_packet_state.udp_state = udp_socket_group_create_and_bind(2, 1, NULL, local_port, local_port + 1);
    if (tx_packet_state.udp_state == NULL)
    {
        printf("Cannot allocate the UDP ports\n");
        exit(2);
    }
    udp_socket_set_nat(tx_packet_state.udp_state, TRUE);
    udp_socket_set_far(tx_packet_state.udp_state, (struct sockaddr_in *) &tx_packet_state.farx);

    /* The TPKT transport needs to callback to the T.38 engine through a "shim" routine, that adds
       sequence numbers. */
    tpkt_init(&tx_packet_state.tpkt_state, process_rx_tpkt_packet, &tx_packet_state.t38_state);

    /* The UDPTL transport needs to callback to the T.38 engine through a "shim" routine, just
       to avoid complaints aboue parameter types */
    if ((udptl_init(&tx_packet_state.udptl_state, redundancy_config, 3, 3, process_rx_udptl_packet, &tx_packet_state.t38_state.t38)) == NULL)
    {
        printf("Test failed\n");
        exit(2);
    }

    /* The RTP transport needs to callback to the T.38 engine through a "shim" routine, that deals
       with the additional parameters of an RTP transport. */
    if ((rfc3550_init(&tx_packet_state.rfc3550_state, TRUE)) == NULL)
    {
        printf("Test failed\n");
        exit(2);
    }
    if ((rfc2198_init(&tx_packet_state.rfc2198_state, &tx_packet_state.rfc3550_state, 3, RFC3550_PAYLOAD_TYPE_RFC2198)) == NULL)
    {
        printf("Test failed\n");
        exit(2);
    }
    rfc3550_set_payload_handler(&tx_packet_state.rfc3550_state, RFC3550_PAYLOAD_TYPE_T38, process_rx_rtp_packet, &tx_packet_state.t38_state);

    exchange_handling(caller, local_port, far_port);
    return  0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
