/*
 * Vale - a library for media streaming.
 *
 * tcp.c - A simple abstraction of TCP connections.
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2007 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: tcp.c,v 1.3 2007/04/13 12:24:49 steveu Exp $
 */

/*! \file */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <string.h>
#include <sys/time.h>
#include <sys/uio.h>
#include <signal.h>
#include <errno.h>
#include <unistd.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <fcntl.h>

#include "vale/tcp.h"

struct tcp_state_s
{
    int fd;
    struct sockaddr_storage local;
    struct sockaddr_storage far;
    struct tcp_state_s *next;
    struct tcp_state_s *prev;
};

tcp_state_t *tcp_socket_create(void)
{
    int fd;
    long flags;
    int opt;
    struct linger linger;
    tcp_state_t *s;
    
    if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
        return NULL;
    flags = fcntl(fd, F_GETFL);
    if (fcntl(fd, F_SETFL, flags | O_NONBLOCK) < 0)
        return NULL;
    opt = 1;
    setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (void *) &opt, sizeof(opt));
    linger.l_onoff = 1;
    linger.l_linger = 5;
    setsockopt(fd, SOL_SOCKET, SO_LINGER, (void *) &linger, sizeof(linger));
    
    if ((s = malloc(sizeof(*s))) == NULL)
    {
        close(fd);
        return NULL;
    }
    memset(s, 0, sizeof(*s));
    ((struct sockaddr_in *) &s->far)->sin_family = AF_INET;
    ((struct sockaddr_in *) &s->local)->sin_family = AF_INET;
    s->fd = fd;
    s->next = NULL;
    s->prev = NULL;
    return s;
}
/*- End of function --------------------------------------------------------*/

int tcp_socket_destroy(tcp_state_t *s)
{
    if (s == NULL)
        return -1;
    if (s->fd >= 0)
        close(s->fd);
    free(s);
    return 0;
}
/*- End of function --------------------------------------------------------*/

tcp_state_t *tcp_socket_create_and_bind(struct in_addr *addr, int lowest_port, int highest_port)
{
    tcp_state_t *s;
    struct sockaddr_in sockaddr;
    int port;
    int starting_point;

    if ((s = tcp_socket_create()) == NULL)
        return NULL;

    port = lowest_port;
    if (highest_port != lowest_port)
        port += (rand()%(highest_port - lowest_port + 1));

    starting_point = port;
    for (;;)
    {
        memset(&sockaddr, 0, sizeof(sockaddr));
        sockaddr.sin_family = AF_INET;
        if (addr)
            sockaddr.sin_addr = *addr;
        else
            memset(&sockaddr.sin_addr, 0, sizeof(sockaddr.sin_addr));
        sockaddr.sin_port = htons(port);
        if (tcp_socket_set_local(s, &sockaddr) == 0)
        {
            /* We must have bound the port OK. */
            return s;
        }
        if (errno != EADDRINUSE)
        {
            tcp_socket_destroy(s);
            return NULL;
        }
        if (++port > highest_port)
            port = lowest_port;
        if (port == starting_point)
            break;
    }
    /* Unravel what we did so far, and give up */
    tcp_socket_destroy(s);
    return NULL;
}
/*- End of function --------------------------------------------------------*/

int tcp_socket_set_local(tcp_state_t *s, const struct sockaddr_in *us)
{
    int res;
    long flags;

    if (s == NULL  ||  s->fd < 0)
        return -1;

    if (((struct sockaddr_in *) &s->local)->sin_addr.s_addr  ||  ((struct sockaddr_in *) &s->local)->sin_port)
    {
        /* We are already bound, so we need to re-open the socket to unbind it */
        close(s->fd);
        if ((s->fd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
            return -1;
        flags = fcntl(s->fd, F_GETFL);
        if (fcntl(s->fd, F_SETFL, flags | O_NONBLOCK) < 0)
        {
            close(s->fd);
            return -1;
        }
    }
    ((struct sockaddr_in *) &s->local)->sin_port = us->sin_port;
    ((struct sockaddr_in *) &s->local)->sin_addr.s_addr = us->sin_addr.s_addr;
    if ((res = bind(s->fd, (struct sockaddr *) &s->local, sizeof(s->local))) < 0)
    {
        ((struct sockaddr_in *) &s->local)->sin_port = 0;
        ((struct sockaddr_in *) &s->local)->sin_addr.s_addr = 0;
    }
    return res;
}
/*- End of function --------------------------------------------------------*/

void tcp_socket_set_far(tcp_state_t *s, const struct sockaddr_in *far)
{
    ((struct sockaddr_in *) &s->far)->sin_port = far->sin_port;
    ((struct sockaddr_in *) &s->far)->sin_addr.s_addr = far->sin_addr.s_addr;
}
/*- End of function --------------------------------------------------------*/

int tcp_socket_set_tos(tcp_state_t *s, int tos)
{
    if (s == NULL)
        return -1;
    return setsockopt(s->fd, IPPROTO_IP, IP_TOS, &tos, sizeof(tos));
}
/*- End of function --------------------------------------------------------*/

int tcp_socket_restart(tcp_state_t *s)
{
    if (s == NULL)
        return -1;
    memset(&((struct sockaddr_in *) &s->far)->sin_addr.s_addr, 0, sizeof(((struct sockaddr_in *) &s->far)->sin_addr.s_addr));
    ((struct sockaddr_in *) &s->far)->sin_port = 0;
    return 0;
}
/*- End of function --------------------------------------------------------*/

int tcp_socket_fd(tcp_state_t *s)
{
    if (s == NULL)
        return -1;
    return s->fd;
}
/*- End of function --------------------------------------------------------*/

const struct sockaddr_in *tcp_socket_get_local(tcp_state_t *s)
{
    static const struct sockaddr_in dummy = {0};

    if (s)
        return (struct sockaddr_in *) &s->local;
    return &dummy;
}
/*- End of function --------------------------------------------------------*/

const struct sockaddr_in *tcp_socket_get_far(tcp_state_t *s)
{
    static const struct sockaddr_in dummy = {0};

    if (s)
        return (struct sockaddr_in *) &s->far;
    return &dummy;
}
/*- End of function --------------------------------------------------------*/

int tcp_socket_connect(tcp_state_t *s)
{
    if (connect(s->fd, (struct sockaddr *) &s->far, sizeof(s->far)) < 0)
    {
        if (errno != EINPROGRESS)
            return -1;
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

int tcp_socket_connection_result(tcp_state_t *s)
{
    int res;
    int val;
    socklen_t val_len;
    
    val_len = sizeof(val);
    if ((res = getsockopt(tcp_socket_fd(s), SOL_SOCKET, SO_ERROR, &val, &val_len)) < 0)
        return res;
    if (val)
    {
        errno = val;
        return -1;
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

int tcp_socket_listen(tcp_state_t *s)
{
    return listen(s->fd, 5);
}
/*- End of function --------------------------------------------------------*/

tcp_state_t *tcp_socket_accept(tcp_state_t *s)
{
    long flags;
    socklen_t sock_len;
    tcp_state_t *sx;

    if ((sx = malloc(sizeof(*sx))) == NULL)
        return NULL;
    memset(sx, 0, sizeof(*sx));
    sock_len = sizeof(sx->far);
    if ((sx->fd = accept(s->fd, (struct sockaddr *) &sx->far, &sock_len)) < 0)
    {
        free(sx);
        return NULL;
    }
    flags = fcntl(sx->fd, F_GETFL);
    if (fcntl(sx->fd, F_SETFL, flags | O_NONBLOCK) < 0)
    {
        close(sx->fd);
        free(sx);
        return NULL;
    }
    memcpy(&sx->local, &s->local, sizeof(sx->local));
    sx->next = NULL;
    sx->prev = NULL;
    return sx;
}
/*- End of function --------------------------------------------------------*/

int tcp_socket_recv(tcp_state_t *s, void *buf, size_t size, int flags)
{
    return recv(s->fd, buf, size, flags);
}
/*- End of function --------------------------------------------------------*/

int tcp_socket_send(tcp_state_t *s, const void *buf, size_t size, int flags)
{
    if (s == NULL  ||  s->fd < 0)
        return 0;
    return send(s->fd, buf, size, flags);
}
/*- End of function --------------------------------------------------------*/

int tcp_socket_send_with_header(tcp_state_t *s, const void *hdr_buf, size_t hdr_size, const void *buf, size_t size)
{
    struct iovec vector[3];

    if (s == NULL  ||  s->fd < 0)
        return 0;
    vector[0].iov_len = hdr_size;
    vector[0].iov_base = (uint8_t *) hdr_buf;
    vector[1].iov_len = size;
    vector[1].iov_base = (uint8_t *) buf;
    return writev(s->fd, vector, 2);
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
