/*
 * Vale - a library for media streaming.
 *
 * tpkt.c - An implementation of RFC1006, mostly for T.38
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2005 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: tpkt.c,v 1.2 2007/04/13 12:24:49 steveu Exp $
 */

/*! \file */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <inttypes.h>
#if defined(HAVE_TGMATH_H)
#include <tgmath.h>
#endif
#if defined(HAVE_MATH_H)
#include <math.h>
#endif
#include <memory.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <sys/select.h>
#include <sys/time.h>

#include "vale/tpkt.h"

#define FALSE 0
#define TRUE (!FALSE)

int tpkt_rx_chunk(tpkt_state_t *s, const uint8_t buf[], int len)
{
    int ptr;
    int msg_len;

    ptr = 0;
    if (s->msg_len != 0)
    {
        /* We have a fragment in our buffer to deal with */
        if (s->msg_len <= 4)
        {
            /* Try to achieve in place processing */
            if (s->msg_len < 4)
            {
                /* Try to get a complete header into our buffer */
                if (s->msg_len + len < 4)
                {
                    /* Still too little. Save this piece, and move on */
                    memcpy(s->msg + s->msg_len, buf, len);
                    s->msg_len += len;
                    return 0;
                }
                /* We can complete the header */
                ptr = 4 - s->msg_len;
                memcpy(s->msg + s->msg_len, buf, ptr);
                s->msg_len = 4;
            }
            /* Analyse the header */
            msg_len = ntohs(((int16_t *) s->msg)[1]);
            if (s->msg[0] != 3  ||  msg_len < 4)
            {
                /* The header does not make sense */
                return -1;
            }
            if (s->msg_len + len - ptr < msg_len)
            {
                /* Save the piece we have and move on */
                memcpy(s->msg + s->msg_len, buf + ptr, len - ptr);
                s->msg_len += (len - ptr);
                return 0;
            }
            /* We have the whole thing, which can be processed in place,
               in the incoming buffer. */
            s->rx_packet_handler(s->user_data, buf + ptr, msg_len - 4);
            ptr += (msg_len - 4);
            s->msg_len = 0;
        }
        else
        {
            /* In place processing is not possible */
            msg_len = ntohs(((int16_t *) s->msg)[1]);
            /* Let's recheck the header. It was checked before, but
               we might have been called again when we should not have
               been. */
            if (s->msg[0] != 3  ||  msg_len < 4)
            {
                /* The header does not make sense */
                return -1;
            }
            if (s->msg_len + len < msg_len)
            {
                /* Save the piece we have and move on */
                memcpy(s->msg + s->msg_len, buf, len);
                s->msg_len += len;
                return 0;
            }
            /* We can now process this in our local buffer */
            ptr = (msg_len - s->msg_len);
            memcpy(s->msg + s->msg_len, buf, ptr);
            s->msg_len += ptr;
            s->rx_packet_handler(s->user_data, s->msg + 4, msg_len - 4);
            s->msg_len = 0;
        }
    }
    /* We have no stored fragments to deal with. We are working
       from a clean position in the incoming buffer. */
    for (  ;  ptr < len;  ptr += msg_len)
    {
        /* See if we can process things in place, in the incoming
           buffer. */
        if ((len - ptr) >= 4)
        {
            msg_len = ntohs(((int16_t *) (buf + ptr))[1]);
            if (buf[ptr] != 3  ||  msg_len < 4)
            {
                /* The header does not make sense */
                return -1;
            }
            if ((len - ptr) >= msg_len)
            {
                /* We have the whole thing, which can be processed
                   in place, in the incoming buffer. */
                s->rx_packet_handler(s->user_data, buf + ptr + 4, msg_len - 4);
                continue;
            }
        }
        /* We don't have the whole message. Save what we have, and
           move on. */
        memcpy(s->msg, buf + ptr, len - ptr);
        s->msg_len = len - ptr;
        return 0;
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

int tpkt_build_packet(tpkt_state_t *s, uint8_t buf[], const uint8_t msg[], int msg_len)
{
    int len;

    if (msg_len < 0  ||  msg_len > 65531)
        return -1;
    /* Build the TPKT packet */
    len = 0;
    /* Encode the sequence number */
    /* Version */
    buf[len++] = 3;
    /* Reserved */
    buf[len++] = 0;
    /* Packet length - this includes the length of the header itself */
    buf[len++] = ((msg_len + 4) >> 8) & 0xFF;
    buf[len++] = (msg_len + 4) & 0xFF;

    memcpy(&buf[len], msg, msg_len);
    len += msg_len;

    return len;
}
/*- End of function --------------------------------------------------------*/

int tpkt_build_header(tpkt_state_t *s, uint8_t buf[], const uint8_t msg[], int msg_len)
{
    int len;

    if (msg_len < 0  ||  msg_len > 65531)
        return -1;
    /* Build the TPKT packet */
    len = 0;
    /* Encode the sequence number */
    /* Version */
    buf[len++] = 3;
    /* Reserved */
    buf[len++] = 0;
    /* Packet length - this includes the length of the header itself */
    buf[len++] = ((msg_len + 4) >> 8) & 0xFF;
    buf[len++] = (msg_len + 4) & 0xFF;

    return len;
}
/*- End of function --------------------------------------------------------*/

tpkt_state_t *tpkt_init(tpkt_state_t *s,
                        tpkt_rx_packet_handler_t rx_packet_handler,
                        void *user_data)
{
    if (rx_packet_handler == NULL)
        return NULL;
    if (s == NULL)
    {
        if ((s = (tpkt_state_t *) malloc(sizeof(*s))) == NULL)
            return NULL;
    }
    memset(s, 0, sizeof(*s));
    
    s->rx_packet_handler = rx_packet_handler;
    s->user_data = user_data;
    return s;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
